﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/os.h>

#include <nn/apm/detail/apm_Log.h>

#include <nne/soctherm/soctherm.h>

#include "apm_SettingsHolder-spec.nx.h"
#include "apm_SocthermManager-spec.nx.h"

namespace nn { namespace apm { namespace server {

namespace {

nne::soctherm::OcAlarm ConvertOcAlarmToSocthermOcAlarm(OcAlarm ocAlarm) NN_NOEXCEPT
{
    nne::soctherm::OcAlarm socthermOcAlarm = nne::soctherm::OcAlarm_Oc1;

    switch ( ocAlarm )
    {
    case OcAlarm_Oc1:
        socthermOcAlarm = nne::soctherm::OcAlarm_Oc1;
        break;
    case OcAlarm_Oc2:
        socthermOcAlarm = nne::soctherm::OcAlarm_Oc2;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return socthermOcAlarm;
}

int ConvertOcAlarmToIndex(OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(static_cast<int>(ocAlarm), 0, OcAlarmMax - 1);

    return static_cast<int>(ocAlarm);
}

} // namespace

void SocthermManager::Initialize(SettingsHolder* pSettingsHolder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSettingsHolder);

    NN_SDK_ASSERT_EQUAL(State_NotInitialized, m_State);

    nne::soctherm::Initialize();

    m_State = State_Initialized;

    // Platform Configuration から Throttling の設定をロードする。
    m_pSettingsHolder = pSettingsHolder;
    m_IsThrottlingForUndockEnabled = pSettingsHolder->IsThrottlingForUndockEnabled();
    m_IsThrottlingForSmpdEnabled = pSettingsHolder->IsThrottlingForSmpdEnabled();

    // 前提条件チェックにより State_Initialized の設定より前にはできない。
    DisableThrottling(OcAlarm_Oc1);

    // SIGLO-42998 WA: 一度すべて Disabled にする。
    DisableThrottling(OcAlarm_Oc2);

    // 前提条件チェックにより State_Initialized の設定より前にはできない。
    EnableThrottling(OcAlarm_Oc2);
}

void SocthermManager::Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(State_Initialized, m_State);

    DisableThrottling(OcAlarm_Oc1);
    DisableThrottling(OcAlarm_Oc2);

    nne::soctherm::Finalize();

    m_State = State_NotInitialized;
}

void SocthermManager::Suspend() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(State_Initialized, m_State);

    // m_ThrottlingEnabled は維持。
    nne::soctherm::EnableOcThrottle(nne::soctherm::OcAlarm_Oc1, false);
    nne::soctherm::EnableOcThrottle(nne::soctherm::OcAlarm_Oc2, false);

    nne::soctherm::OcInterruptSwitch(nne::soctherm::OcAlarm_Oc1, false);
    nne::soctherm::OcInterruptSwitch(nne::soctherm::OcAlarm_Oc2, false);

    nne::soctherm::Finalize();

    m_State = State_Suspended;
}

void SocthermManager::Resume() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(State_Suspended, m_State);

    nne::soctherm::Initialize();

    // 以前の状態 m_Polarity の復旧。
    nne::soctherm::SetOcAlarmPolarity(nne::soctherm::OcAlarm_Oc1, m_Polarity[ConvertOcAlarmToIndex(OcAlarm_Oc1)]);
    nne::soctherm::SetOcAlarmPolarity(nne::soctherm::OcAlarm_Oc2, m_Polarity[ConvertOcAlarmToIndex(OcAlarm_Oc2)]);

    nne::soctherm::OcInterruptSwitch(nne::soctherm::OcAlarm_Oc1, m_ThrottlingEnabled[ConvertOcAlarmToIndex(OcAlarm_Oc1)]);
    nne::soctherm::OcInterruptSwitch(nne::soctherm::OcAlarm_Oc2, m_ThrottlingEnabled[ConvertOcAlarmToIndex(OcAlarm_Oc2)]);

    // 以前の状態 m_ThrottlingEnabled の復旧。
    nne::soctherm::EnableOcThrottle(nne::soctherm::OcAlarm_Oc1, m_ThrottlingEnabled[ConvertOcAlarmToIndex(OcAlarm_Oc1)]);
    nne::soctherm::EnableOcThrottle(nne::soctherm::OcAlarm_Oc2, m_ThrottlingEnabled[ConvertOcAlarmToIndex(OcAlarm_Oc2)]);

    m_State = State_Initialized;
}

void SocthermManager::EnableThrottling(OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    // TODO: OcAlarm はドライバの仕様に紐付くので上層に公開する名前として不適当。
    // Undock, Smpl のように目的で再定義することが望ましい。

    // SIGLO-42998 WA: m_ThrottlingEnabled[*] = true を行われる前に return させる。
    if (!m_IsThrottlingForUndockEnabled && ocAlarm == OcAlarm_Oc1)
    {
        return;
    }

    if (!m_IsThrottlingForSmpdEnabled && ocAlarm == OcAlarm_Oc2)
    {
        return;
    }

    // スリープ遷移後に上位層から要求があった場合 Resume 時に反映する。
    m_ThrottlingEnabled[ConvertOcAlarmToIndex(ocAlarm)] = true;

    if ( m_State == State_Initialized )
    {
        nne::soctherm::OcInterruptSwitch(ConvertOcAlarmToSocthermOcAlarm(ocAlarm), true);
        nne::soctherm::EnableOcThrottle(ConvertOcAlarmToSocthermOcAlarm(ocAlarm), true);
    }
}

void SocthermManager::DisableThrottling(OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( m_State == State_Initialized )
    {
        // 運悪く IsInterruptedBy でチェックされずスロットリング状態で Disable を迎えた場合、ここで取得を開始する。
        if ( nne::soctherm::IsOcInterruptedBy(ConvertOcAlarmToSocthermOcAlarm(ocAlarm)) )
        {
            StartThrottlingDuration(ocAlarm);
        }

        UpdateLastThrottlingState(ocAlarm);

        nne::soctherm::EnableOcThrottle(ConvertOcAlarmToSocthermOcAlarm(ocAlarm), false);

        // 無効にするとき必ずクリアする。
        // 現在の実装では冗長であるが実装上の 2 つの意味があるので（割り込みフラグの消去とスロットリング状態の解除）維持する。
        ClearThrottling(ocAlarm);

        EndThrottlingDuration(ocAlarm);

        nne::soctherm::OcInterruptSwitch(ConvertOcAlarmToSocthermOcAlarm(ocAlarm), false);
    }

    // スリープ遷移後に上位層から要求があった場合 Resume 時に反映する。
    m_ThrottlingEnabled[ConvertOcAlarmToIndex(ocAlarm)] = false;
}

bool SocthermManager::IsThrottlingEnabled(OcAlarm ocAlarm) NN_NOEXCEPT
{
    if ( m_State == State_Initialized )
    {
        return m_ThrottlingEnabled[ConvertOcAlarmToIndex(ocAlarm)];
    }

    return false;
}

void SocthermManager::ClearThrottling(OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( m_State == State_Initialized )
    {
        nne::soctherm::ClearOcStatus(ConvertOcAlarmToSocthermOcAlarm(ocAlarm));
    }
}

bool SocthermManager::IsInterruptedBy(OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    bool interrupted = false;

    if ( m_State == State_Initialized )
    {
        if ( nne::soctherm::IsOcInterruptedBy(ConvertOcAlarmToSocthermOcAlarm(ocAlarm)) )
        {
            StartThrottlingDuration(ocAlarm);
            interrupted = true;
        }
    }

    return interrupted;
}

void SocthermManager::StartThrottlingDuration(OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( m_State == State_Initialized )
    {
        if ( !m_ThrottlingInvoked[ConvertOcAlarmToIndex(ocAlarm)] )
        {
            m_ThrottlingInvoked[ConvertOcAlarmToIndex(ocAlarm)]
                =  ((nne::soctherm::GetThrottleStatus() & 0x00000ff0) != 0)
                || ((nne::soctherm::GetCpuPskipStatus() & 0x00000001) != 0)
                || ((nne::soctherm::GetGpuPskipStatus() & 0x00000001) != 0);

            if ( m_ThrottlingInvoked[ConvertOcAlarmToIndex(ocAlarm)] )
            {
                m_ThrottlingStartTick[ConvertOcAlarmToIndex(ocAlarm)] = nn::os::GetSystemTick();
            }
        }
    }
}

void SocthermManager::UpdateLastThrottlingState(OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( m_State == State_Initialized )
    {
        if ( m_ThrottlingInvoked[ConvertOcAlarmToIndex(ocAlarm)] )
        {
            GetThrottlingStateInternal(m_LastThrottlingState[ConvertOcAlarmToIndex(ocAlarm)], ocAlarm);
        }
    }
}

void SocthermManager::EndThrottlingDuration(OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( m_State == State_Initialized )
    {
        if ( m_ThrottlingInvoked[ConvertOcAlarmToIndex(ocAlarm)] )
        {
            GetThrottlingDurationInternal(m_LastThrottlingState[ConvertOcAlarmToIndex(ocAlarm)], ocAlarm);
            m_ThrottlingInvoked[ConvertOcAlarmToIndex(ocAlarm)] = false;
            m_ThrottlingStartTick[ConvertOcAlarmToIndex(ocAlarm)] = nn::os::Tick(0);
        }
    }
}

void SocthermManager::GetThrottlingState(ThrottlingState* pOutThrottlingState, OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( m_State == State_Initialized )
    {
        GetThrottlingStateInternal(*pOutThrottlingState, ocAlarm);
        GetThrottlingDurationInternal(*pOutThrottlingState, ocAlarm);
    }
    else
    {
        // 明らかに無効な値を入れる。
        (*pOutThrottlingState).Clear();
    }
}

void SocthermManager::GetLastThrottlingState(ThrottlingState* pOutThrottlingState, OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( m_State == State_Initialized )
    {
        *pOutThrottlingState = m_LastThrottlingState[ConvertOcAlarmToIndex(ocAlarm)];
    }
}

void SocthermManager::ClearLastThrottlingState(OcAlarm ocAlarm) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( m_State == State_Initialized )
    {
        m_LastThrottlingState[ConvertOcAlarmToIndex(ocAlarm)].Clear();
    }
}

void SocthermManager::GetThrottlingStateInternal(ThrottlingState& outThrottlingState, OcAlarm ocAlarm) NN_NOEXCEPT
{
    // m_State のチェックは呼び出し元で実行されている前提。

    outThrottlingState.throttlingRateCpuDividend = nne::soctherm::GetCpuPskipDividend(ConvertOcAlarmToSocthermOcAlarm(ocAlarm));
    outThrottlingState.throttlingRateCpuDivisor = nne::soctherm::GetCpuPskipDivisor(ConvertOcAlarmToSocthermOcAlarm(ocAlarm));

    switch ( GetGpuPskipThrottleDepth(ConvertOcAlarmToSocthermOcAlarm(ocAlarm)) )
    {
    case nne::soctherm::ThrottleDepth_Low:
        outThrottlingState.throttlingRateGpuDividend = 1;
        outThrottlingState.throttlingRateGpuDivisor = 2;
        break;
    case nne::soctherm::ThrottleDepth_Medium:
        outThrottlingState.throttlingRateGpuDividend = 1;
        outThrottlingState.throttlingRateGpuDivisor = 4;
        break;
    case nne::soctherm::ThrottleDepth_High:
        outThrottlingState.throttlingRateGpuDividend = 1;
        outThrottlingState.throttlingRateGpuDivisor = 8;
        break;
    default:
        outThrottlingState.throttlingRateGpuDividend = 1;
        outThrottlingState.throttlingRateGpuDivisor = 1;
        break;
    }

    outThrottlingState.throttleStatus = nne::soctherm::GetThrottleStatus();
    outThrottlingState.cpuPSkipStatus = nne::soctherm::GetCpuPskipStatus();
    outThrottlingState.gpuPSkipStatus = nne::soctherm::GetGpuPskipStatus();

    //DumpStatus();

    outThrottlingState.enabled = m_ThrottlingEnabled[ConvertOcAlarmToIndex(ocAlarm)];
    outThrottlingState.invoked = m_ThrottlingInvoked[ConvertOcAlarmToIndex(ocAlarm)];
}

void SocthermManager::GetThrottlingDurationInternal(ThrottlingState& outThrottlingState, OcAlarm ocAlarm) NN_NOEXCEPT
{
    // m_State のチェックは呼び出し元で実行されている前提。

    if ( m_ThrottlingInvoked[ConvertOcAlarmToIndex(ocAlarm)] )
    {
        outThrottlingState.durationNanoSeconds = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()
            - m_ThrottlingStartTick[ConvertOcAlarmToIndex(ocAlarm)]).GetNanoSeconds();
    }
    else
    {
        outThrottlingState.durationNanoSeconds = 0;
    }
}

void SocthermManager::SetPolarity(OcAlarm ocAlarm, Polarity polarity) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( polarity == Polarity_AssertHigh )
    {
        m_Polarity[ConvertOcAlarmToIndex(ocAlarm)] = true;
    }
    else if ( polarity == Polarity_AssertLow )
    {
        m_Polarity[ConvertOcAlarmToIndex(ocAlarm)] = false;
    }

    if ( m_State == State_Initialized )
    {
        nne::soctherm::SetOcAlarmPolarity(ConvertOcAlarmToSocthermOcAlarm(ocAlarm), m_Polarity[ConvertOcAlarmToIndex(ocAlarm)]);
    }
}

Polarity SocthermManager::GetPolarity(OcAlarm ocAlarm) NN_NOEXCEPT
{
    return (m_Polarity[ConvertOcAlarmToIndex(ocAlarm)] ? Polarity_AssertHigh : Polarity_AssertLow);
}

void SocthermManager::DumpStatus() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(State_NotInitialized, m_State);

    if ( m_State == State_Initialized )
    {
        NN_DETAIL_APM_TRACE_V3("STATUS : %08x\n", nne::soctherm::GetThrottleStatus());
        NN_DETAIL_APM_TRACE_V3("CPSKIP : %08x\n", nne::soctherm::GetCpuPskipStatus());
        NN_DETAIL_APM_TRACE_V3("GPSKIP : %08x\n", nne::soctherm::GetGpuPskipStatus());
    }
}

}}} // namespace nn::apm::server
