﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/apm/apm_ManagerImpl.h>
#include <nn/apm/apm_ResultPrivate.h>
#include <nn/apm/detail/apm_Log.h>
#include <nn/apm/server/apm.h>

#include "apm_PerformanceList.h"
#include "apm_PerformanceServer.h"
#include "apm_SessionImpl.h"

namespace nn { namespace apm { namespace server {

PerformanceConfiguration Session::GetPerformanceConfiguration(nn::apm::PerformanceMode performanceMode) NN_NOEXCEPT
{
    PerformanceConfiguration performanceConfiguration = GetDefaultPerformanceConfiguration(performanceMode);

    switch ( performanceMode )
    {
    case PerformanceMode_Normal:
        if ( m_ConfigurationNormal != PerformanceConfiguration_Invalid )
        {
            performanceConfiguration = m_ConfigurationNormal;
        }
        break;
    case PerformanceMode_Boost:
        if ( m_ConfigurationBoost != PerformanceConfiguration_Invalid )
        {
            performanceConfiguration = m_ConfigurationBoost;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return performanceConfiguration;
}

nn::Result Session::SetPerformanceConfigurationInternal(nn::apm::PerformanceMode performanceMode, nn::apm::PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT
{
    switch ( performanceMode )
    {
    case PerformanceMode_Normal:
        m_ConfigurationNormal = performanceConfiguration;
        break;
    case PerformanceMode_Boost:
        m_ConfigurationBoost = performanceConfiguration;
        break;
    default:
        return ResultUnsupportedMode();
    }

    UpdatePerformance();

    NN_RESULT_SUCCESS;
}

nn::Result Session::GetPerformanceConfiguration(nn::sf::Out<nn::apm::PerformanceConfiguration> pOutPerformanceConfiguration, nn::apm::PerformanceMode performanceMode) NN_NOEXCEPT
{
    switch ( performanceMode )
    {
    case PerformanceMode_Normal:
    case PerformanceMode_Boost:
        *pOutPerformanceConfiguration = GetPerformanceConfiguration(performanceMode);
        break;
    default:
        return ResultUnsupportedMode();
    }

    NN_RESULT_SUCCESS;
}

SessionImpl::SessionImpl(ManagerImpl* pParent) NN_NOEXCEPT
    : Session(),
      m_Parent(pParent, true)
{
    NN_DETAIL_APM_TRACE("Create the session.\n");

    AddSessionToList(this);
    UpdatePerformance();
}

SessionImpl::~SessionImpl() NN_NOEXCEPT
{
    NN_DETAIL_APM_TRACE("Destroy the session.\n");

    RemoveSessionFromList(this);
    UpdatePerformance();
}

nn::Result SessionImpl::SetPerformanceConfiguration(nn::apm::PerformanceMode performanceMode, nn::apm::PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT
{
    PerformanceType type = CheckPerformanceList(performanceMode, performanceConfiguration);

    if ( type != PerformanceType_Privileged && type != PerformanceType_Public )
    {
        return ResultUnsupportedConfiguration();
    }
    else if ( type == PerformanceType_Privileged && !GetBatteryDrainingEnabled() )
    {
        return ResultPrivilegedConfiguration();
    }

    NN_RESULT_DO(SetPerformanceConfigurationInternal(performanceMode, performanceConfiguration));

    NN_RESULT_SUCCESS;
}

SessionPrivilegedImpl::SessionPrivilegedImpl(ManagerPrivilegedImpl* pParent) NN_NOEXCEPT
    : Session(),
      m_Parent(pParent, true)
{
    NN_DETAIL_APM_TRACE("Create the privileged session.\n");

    AddSessionToList(this);
    UpdatePerformance();
}

SessionPrivilegedImpl::~SessionPrivilegedImpl() NN_NOEXCEPT
{
    NN_DETAIL_APM_TRACE("Destroy the privileged session.\n");

    RemoveSessionFromList(this);
    UpdatePerformance();
}

nn::Result SessionPrivilegedImpl::SetPerformanceConfiguration(nn::apm::PerformanceMode performanceMode, nn::apm::PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT
{
    PerformanceType type = CheckPerformanceList(performanceMode, performanceConfiguration);

    if ( type != PerformanceType_Privileged && type != PerformanceType_Public )
    {
        return ResultUnsupportedConfiguration();
    }

    NN_RESULT_DO(SetPerformanceConfigurationInternal(performanceMode, performanceConfiguration));

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::apm::server
