﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt.h>

#include <nn/apm/detail/apm_Log.h>
#include <nn/apm/server/apm.h>
#include <nn/bpc/bpc_BoardPowerControl.h>
#include <nn/os.h>
#include <nn/pcv/pcv_Immediate.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nn/util/util_IntrusiveList.h>

#include "apm_SettingsHolder-spec.nx.h"
#include "apm_PerformanceList.h"
#include "apm_PerformanceServer.h"
#include "apm_PerformanceRequests.h"
#include "apm_SessionImpl.h"
#include "apm_SocthermManager-spec.nx.h"

namespace nn { namespace apm { namespace server {

namespace {

#define NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(result) \
    do \
    { \
        if ( (static_cast<nn::Result>(result)).IsFailure() ) \
        { \
            NN_DETAIL_APM_WARN("%s failed.\n", NN_MACRO_STRINGIZE(result)); \
        } \
    } while ( NN_STATIC_CONDITION(false) )

PerformanceMode g_PerformanceMode = PerformanceMode_Normal;
PerformanceMode g_RequestedPerformanceMode = PerformanceMode_Normal;

nn::psm::Session     g_PsmSession;

nn::os::SystemEventType g_PsmStateChangedEvent;
nn::os::SystemEventType g_ThrottlingDetected;
nn::os::SystemEventType g_ThrottlingFinished;

nn::os::SystemEventType g_PerformanceModeChangedEvent;
nn::os::SystemEventType g_SleepRequiredByLowVoltageEvent;

nn::os::BarrierType g_SdevThrottlingDelayFinished;

nn::util::IntrusiveList<Session, nn::util::IntrusiveListBaseNodeTraits<Session>> g_SessionList;

BasePerformanceConfiguration g_BasePerformanceConfiguration = BasePerformanceConfiguration_Application;

bool g_ExternalAccessEnabled = true;

SettingsHolder g_SettingsHolder;
SocthermManager g_SocthermManager;

RequestManager g_RequestManager;

const nn::fgm::Setting CpuOverclockSetting = 1785000000;

bool g_EnoughPowerChargeEmulationEnabled = false;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
};

StaticMutex g_PerformanceMutex = { NN_OS_MUTEX_INITIALIZER(false) };
StaticMutex g_SessionListMutex = { NN_OS_MUTEX_INITIALIZER(false) };

SettingsHolder::PerformanceModePolicy g_LastPerformanceModePolicy = SettingsHolder::PerformanceModePolicy::Auto;

} // namespace

void TransitPerformanceMode(PerformanceMode performanceMode) NN_NOEXCEPT
{
    PerformanceMode prePerformanceMode = g_PerformanceMode;
    g_PerformanceMode = performanceMode;

    // 単純化の為 UpdatePerformance は前状態にかかわらず呼び出す。
    UpdatePerformance();

    // Signal は本当に状態が変わった時だけにしたいので prePerfromanceMode と比較する。
    if ( prePerformanceMode != performanceMode )
    {
        nn::os::SignalSystemEvent(&g_PerformanceModeChangedEvent);
    }
}

bool IsEnoughPower() NN_NOEXCEPT
{
    bool isEnoughPower = false;

    if ( g_ExternalAccessEnabled )
    {
        isEnoughPower = nn::psm::IsEnoughPowerSupplied();
    }

    return isEnoughPower;
}

bool CheckBoostModeIsAvailable() NN_NOEXCEPT
{
    bool isEnoughPower      = false;
    bool isEnoughBattery    = false;
    bool acOk               = false;
    bool boostModeRequested = false;

    if ( g_ExternalAccessEnabled )
    {
        isEnoughPower      = nn::psm::IsEnoughPowerSupplied();
        isEnoughBattery    = (nn::psm::GetBatteryVoltageState() == nn::psm::BatteryVoltageState_Good);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bpc::GetAcOk(&acOk));
        boostModeRequested = (g_RequestedPerformanceMode        == nn::apm::PerformanceMode_Boost);
    }

    return isEnoughPower && isEnoughBattery && acOk && boostModeRequested;
}

void UpdateExternalRequests(PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT
{
    if ( g_ExternalAccessEnabled )
    {
        g_RequestManager.Update(performanceConfiguration);
    }
}

void EnableFastBatteryCharging() NN_NOEXCEPT
{
    if ( g_ExternalAccessEnabled )
    {
        nn::psm::EnableFastBatteryCharging();
    }
}

void DisableFastBatteryCharging() NN_NOEXCEPT
{
    if ( g_ExternalAccessEnabled )
    {
        nn::psm::DisableFastBatteryCharging();
    }
}

PerformanceConfiguration GetCandidatePerformanceConfiguration(PerformanceMode performanceMode) NN_NOEXCEPT
{
    PerformanceConfiguration performanceConfiguration = PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz;

    if ( g_BasePerformanceConfiguration == BasePerformanceConfiguration_Application )
    {
        PerformanceConfiguration defaultConfig;

        defaultConfig = GetDefaultPerformanceConfiguration(performanceMode);

        g_SessionListMutex.lock();

        if ( g_SessionList.empty() )
        {
            performanceConfiguration = defaultConfig;
        }
        else
        {
            // 実装の時点ではクライアントは 1 つだけに対応。
            Session& reference = g_SessionList.front();

            performanceConfiguration = reference.GetPerformanceConfiguration(g_PerformanceMode);

            if ( performanceConfiguration == PerformanceConfiguration_Invalid )
            {
                performanceConfiguration = defaultConfig;
            }
        }

        g_SessionListMutex.unlock();
    }
    else if ( g_BasePerformanceConfiguration == BasePerformanceConfiguration_Background )
    {
        performanceConfiguration = PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz;
    }

    return performanceConfiguration;
}

// g_PerformanceMode と g_SessionList の操作を受けて、設定を更新する。
void UpdatePerformance() NN_NOEXCEPT
{
    PerformanceConfiguration performanceConfiguration = GetCandidatePerformanceConfiguration(g_PerformanceMode);
    bool isThrottlingNeeded = g_RequestManager.IsThrottlingNeeded(performanceConfiguration);

    if ( isThrottlingNeeded )
    {
        DisableFastBatteryCharging();
    }

    // 性能設定とスロットリングの切り替えは不可分である。
    g_PerformanceMutex.lock();

    if ( isThrottlingNeeded )
    {
        g_SocthermManager.EnableThrottling(OcAlarm_Oc1);
    }

    NN_DETAIL_APM_INFO("[server] PerformanceConfiguration : 0x%08x\n", performanceConfiguration);
    NN_DETAIL_APM_INFO("[server] Oc1ThrottlingEnabled     : %d\n", g_SocthermManager.IsThrottlingEnabled(OcAlarm_Oc1));

    UpdateExternalRequests(performanceConfiguration);

    if ( !isThrottlingNeeded )
    {
        g_SocthermManager.DisableThrottling(OcAlarm_Oc1);
    }

    g_PerformanceMutex.unlock();

    if ( !isThrottlingNeeded )
    {
        EnableFastBatteryCharging();
    }

    // PerformanceConfiguration の記録。
    nn::erpt::Context context(nn::erpt::PerformanceInfo);

    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PerformanceMode, g_PerformanceMode));

    // SHOULD BE FIXED: static_cast は Error report の型を uint32_t にした為の禍根。
    // APM の PerformanceConfiguration は U 無 enum なので int32_t であるべきだった。
    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PerformanceConfiguration, static_cast<uint32_t>(performanceConfiguration)));

    NN_DETAIL_APM_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
}

bool GetBatteryDrainingEnabled() NN_NOEXCEPT
{
    return g_SettingsHolder.IsBatteryDrainingEnabled();
}

PerformanceConfiguration GetDefaultPerformanceConfiguration(PerformanceMode performanceMode) NN_NOEXCEPT
{
    PerformanceConfiguration performanceConfiguration;

    switch ( performanceMode )
    {
    case PerformanceMode_Normal:
        performanceConfiguration = DefaultPerformanceConfigurationForNormalMode;
        break;
    case PerformanceMode_Boost:
        performanceConfiguration = DefaultPerformanceConfigurationForBoostMode;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return performanceConfiguration;
}

void Initialize() NN_NOEXCEPT
{
    // サーバスレッドとハンドラスレッドの同期用バリア
    nn::os::InitializeBarrier(&g_SdevThrottlingDelayFinished, 2);

    g_RequestManager.Initialize();

    nn::pcv::InitializeForImmediate();

    nn::psm::Initialize();
    nn::psm::OpenSession(&g_PsmSession);
    nn::psm::SetChargerTypeChangeEventEnabled(&g_PsmSession, false);
    nn::psm::SetPowerSupplyChangeEventEnabled(&g_PsmSession, true); // EnoughPowerSupplied の切り替わりの取得
    nn::psm::SetBatteryVoltageStateChangeEventEnabled(&g_PsmSession, true);
    nn::psm::BindStateChangeEvent(&g_PsmStateChangedEvent, &g_PsmSession);

    nn::bpc::InitializeBoardPowerControl();

    g_SettingsHolder.LoadSettings();

    g_SocthermManager.Initialize(&g_SettingsHolder);

    nn::os::CreateSystemEvent(&g_PerformanceModeChangedEvent, nn::os::EventClearMode_ManualClear, true);
    nn::os::CreateSystemEvent(&g_SleepRequiredByLowVoltageEvent, nn::os::EventClearMode_ManualClear, true);
    nn::os::CreateSystemEvent(&g_ThrottlingDetected, nn::os::EventClearMode_ManualClear, false);
    nn::os::CreateSystemEvent(&g_ThrottlingFinished, nn::os::EventClearMode_ManualClear, false);

    if ( g_SettingsHolder.IsEnoughPowerChargeEmulationRequired() )
    {
        nn::psm::EnableEnoughPowerChargeEmulation();
        g_EnoughPowerChargeEmulationEnabled = true;
    }
    else
    {
        nn::psm::DisableEnoughPowerChargeEmulation();
        g_EnoughPowerChargeEmulationEnabled = false;
    }

    // 初期状態はノーマルモード。
    g_PerformanceMode = PerformanceMode_Normal;

    // Throttling 要不要判定のために直前の PerformanceModePolicy を格納する(auto -> auto の時だけ有効)
    g_LastPerformanceModePolicy = g_SettingsHolder.GetPerformanceModePolicy();

    // 性能を初期化する。
    UpdatePerformance();
}

void Finalize() NN_NOEXCEPT
{
    nn::os::DestroySystemEvent(&g_ThrottlingDetected);
    nn::os::DestroySystemEvent(&g_ThrottlingFinished);
    nn::os::DestroySystemEvent(&g_PerformanceModeChangedEvent);
    nn::os::DestroySystemEvent(&g_SleepRequiredByLowVoltageEvent);

    g_SocthermManager.Finalize();

    nn::bpc::FinalizeBoardPowerControl();

    nn::psm::UnbindStateChangeEvent(&g_PsmSession);
    nn::psm::CloseSession(&g_PsmSession);
    nn::psm::Finalize();

    nn::pcv::FinalizeForImmediate();

    g_RequestManager.Finalize();

    nn::os::FinalizeBarrier(&g_SdevThrottlingDelayFinished);
}

nn::os::SystemEventType* GetEventForUpdate(EventTarget eventTarget) NN_NOEXCEPT
{
    nn::os::SystemEventType* pSystemEvent;

    switch ( eventTarget )
    {
    case EventTarget_PsmStateChanged:
        pSystemEvent = &g_PsmStateChangedEvent;
        break;
    case EventTarget_ThrottlingDetected:
        pSystemEvent = &g_ThrottlingDetected;
        break;
    case EventTarget_ThrottlingFinished:
        pSystemEvent = &g_ThrottlingFinished;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return pSystemEvent;
}

void Update() NN_NOEXCEPT
{
    if ( g_SettingsHolder.IsEnoughPowerChargeEmulationRequired() && !g_EnoughPowerChargeEmulationEnabled )
    {
        nn::psm::EnableEnoughPowerChargeEmulation();
        g_EnoughPowerChargeEmulationEnabled = true;
    }

    bool cpuOverclockEnabled = g_SettingsHolder.IsSdevCpuOverclockEnabled() && IsEnoughPower();
    if ( cpuOverclockEnabled )
    {
        g_RequestManager.BeginCpuFixedSetting(CpuOverclockSetting);
    }
    else
    {
        g_RequestManager.EndCpuFixedSetting();
    }

    PerformanceMode performanceMode = PerformanceMode_Normal;

    auto performanceModePolicy = g_SettingsHolder.GetPerformanceModePolicy();

    switch ( performanceModePolicy )
    {
    case SettingsHolder::PerformanceModePolicy::Boost:
        performanceMode = IsEnoughPower() ? PerformanceMode_Boost : PerformanceMode_Normal;
        break;
    case SettingsHolder::PerformanceModePolicy::Normal:
        performanceMode = PerformanceMode_Normal;
        break;
    case SettingsHolder::PerformanceModePolicy::Auto:
        performanceMode = CheckBoostModeIsAvailable() ? PerformanceMode_Boost : PerformanceMode_Normal;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_DETAIL_APM_INFO("[server] PerformanceMode       : %d -> %d\n", g_PerformanceMode, performanceMode);
    NN_DETAIL_APM_INFO("[server] PerformanceModePolicy : %d -> %d\n", g_LastPerformanceModePolicy, performanceModePolicy);
    NN_DETAIL_APM_INFO("[server] CpuOverclockEnabled   : %d\n", cpuOverclockEnabled);

    if ( g_SettingsHolder.IsSdevThrottlingEnabled() )
    {
        g_PerformanceMutex.lock();

        bool oc1ThrottlingEnabled = g_SocthermManager.IsThrottlingEnabled(OcAlarm_Oc1);
        NN_DETAIL_APM_INFO("[server] Oc1ThrottlingEnabled  : %d\n", oc1ThrottlingEnabled);

        // 性能モードポリシー Auto かつこれから遷移する先が NormalMode かつ IsThrottlingEnabled の場合のみ Polarity を High にする
        if ( g_LastPerformanceModePolicy == SettingsHolder::PerformanceModePolicy::Auto
            && performanceModePolicy == SettingsHolder::PerformanceModePolicy::Auto
            && g_PerformanceMode == PerformanceMode_Boost
            && performanceMode == PerformanceMode_Normal
            && oc1ThrottlingEnabled )
        {
            // OC1 の極性変更によりスロットリングの割り込みが発生し APM のスロットリングハンドリングスレッドが起床する
            g_SocthermManager.SetPolarity(OcAlarm_Oc1, Polarity_AssertHigh);
            g_PerformanceMutex.unlock();

            // SIGLO-77507: 設定された時間スロットリング状態にあることの保証
            nn::os::AwaitBarrier(&g_SdevThrottlingDelayFinished);
        }
        else
        {
            g_PerformanceMutex.unlock();
        }
    }

    // 判定に使用し終えたので g_LastPerfomanceModePolicy を更新
    g_LastPerformanceModePolicy = performanceModePolicy;

    TransitPerformanceMode(performanceMode);

    if ( !g_SettingsHolder.IsEnoughPowerChargeEmulationRequired() && g_EnoughPowerChargeEmulationEnabled )
    {
        nn::psm::DisableEnoughPowerChargeEmulation();
        g_EnoughPowerChargeEmulationEnabled = false;
    }
}

void HandleSocthermInterrupts() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_PerformanceMutex);

    if ( g_SocthermManager.IsInterruptedBy(OcAlarm_Oc1) )
    {
        //NN_DETAIL_APM_TRACE_V3("Dump before interrupt handler.\n");
        //g_SocthermManager.DumpStatus();

        nn::os::SignalSystemEvent(&g_ThrottlingDetected);

        if ( g_SettingsHolder.IsSdevThrottlingEnabled() )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(g_SettingsHolder.GetSdevThrottlingAdditionalDelayUs()));
        }

        // 現在のノーマルモードの性能を取得する。
        PerformanceConfiguration performanceConfiguration = GetCandidatePerformanceConfiguration(PerformanceMode_Normal);

        // PCV に直接ノーマルモードの性能を設定する。
        nn::pcv::SetClockRateImmediately(nn::pcv::Module_Cpu, static_cast<nn::pcv::ClockHz>(g_RequestManager.GetCpuMinSetting(performanceConfiguration)));
        nn::pcv::SetClockRateImmediately(nn::pcv::Module_Gpu, static_cast<nn::pcv::ClockHz>(g_RequestManager.GetGpuMinSetting(performanceConfiguration)));
        nn::pcv::SetClockRateImmediately(nn::pcv::Module_Emc, static_cast<nn::pcv::ClockHz>(g_RequestManager.GetEmcMinSetting(performanceConfiguration)));

        // スロットリングを無効化する。
        g_SocthermManager.DisableThrottling(OcAlarm_Oc1);

        // スロットリングを解除する。
        g_SocthermManager.ClearThrottling(OcAlarm_Oc1);

        if ( g_SettingsHolder.IsSdevThrottlingEnabled() )
        {
            g_SocthermManager.SetPolarity(OcAlarm_Oc1, Polarity_AssertLow);

            // SIGLO-77507: 設定された時間スロットリング状態にあることの保証
            // スレッドの優先度および lock_guard の範囲に依存せずに Polarity の変更順序を保証するため Polarity_AssertLow より後に Await する
            nn::os::AwaitBarrier(&g_SdevThrottlingDelayFinished);
        }

        //NN_DETAIL_APM_TRACE_V3("Dump after interrupt handler.\n");
        //g_SocthermManager.DumpStatus();

        nn::os::SignalSystemEvent(&g_ThrottlingFinished);
    }

    // Voltage Comparator の Throttling の場合スリープイベントをシグナルする。
    if  ( g_SocthermManager.IsInterruptedBy(OcAlarm_Oc2) )
    {
        NN_DETAIL_APM_TRACE("Throttling worked because of low voltage.\n");

        nn::os::SignalSystemEvent(&g_SleepRequiredByLowVoltageEvent);

        // 一度寝て Resume するまで復活しない。
        g_SocthermManager.DisableThrottling(OcAlarm_Oc2);

        // IsInterruptedBy で参照する割込みフラグ、兼、スロットリングの発動フラグ。
        g_SocthermManager.ClearThrottling(OcAlarm_Oc2);
    }
}

void RequestPerformanceMode(PerformanceMode performanceMode) NN_NOEXCEPT
{
    g_RequestedPerformanceMode = performanceMode;
    Update();
}

PerformanceMode GetPerformanceMode() NN_NOEXCEPT
{
    return g_PerformanceMode;
}

void GetEventPtr(nn::os::SystemEventType** pOutEventPtr, nn::apm::EventTarget target) NN_NOEXCEPT
{
    switch ( target )
    {
    case nn::apm::EventTarget_PerformanceModeChanged:
        *pOutEventPtr = &g_PerformanceModeChangedEvent;
        break;
    case nn::apm::EventTarget_SleepRequiredByLowVoltage:
        *pOutEventPtr = &g_SleepRequiredByLowVoltageEvent;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void AddSessionToList(Session* pSession) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_SessionListMutex);

    g_SessionList.push_front(*pSession);
}

void RemoveSessionFromList(Session* pSession) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_SessionListMutex);

    for ( auto itr  = g_SessionList.cbegin(); itr != g_SessionList.cend(); itr++ )
    {
        if ( &(*itr) == pSession )
        {
            g_SessionList.erase(itr);
            break;
        }
    }
}

void SetBasePerformanceConfiguration(BasePerformanceConfiguration basePerformanceConfiguration) NN_NOEXCEPT
{
    g_BasePerformanceConfiguration = basePerformanceConfiguration;
    Update();
}

void SetExternalAccessEnabled(bool enabled) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_PerformanceMutex);

    if ( enabled && !g_ExternalAccessEnabled )
    {
        g_SocthermManager.Resume();
        g_SocthermManager.EnableThrottling(OcAlarm_Oc2);
    }
    else if ( !enabled && g_ExternalAccessEnabled )
    {
        g_SocthermManager.Suspend();
    }
    g_ExternalAccessEnabled = enabled;
}

void GetThrottlingState(ThrottlingState* pOutThrottlingState) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_PerformanceMutex);

    // 現在 OC2 を取得するケースは無いのでこれより上層では OC は指定不要。
    g_SocthermManager.GetThrottlingState(pOutThrottlingState, OcAlarm_Oc1);
}

void GetLastThrottlingState(ThrottlingState* pOutThrottlingState) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_PerformanceMutex);

    // 現在 OC2 を取得するケースは無いのでこれより上層では OC は指定不要。
    g_SocthermManager.GetLastThrottlingState(pOutThrottlingState, OcAlarm_Oc1);
}

void ClearLastThrottlingState() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_PerformanceMutex);

    // 現在 OC2 を取得するケースは無いのでこれより上層では OC は指定不要。
    g_SocthermManager.ClearLastThrottlingState(OcAlarm_Oc1);
}

void LoadAndApplySettings() NN_NOEXCEPT
{
    // 動的に変更可能な Settings の値のみの更新
    g_SettingsHolder.ReloadSettings();

    Update();
    // PSM の EnoughPowerChargeEmulation への遷移が Update に間に合わない場合は PSM の StateChangeEvent によって Update させる
}

}}} // namespace nn::apm::server
