﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_IntegratedDisplayManager.h>

#include <algorithm>
#include <cmath>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/vi/vi_PixelFormat.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/am/service/am_StuckChecker.h>

#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/am/service/am_AccountWrapper.h>
#include <nn/am/service/display/am_DisplayControlConfig.h>
#include <nn/am/service/display/am_DisplayControlLog.h>
#include <nn/am/service/display/am_DisplayPrimitiveOperation.h>
#include <nn/am/service/display/am_DisplayCaptureBufferOperation.h>
#include <nn/am/service/display/am_ScreenShotOperation.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>

namespace nn{ namespace am{ namespace service{ namespace display{

    IntegratedDisplayManager::IntegratedDisplayManager(const IntegratedDisplayParameter& param) NN_NOEXCEPT
        : m_Parameter(param)
        , m_SharedBufferHeapMemoryHandle(0)
        , m_SharedBufferHandle()
    {
        InitializeImpl(param);
        DisplayCaptureBufferOperation::Initialize(param, m_SharedBufferHandle);
        m_ApplicationCopyrightManager.Initialize(detail::ExtraSharedLayerResourceAccessor(this));
        m_AppletPresentationManager.Initialize(detail::AppletPresentationResourceAccessor(this));
        m_Mode = Mode_System;
        m_IsOperationModeLayerActive = false;

        // 起動ロゴを展開
        NN_AM_DISPCTRL_LOG_TRACE("  expanding startup log on shared frame buffer\n");
        {
            nn::vi::fbshare::SharedTextureOption option = {};
            option.transform   = nn::vi::ImageTransform_None;
            option.colorOption = nn::vi::fbshare::SharedTextureColorOption_None;
            option.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_Opaque;
            option.stacks      = OperationModeLayerStackFlags;
            NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::ExpandStartupLogoOnSharedFrameBuffer(m_SharedBufferHandle, param.startupLogoBufferIndex, option));
        }

        // 遷移用バッファをクリア
        NN_AM_DISPCTRL_LOG_TRACE("  filling expanding startup log on shared frame buffer\n");
        DisplayCaptureBufferOperation::FillBuffer(CaptureBufferIndex_LastForeground, ColorOpaqueBlackRgba32, false, false);
        DisplayCaptureBufferOperation::FillBuffer(CaptureBufferIndex_LastApplication, ColorOpaqueBlackRgba32, false, false);
        DisplayCaptureBufferOperation::FillBuffer(CaptureBufferIndex_CallerApplet, ColorOpaqueBlackRgba32, false, false);
    }

    IntegratedDisplayManager::~IntegratedDisplayManager() NN_NOEXCEPT
    {
        m_AppletPresentationManager.Finalize();
        m_ApplicationCopyrightManager.Finalize();
        DisplayCaptureBufferOperation::Finalize();
        FinalizeImpl();
    }

    void IntegratedDisplayManager::InitializeImpl(const IntegratedDisplayParameter& param) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("initializing IntegratedDisplayManager: %d buffer(s)\n", param.layout.count);

        // layout から必要なメモリサイズを計算
        int64_t requiredMemorySize = 0;
        for(int32_t i = 0; i < param.layout.count; i++)
        {
            auto& e = param.layout.entries[i];
            requiredMemorySize = std::max(requiredMemorySize, e.offset + e.size);
        }

        // 共有バッファを作成
        NN_AM_DISPCTRL_LOG_TRACE("  creating shared buffer on vi-process static-storage\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::CreateSharedBufferOnStaticStorage(&m_SharedBufferHandle, param.layout));

        // 出画用レイヤを作成
        NN_AM_DISPCTRL_LOG_TRACE("  initializing system-shared-fulscreen-layer\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_SystemSharedFullscreenLayer.Initialize(m_SharedBufferHandle, true, detail::IntegratedDisplayZOrder_SystemSharedFullscreen));
        // SIGLO-68694
        //   PartialLA と OMM 用のレイヤは必要なときに必要な間だけ作成する
        //NN_ABORT_UNLESS_RESULT_SUCCESS(m_SystemSharedPartialLayer.Initialize(m_SharedBufferHandle, false, detail::IntegratedDisplayZOrder_SystemSharedPartial));
        //NN_ABORT_UNLESS_RESULT_SUCCESS(m_OperationModeLayer.Initialize(m_SharedBufferHandle, true, detail::IntegratedDisplayZOrder_OperationMode));

        NN_AM_DISPCTRL_LOG_TRACE("initialized IntegratedDisplayManager\n");
    }

    void IntegratedDisplayManager::FinalizeImpl() NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("finalizing IntegratedDisplayManager\n");

        if(m_ApplicationCopyrightManager.IsInitialized())
        {
            m_ApplicationCopyrightManager.Finalize();
        }

        m_SystemSharedFullscreenLayer.Finalize();
        if(m_SystemSharedPartialLayer.IsInitialized())
        {
            m_SystemSharedPartialLayer.Finalize();
        }
        if(m_OperationModeLayer.IsInitialized())
        {
            m_OperationModeLayer.Finalize();
        }

        DisplayPrimitiveOperation::DestroySharedBuffer(m_SharedBufferHandle);

        if(m_SharedBufferHeapMemoryHandle != 0)
        {
            DisplayPrimitiveOperation::FreeDisplayProcessHeapBlock(m_SharedBufferHeapMemoryHandle);
        }

        m_SharedBufferHeapMemoryHandle = 0;
        m_SharedBufferHandle = {};
        NN_AM_DISPCTRL_LOG_TRACE("finalized IntegratedDisplayManager\n");
    }

    applet::AppletResourceUserId IntegratedDisplayManager::GetAruid() const NN_NOEXCEPT
    {
        return {0};
    }

    void IntegratedDisplayManager::ChangeToSystemMode() NN_NOEXCEPT
    {
        for(int i = 0; i < detail::IntegratedDisplayLayerPool::LayerCount; i++)
        {
            if(auto pVisMgr = m_AppletDisplayLayerPool.GetVisibilityManagerAt(i))
            {
                pVisMgr->Deactivate();
            }
        }

        m_AppletPresentationManager.Suspend();
        m_Mode = Mode_System;
    }

    void IntegratedDisplayManager::ChangeToAppletMode() NN_NOEXCEPT
    {
        EndOperationModeLayer();

        for(int i = 0; i < detail::IntegratedDisplayLayerPool::LayerCount; i++)
        {
            if(auto pVisMgr = m_AppletDisplayLayerPool.GetVisibilityManagerAt(i))
            {
                pVisMgr->Activate();
            }
        }

        m_AppletPresentationManager.Resume();
        m_Mode = Mode_Applet;
    }

    bool IntegratedDisplayManager::IsSystemBufferSharingEnabled() const NN_NOEXCEPT
    {
        return m_Parameter.isBufferShaingEnabled;
    }

    nn::vi::fbshare::SharedBufferHandle IntegratedDisplayManager::AcquireSystemSharedBufferHandle(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT
    {
        if(!IsSystemBufferSharingEnabled())
        {
            return {};
        }

        if(auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle))
        {
            if(!pLayer->IsSharedBufferExported())
            {
                pLayer->ExportSharedBuffer(m_SharedBufferHandle);
            }

            return m_SharedBufferHandle;
        }
        else
        {
            return {};
        }
    }

    IntegratedDisplayLayerHandle IntegratedDisplayManager::CreateIntegratedDisplayLayer() NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("creating IntegratedDisplayLayer\n");

        auto h = m_AppletDisplayLayerPool.Acquire();

        if(auto pLayer = m_AppletDisplayLayerPool.GetLayer(h))
        {
            pLayer->Initialize();
            auto pVisMgr = m_AppletDisplayLayerPool.GetVisibilityManager(h);
            NN_ABORT_UNLESS_NOT_NULL(pVisMgr);
            pVisMgr->Initialize();
            if(m_Mode == Mode_Applet)
            {
                pVisMgr->Activate();
            }
        }

        return h;
    }

    void IntegratedDisplayManager::DestroyIntegratedDisplayLayer(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("destroying IntegratedDisplayLayer\n");
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_SDK_REQUIRES_NOT_NULL(pLayer);
        NN_SDK_REQUIRES(pLayer->GetState() != detail::IntegratedDisplayLayerState_NotInitialized);
        auto pVisibilityManager = m_AppletDisplayLayerPool.GetVisibilityManager(handle);

        // 非表示にしておく
        SetLayerVisibility(handle, false, detail::IntegratedDisplayZOrder_Default, false);
        NN_SDK_ASSERT_NOT_EQUAL(handle, m_AppletPresentationManager.GetCurrentActiveFullscreenLayerHandle());
        NN_SDK_ASSERT_NOT_EQUAL(handle, m_AppletPresentationManager.GetCurrentActivePartialLayerHandle());
        NN_SDK_ASSERT_NOT_EQUAL(handle, m_AppletPresentationManager.GetCurrentActiveOverlayLayerHandle());

        pVisibilityManager->Finalize();
        pLayer->Finalize();
        m_AppletDisplayLayerPool.Release(handle);
    }

    detail::IntegratedDisplayLayerUserClass IntegratedDisplayManager::GetUserClass(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        return pLayer->GetUserClass();
    }

    detail::IntegratedDisplayLayerBufferOwner IntegratedDisplayManager::GetBufferOwner(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        return pLayer->GetBufferOwner();
    }

    detail::IntegratedDisplayLayerDestinationLayer IntegratedDisplayManager::GetDestinationLayer(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        return pLayer->GetDestinationLayer();
    }

    bool IntegratedDisplayManager::SpecifyUserClass(IntegratedDisplayLayerHandle handle, detail::IntegratedDisplayLayerUserClass value, applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        if(pLayer->SpecifyUserClass(value, aruid))
        {
            OnIntegratedLayerConfigurationDetermined(handle);
            return true;
        }
        return false;
    }

    bool IntegratedDisplayManager::SpecifyBufferOwner(IntegratedDisplayLayerHandle handle, detail::IntegratedDisplayLayerBufferOwner value) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        if(pLayer->SpecifyBufferOwner(value, m_SharedBufferHandle))
        {
            OnIntegratedLayerConfigurationDetermined(handle);
            return true;
        }
        return false;
    }

    bool IntegratedDisplayManager::SpecifyDestinationLayer(IntegratedDisplayLayerHandle handle, detail::IntegratedDisplayLayerDestinationLayer value, nn::vi::IndirectLayerHandleType indirectLayerHandle) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        if(pLayer->SpecifyDestinationLayer(value, indirectLayerHandle, m_Parameter.indirectProducerFlipOffset))
        {
            OnIntegratedLayerConfigurationDetermined(handle);
            return true;
        }
        return false;
    }

    void IntegratedDisplayManager::OnIntegratedLayerConfigurationDetermined(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        auto pVisMgr = m_AppletDisplayLayerPool.GetVisibilityManager(handle);
        NN_ABORT_UNLESS_NOT_NULL(pVisMgr);
        pVisMgr->SetLayerId(pLayer->GetDirectLayerId());
        UpdateDebugLayerStackEnabledImpl(handle);
        UpdateRecordingLayerStackEnabledImpl(handle);
        UpdateActiveLayerImpl();
    }

    void IntegratedDisplayManager::SetScreenShotPermissionGetterFunction(IntegratedDisplayLayerHandle handle, bool (*func)(void*), void* userPtr) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        pLayer->SetScreenShotPermissionGetterFunction(func, userPtr);
    }

    void IntegratedDisplayManager::SetLayerVisibility(IntegratedDisplayLayerHandle handle, bool isVisible, int32_t zOrder, bool isCaptureTarget) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("setting layer visibility(h=%lld,vis=%s,z=%d,cap=%s)\n", handle, isVisible ? "y" : "n", zOrder, isCaptureTarget ? "y" : "n");
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        pLayer->SetVisibility(isVisible, zOrder, isCaptureTarget);

        if(pLayer->GetState() == detail::IntegratedDisplayLayerState_Determined)
        {
            UpdateActiveLayerImpl();
        }
    }

    void IntegratedDisplayManager::SetLayerTransitionLayerRequested(IntegratedDisplayLayerHandle handle, bool value) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        pLayer->SetTransitionLayerRequested(value);
    }

    void IntegratedDisplayManager::SetLayerConductorIfForeground(IntegratedDisplayLayerHandle handle, bool value) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        pLayer->SetConductorIfForeground(value);
        UpdateConductorImpl(handle);
    }

    void IntegratedDisplayManager::SetRecordingLayerStackEnabled(IntegratedDisplayLayerHandle handle, bool value) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        pLayer->SetRecordingLayerStackRequested(value);
        UpdateRecordingLayerStackEnabledImpl(handle);
    }

    void IntegratedDisplayManager::SetDebugLayerStackEnabled(IntegratedDisplayLayerHandle handle, bool value) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        pLayer->SetDebugLayerStackRequested(value);
        UpdateDebugLayerStackEnabledImpl(handle);
    }

    nn::vi::LayerId IntegratedDisplayManager::GetDirectLayerId(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        return pLayer->GetDirectLayerId();
    }

    nn::vi::IndirectProducerHandleType IntegratedDisplayManager::GetIndirectProducerHandle(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        return pLayer->GetIndirectProducerHandle();
    }

    nn::vi::fbshare::SharedLayerHandle IntegratedDisplayManager::GetSharedLayerHandle(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);
        return pLayer->GetSharedLayerHandle();
    }


    void IntegratedDisplayManager::UpdateActiveLayerImpl() NN_NOEXCEPT
    {
        m_AppletPresentationManager.UpdateActiveLayer();
    }

    void IntegratedDisplayManager::UpdateConductorImpl(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT
    {
        m_AppletPresentationManager.NotifyConductorChanged(handle);
    }

    void IntegratedDisplayManager::UpdateRecordingLayerStackEnabledImpl(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);

        auto pVisMgr = m_AppletDisplayLayerPool.GetVisibilityManager(handle);
        NN_ABORT_UNLESS_NOT_NULL(pVisMgr);

        if(pLayer->GetState() == detail::IntegratedDisplayLayerState_Determined)
        {
            if(pLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application
                && pLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet
                && pLayer->GetDestinationLayer() == detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer
                )
            {
                pVisMgr->SetMovieRecordingVisibility(pLayer->IsRecordingLayerStackRequested());
            }
        }
    }

    void IntegratedDisplayManager::UpdateDebugLayerStackEnabledImpl(IntegratedDisplayLayerHandle handle) NN_NOEXCEPT
    {
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);

        auto pVisMgr = m_AppletDisplayLayerPool.GetVisibilityManager(handle);
        NN_ABORT_UNLESS_NOT_NULL(pVisMgr);

        if(pLayer->GetState() == detail::IntegratedDisplayLayerState_Determined)
        {
            if(pLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application
                && pLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet
                && pLayer->GetDestinationLayer() == detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer
                )
            {
                pVisMgr->SetDebugScreenshotVisibility(pLayer->IsDebugLayerStackRequested());
            }
        }
    }

    void IntegratedDisplayManager::ImmediateCaptureCurrentFullscreenLayerImpl(CaptureBufferIndex index, bool isScreenShotPermitted) NN_NOEXCEPT
    {
        m_AppletPresentationManager.ImmediateCaptureCurrentFullscreenLayer(index, isScreenShotPermitted);
    }


    //-----------------------------------------

    nn::Result IntegratedDisplayManager::ReadCaptureBufferImage(bool* pOutIsScreenShotPermitted, void* buffer, size_t bufferSize, CaptureBufferIndex srcIndex) NN_NOEXCEPT
    {
        NN_RESULT_DO(DisplayCaptureBufferOperation::ReadBuffer(buffer, bufferSize, srcIndex));
        *pOutIsScreenShotPermitted = DisplayCaptureBufferOperation::IsBufferScreenshotPermittedForFader(srcIndex);
        NN_RESULT_SUCCESS;
    }

    nn::Result IntegratedDisplayManager::ClearCaptureBufferImage(CaptureBufferIndex index, Bit32 color, bool isCaptureEnabled) NN_NOEXCEPT
    {
        DisplayCaptureBufferOperation::FillBuffer(index, color, isCaptureEnabled, false);
        NN_RESULT_SUCCESS;
    }

    nn::Result IntegratedDisplayManager::CopyCaptureBufferImage(CaptureBufferIndex dstIndex, CaptureBufferIndex srcIndex) NN_NOEXCEPT
    {
        DisplayCaptureBufferOperation::CopyBuffer(dstIndex, srcIndex, m_Parameter.captureBufferTimeout);
        NN_RESULT_SUCCESS;
    }

    nn::Result IntegratedDisplayManager::TryCaptureCurrentForegroundImage(CaptureBufferIndex index) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("trying capturing current foreground image to capture buffer %d\n", index);

        bool existsTarget = false;
        bool isTargetScreenShotPermitted = false;
        auto hLayer = m_AppletPresentationManager.GetCurrentActiveFullscreenLayerHandle();
        if(auto pLayer = m_AppletDisplayLayerPool.GetLayer(hLayer))
        {
            if(pLayer->IsCaptureTarget())
            {
                existsTarget = true;
                isTargetScreenShotPermitted = pLayer->InvokeScreenShotPermissionGetterFunction();
            }
        }

        if(existsTarget)
        {
            NN_AM_DISPCTRL_LOG_TRACE("  capture (screenshot=%s)\n", isTargetScreenShotPermitted ? "y" : "n");
            DisplayCaptureBufferOperation::CaptureDisplay(index, nn::vi::LayerStack_LastFrame, isTargetScreenShotPermitted, false, m_Parameter.captureBufferTimeout);
        }
        else
        {
            NN_AM_DISPCTRL_LOG_TRACE("  no capture target. clearing capture buffer.\n");
            DisplayCaptureBufferOperation::FillBuffer(index, CaptureBufferColorBlack, true, false);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result IntegratedDisplayManager::TryCaptureCurrentLayerImage(
        CaptureBufferIndex index,
        IntegratedDisplayLayerHandle handle,
        bool isScreenShotPermitted,
        bool isImmediate
    ) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("trying capturing current layer image to capture buffer %d (h=%lld)\n", index, handle);

        auto pLayer = m_AppletDisplayLayerPool.GetLayer(handle);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);

        // SIGLO-68658
        //   LA が自撮り API を呼び出す直前に HOME ボタンを押された場合、
        //   呼出元 LA が FG でない場合がある。
        //   BG の LA からの呼出に対しては
        //   - 自前バッファモード：従来どおり最終出画をキャプチャする
        //   - 共有バッファモード：LastFG を CallerApplet にコピーする
        //   としておく。
        //
        //   LA が自撮り API を呼ぼうとした時点で LA の最終フレームの描画は行われているため、
        //   LastFG の内容が LA の最終フレームということになる（間に複雑な遷移があると意図しない絵になるかもしれない）

        bool isTargetScreenShotPermitted = isScreenShotPermitted && pLayer->InvokeScreenShotPermissionGetterFunction();

        if(pLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet)
        {
            // 自前レイヤ: 即時キャプチャ
            auto layerId = pLayer->GetDirectLayerId();
            if(layerId)
            {
                auto pVisMgr = m_AppletDisplayLayerPool.GetVisibilityManager(handle);
                NN_ABORT_UNLESS_NOT_NULL(pVisMgr);
                pVisMgr->SetInstantCaptureVisibility(true);
                NN_UTIL_SCOPE_EXIT{ pVisMgr->SetInstantCaptureVisibility(false); };
                NN_AM_DISPCTRL_LOG_TRACE("  capture (screenshot=%s)\n", isTargetScreenShotPermitted ? "y" : "n");
                DisplayCaptureBufferOperation::CaptureDisplay(index, nn::vi::LayerStack_Arbitrary, isTargetScreenShotPermitted, false, m_Parameter.captureBufferTimeout);
            }
            else
            {
                NN_AM_DISPCTRL_LOG_TRACE("  no capture target. clearing capture buffer.\n");
                DisplayCaptureBufferOperation::FillBuffer(index, CaptureBufferColorBlack, true, false);
            }
        }
        else if(pLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_SystemShared)
        {
            if(index != CaptureBufferIndex_CallerApplet)
            {
                // CallerApplet 以外へのキャプチャは無視する
                NN_AM_DISPCTRL_LOG_WARN("requested capture to invalid buffer index (->%d)\n", index);
            }
            else if(m_AppletPresentationManager.GetCurrentActiveFullscreenLayerHandle() == handle)
            {
                // 共有レイヤ(FG): FG を失う時点まで遅延
                NN_AM_DISPCTRL_LOG_TRACE("  requesting caller-applet capture.\n");

                // SIGLO-76657, SIGLO-80771
                // 即時キャプチャを要求されている (SIGLO-80771) か、
                // 互換モードの場合 (SIGLO-76657) 、リクエストされた瞬間にキャプチャする。
                if(isImmediate || m_Parameter.isImmediateCallerAppletCaptureForced)
                {
                    // システム共有レイヤのレイヤスタックは vi が管理しているので非共有レイヤと同じようにキャプチャできない。
                    ImmediateCaptureCurrentFullscreenLayerImpl(index, isScreenShotPermitted);
                }
                else
                {
                    pLayer->SetCallerAppletCaptureRequested(true, isTargetScreenShotPermitted);
                }
            }
            else
            {
                // 共有レイヤ(BG): LastFg をコピー
                NN_AM_DISPCTRL_LOG_TRACE("  copying last-foreground to caller-applet.\n");
                DisplayCaptureBufferOperation::CopyBuffer(CaptureBufferIndex_CallerApplet, CaptureBufferIndex_LastForeground, m_Parameter.captureBufferTimeout);
            }
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result IntegratedDisplayManager::AcquireCaptureBuffer(bool* pOutIsScreenShotPermitted, int* pOutBufferIndex, IntegratedDisplayLayerHandle handle, CaptureBufferIndex captureIndex) NN_NOEXCEPT
    {
        NN_UNUSED(handle);
        NN_RESULT_THROW_UNLESS(m_Parameter.isBufferShaingEnabled, ResultNotSupported());

        // SIGLO-75711
        //   呼出元が FG を取っていることをチェックしない。
        //   FG でないアプレットからの呼出であってもその瞬間のバッファの情報を返す。

        // TORIAEZU:
        //   単にインデックスと撮影フラグだけ取って返る。
        //   使用権の追跡が必要なら実装する。

        int bufferIndex = -1;
        switch(captureIndex)
        {
        case CaptureBufferIndex_LastApplication:
            bufferIndex = m_Parameter.lastApplicationBufferIndex;
            break;
        case CaptureBufferIndex_LastForeground:
            bufferIndex = m_Parameter.lastForegroundBufferIndex;
            break;
        case CaptureBufferIndex_CallerApplet:
            bufferIndex = m_Parameter.callerAppletBufferIndex;
            break;
        default:
            NN_RESULT_THROW(ResultNotSupported());
        }

        *pOutBufferIndex = bufferIndex;
        *pOutIsScreenShotPermitted = DisplayCaptureBufferOperation::IsBufferScreenshotPermittedForFader(captureIndex);
        NN_RESULT_SUCCESS;
    }

    nn::Result IntegratedDisplayManager::ReleaseCaptureBuffer(IntegratedDisplayLayerHandle handle, CaptureBufferIndex index) NN_NOEXCEPT
    {
        NN_UNUSED(handle);
        NN_UNUSED(index);
        NN_RESULT_THROW_UNLESS(m_Parameter.isBufferShaingEnabled, ResultNotSupported());
        // 特になにもしない
        NN_RESULT_SUCCESS;
    }

    //-----------------------------------------

    void IntegratedDisplayManager::SetSystemLayerVisibility(bool value) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("setting system layer visibility (%s)\n", value ? "visible" : "invisible");

        if(value)
        {
            m_AppletPresentationManager.ShowTransitionBuffer();
        }
        else
        {
            m_AppletPresentationManager.HideTransitionBuffer();
        }
    }

    nn::Result IntegratedDisplayManager::ClearTransitionBufferImage(uint32_t color) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("clearing transition buffer(%d) image (0x%08X)\n", m_Parameter.transitionBufferIndex, color);
        //NN_AM_DISPCTRL_LOG_WARN("clearing transition buffer image (0x%08X)\n", color);
        if(m_Parameter.transitionBufferIndex < 0)
        {
            NN_RESULT_SUCCESS;
        }

        nn::vi::fbshare::SharedTextureOption option = {};
        option.transform   = nn::vi::ImageTransform_None;
        option.colorOption = nn::vi::fbshare::SharedTextureColorOption_None;
        option.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_Opaque;
        option.stacks      = TransitionLayerStackFlags;
        return DisplayPrimitiveOperation::FillDetachedSharedFrameBufferColor(
            m_SharedBufferHandle,
            m_Parameter.transitionBufferIndex,
            color,
            option
        );
    }

    //-----------------------------------------
    // OperationModeLayer

    void IntegratedDisplayManager::BeginOperationModeLayer() NN_NOEXCEPT
    {
        // SIGLO-77289:
        // OMM の出画がアプレットの出画と競合するのを回避するため、
        // ディスプレイ制御をシステムモードに変更する必要がある。
        NN_SDK_REQUIRES_EQUAL(m_Mode, Mode_System);

        NN_AM_DISPCTRL_LOG_TRACE("begin operation-mode-layer\n");
        if(m_IsOperationModeLayerActive)
        {
            NN_AM_DISPCTRL_LOG_WARN("operation-mode-layer is already active\n");
            return;
        }
        // SIGLO-68694: 必要になったら初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_OperationModeLayer.Initialize(m_SharedBufferHandle, true, detail::IntegratedDisplayZOrder_OperationMode));
        m_IsOperationModeLayerActive = true;
    }

    void IntegratedDisplayManager::EndOperationModeLayer() NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("end operation-mode-layer\n");
        if(!m_IsOperationModeLayerActive)
        {
            NN_AM_DISPCTRL_LOG_WARN("operation-mode-layer is already inactive\n");
            return;
        }
        HideOperationModeTexture();
        // SIGLO-68694: 不要になったら破棄
        m_OperationModeLayer.Finalize();
        m_IsOperationModeLayerActive = false;
    }

    void IntegratedDisplayManager::SetOperationModeTexture(int x, int y, int width, int height, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("setting operation-mode-texture(%d): x%dy%d w%dh%d\n", m_Parameter.operationModeBufferIndex, x, y, width, height);

        if(!m_IsOperationModeLayerActive)
        {
            NN_AM_DISPCTRL_LOG_WARN("operation-mode-layer is not active\n");
            return;
        }

        nn::vi::fbshare::SharedTextureOption dstOption = {};
        dstOption.transform   = nn::vi::ImageTransform_None;
        dstOption.colorOption = nn::vi::fbshare::SharedTextureColorOption_None;
        dstOption.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_Opaque;
        dstOption.stacks      = OperationModeLayerStackFlags;

        nn::vi::ImageTransform srcTransform = nn::vi::ImageTransform_None;

        NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::SetDetachedSharedFrameBufferSubImage(
            m_SharedBufferHandle,
            static_cast<int64_t>(m_Parameter.operationModeBufferIndex),
            static_cast<int32_t>(x),
            static_cast<int32_t>(y),
            static_cast<int32_t>(width),
            static_cast<int32_t>(height),
            CaptureBufferColorBlack,
            buffer,
            size,
            dstOption,
            srcTransform
        ));
    }

    void IntegratedDisplayManager::ShowOperationModeTexture() NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("show operation-mode-texture\n");

        if(!m_IsOperationModeLayerActive)
        {
            NN_AM_DISPCTRL_LOG_WARN("operation-mode-layer is not active\n");
            return;
        }

        if(m_Parameter.operationModeBufferIndex < 0)
        {
            return;
        }

        int presentedIndex = -1;
        DisplayPrimitiveOperation::SynchronizeSharedLowLevelLayer(&presentedIndex, m_OperationModeLayer.GetLayerId(), m_OperationModeLayer.GetSynchronizedEvent());
        if(presentedIndex != m_Parameter.operationModeBufferIndex)
        {
            auto presentResult = DisplayPrimitiveOperation::PresentDetachedSharedFrameBufferToLowLevelLayer(m_SharedBufferHandle, m_OperationModeLayer.GetLayerId(), static_cast<int64_t>(m_Parameter.operationModeBufferIndex));
            NN_AM_DISPCTRL_LOG_TRACE("  presenting operation-mode-texture(%d) -> %d-%d\n", m_Parameter.operationModeBufferIndex, presentResult.GetModule(), presentResult.GetDescription());
            NN_UNUSED(presentResult);
        }
        else
        {
            NN_AM_DISPCTRL_LOG_TRACE("  no-op (operation-mode-texture is already presented)\n");
        }
    }

    void IntegratedDisplayManager::ShowOperationModeStartupLogoTexture() NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("show startup-logo-texture\n");

        if(!m_IsOperationModeLayerActive)
        {
            NN_AM_DISPCTRL_LOG_WARN("operation-mode-layer is not active\n");
            return;
        }

        if(m_Parameter.startupLogoBufferIndex < 0)
        {
            return;
        }

        int presentedIndex = -1;
        DisplayPrimitiveOperation::SynchronizeSharedLowLevelLayer(&presentedIndex, m_OperationModeLayer.GetLayerId(), m_OperationModeLayer.GetSynchronizedEvent());
        if(presentedIndex != m_Parameter.startupLogoBufferIndex)
        {
            auto presentResult = DisplayPrimitiveOperation::PresentDetachedSharedFrameBufferToLowLevelLayer(m_SharedBufferHandle, m_OperationModeLayer.GetLayerId(), static_cast<int64_t>(m_Parameter.startupLogoBufferIndex));
            NN_AM_DISPCTRL_LOG_TRACE("  presenting startup-logo-texture(%d) -> %d-%d\n", m_Parameter.startupLogoBufferIndex, presentResult.GetModule(), presentResult.GetDescription());
            NN_UNUSED(presentResult);
        }
        else
        {
            NN_AM_DISPCTRL_LOG_TRACE("  no-op (startup-logo-texture is already presented)\n");
        }
    }


    void IntegratedDisplayManager::HideOperationModeTexture() NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("hide operation-mode-texture\n");
        if(!m_IsOperationModeLayerActive)
        {
            //NN_AM_DISPCTRL_LOG_WARN("  operation-mode-layer is not active\n");
        }

        // NOTE: OMM レイヤ以外に影響ないので inactive でも呼ぶ
        DisplayPrimitiveOperation::BlankSharedLowLevelLayer(m_OperationModeLayer.GetLayerId(), m_SharedBufferHandle);
    }

    //-----------------------------------------
    // アプリ権利表記用レイヤーの作成
    // メモリはアプリからもらった TransferMemory を vi に渡す

    Result IntegratedDisplayManager::InitializeApplicationCopyrightSharedBuffer(sf::NativeHandle trmemHandle, size_t trmemSize, int width, int height) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(trmemSize % (256u * 1024) == 0, am::ResultInvalidParameter());
        NN_RESULT_THROW_UNLESS(!m_ApplicationCopyrightManager.IsLayerCreated(), am::ResultInvalidCall());

        vi::fbshare::SharedMemoryPoolLayout layout = {};
        layout.count = 1;
        layout.entries[0].width  = width;
        layout.entries[0].height = height;
        layout.entries[0].offset = 0;
        layout.entries[0].size   = trmemSize;

        NN_RESULT_DO(m_ApplicationCopyrightManager.CreateLayer(
            std::move(trmemHandle),
            trmemSize,
            layout,
            detail::IntegratedDisplayZOrder_ApplicationCopyright,
            ApplicationCopyrightLayerStackFlags
        ));

        // Visibility の初期値は true
        m_ApplicationCopyrightManager.RequestShow(0);
        NN_RESULT_SUCCESS;
    }

    void IntegratedDisplayManager::FinalizeApplicationCopyrightSharedBuffer() NN_NOEXCEPT
    {
        if(m_ApplicationCopyrightManager.IsLayerCreated())
        {
            m_ApplicationCopyrightManager.DestroyLayer();
        }
    }

    Result IntegratedDisplayManager::SetApplicationCopyrightTexture(int x, int y, int width, int height, const void* buffer, size_t size, int originMode) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_ApplicationCopyrightManager.IsLayerCreated(), ResultApplicationCopyrightFrameBufferNotInitialized());

        NN_AM_DISPCTRL_LOG_TRACE("SetApplicationCopyrightTexture(): x%d y%d w%d h%d\n", x, y, width, height);

        // SIGLO-75629
        // アプリが FG の場合にはシステム共有バッファ経由で書き込む
        bool isTemporalBufferRequired = false;
        if(m_Parameter.copyrightTemporalBufferIndex >= 0)
        {
            auto hActiveLayer = m_AppletPresentationManager.GetCurrentActiveFullscreenLayerHandle();
            if(auto pActiveLayer = m_AppletDisplayLayerPool.GetLayer(hActiveLayer))
            {
                if(pActiveLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application)
                {
                    NN_SDK_ASSERT_EQUAL(pActiveLayer->GetBufferOwner(), detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet);
                    isTemporalBufferRequired = true;
                }
            }
        }

        // 引数の x, y, w, h はすべて画面写真／動画上の座標系（左上原点）で測った座標を表す。
        // また buffer はピクセル値を画像の左上から右下に向かって並べたもの(Transform_None)

        if(!isTemporalBufferRequired)
        {
            // 直接書込み
            auto result = m_ApplicationCopyrightManager.SetTexture(0, x, y, width, height, buffer, size, originMode);
            NN_AM_DISPCTRL_LOG_TRACE("  setting application-copyright-texture(direct) -> %d-%d\n", result.GetModule(), result.GetDescription());
            NN_RESULT_DO(result);
        }
        else
        {
            // 一時バッファ経由で書込み
            auto result = m_ApplicationCopyrightManager.SetTextureThroughTempTexture(0, x, y, width, height, buffer, size, originMode, m_Parameter.copyrightTemporalBufferIndex);
            NN_AM_DISPCTRL_LOG_TRACE("  setting application-copyright-texture(temp) -> %d-%d\n", result.GetModule(), result.GetDescription());
            NN_RESULT_DO(result);
        }

        NN_RESULT_SUCCESS;
    }

    bool IntegratedDisplayManager::IsApplicationCopyrightTextureSet() NN_NOEXCEPT
    {
        return
            m_ApplicationCopyrightManager.IsLayerCreated() &&
            m_ApplicationCopyrightManager.HasImage(0);
    }

    void IntegratedDisplayManager::GetApplicationCopyrightInfoForScreenShot(capsrv::ScreenShotAttribute* pAttribute) NN_NOEXCEPT
    {
        if (m_ApplicationCopyrightManager.IsLayerCreated() &&
            m_ApplicationCopyrightManager.HasImage(0) &&
            m_ApplicationCopyrightManager.IsPresentationRequested())
        {
            uint32_t flags =
                nn::capsrv::detail::ScreenShotAttributeFlag_IsCopyrightImageComposited |
                nn::capsrv::detail::ScreenShotAttributeFlag_HasUneditableArea;

            const int sbufW = m_ApplicationCopyrightManager.GetBufferWidth(0);
            const int sbufH = m_ApplicationCopyrightManager.GetBufferHeight(0);
            const int srectX = m_ApplicationCopyrightManager.GetImageRegionX(0);
            const int srectY = m_ApplicationCopyrightManager.GetImageRegionY(0);
            const int srectW = m_ApplicationCopyrightManager.GetImageRegionWidth(0);
            const int srectH = m_ApplicationCopyrightManager.GetImageRegionHeight(0);

            // 静止画の座標系に変換
            const int dbufW = 1280;
            const int dbufH = 720;
            int drectX = static_cast<int>(std::floor(static_cast<float>((srectX         ) * dbufW) / static_cast<float>(sbufW)));
            int drectY = static_cast<int>(std::floor(static_cast<float>((srectY         ) * dbufH) / static_cast<float>(sbufH)));
            int drectR = static_cast<int>(std::ceil (static_cast<float>((srectX + srectW) * dbufW) / static_cast<float>(sbufW)));
            int drectB = static_cast<int>(std::ceil (static_cast<float>((srectY + srectH) * dbufH) / static_cast<float>(sbufH)));

            // 範囲内に収まるように調整
            drectX = std::max<int>(drectX, 0);
            drectY = std::max<int>(drectY, 0);
            drectR = std::min<int>(drectR, dbufW);
            drectB = std::min<int>(drectB, dbufH);

            int drectW = drectR - drectX;
            int drectH = drectB - drectY;

            // 結局範囲が空なら権利表記なし扱いにする
            if(drectH <= 0 || drectW <= 0)
            {
                flags = 0;
                drectX = 0;
                drectY = 0;
                drectW = 0;
                drectH = 0;
            }

            NN_SDK_ASSERT_RANGE (drectX, 0, dbufW);
            NN_SDK_ASSERT_RANGE (drectY, 0, dbufH);
            NN_SDK_ASSERT_MINMAX(drectW, 0, dbufW);
            NN_SDK_ASSERT_MINMAX(drectH, 0, dbufH);
            NN_SDK_ASSERT_MINMAX(drectX + drectW, 0, dbufW);
            NN_SDK_ASSERT_MINMAX(drectY + drectH, 0, dbufH);

            pAttribute->flags |= flags;
            pAttribute->uneditableAreaCoordinateX = static_cast<uint16_t>(drectX);
            pAttribute->uneditableAreaCoordinateY = static_cast<uint16_t>(drectY);
            pAttribute->uneditableAreaWidth       = static_cast<uint16_t>(drectW);
            pAttribute->uneditableAreaHeight      = static_cast<uint16_t>(drectH);
        }
        else
        {
            pAttribute->uneditableAreaCoordinateX = 0;
            pAttribute->uneditableAreaCoordinateY = 0;
            pAttribute->uneditableAreaWidth  = 0;
            pAttribute->uneditableAreaHeight = 0;
        }

        NN_AM_DISPCTRL_LOG_TRACE("application copyright info(x%d,y%d,w%d,h%d;flags=%X)\n",
            pAttribute->uneditableAreaCoordinateX,
            pAttribute->uneditableAreaCoordinateY,
            pAttribute->uneditableAreaWidth,
            pAttribute->uneditableAreaHeight,
            pAttribute->flags
        );

    }

    void IntegratedDisplayManager::SetApplicationCopyrightVisibility(bool isVisible) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("SetApplicationCopyrightVisibility(%s)\n", isVisible ? "true" : "false");
        if(isVisible)
        {
            m_ApplicationCopyrightManager.RequestShow(0);
        }
        else
        {
            m_ApplicationCopyrightManager.RequestHide();
        }
    }

    //-----------------------------------------

    void IntegratedDisplayManager::RequestTakingScreenShot(const ScreenShotRequest& request) NN_NOEXCEPT
    {
        // アプレットモード以外の場合は無視する
        if(m_Mode != Mode_Applet)
        {
            NN_AM_DISPCTRL_LOG_INFO("TakingScreenShotRequest Ignored (invalid-mode)\n");
            return;
        }

        // FG が変わっていないことを確認。
        // FG が変化する際にキャンセル要求が出されるため、ここで ABA 問題は気にしなくてよい。
        auto hLayer = m_AppletPresentationManager.GetCurrentActiveFullscreenLayerHandle();
        auto pLayer = m_AppletDisplayLayerPool.GetLayer(hLayer);
        if(pLayer == nullptr || hLayer != request.GetLayerHandle())
        {
            NN_AM_DISPCTRL_LOG_INFO("TakingScreenShotRequest Expired: fgLayer=%d, reqLayer=%d\n", m_AppletPresentationManager.GetCurrentActiveFullscreenLayerHandle(), request.GetLayerHandle());

            // 失敗を通知
            (void)ScreenShotOperation::NotifyScreenShotFailed(
                request.GetRequestTick(),
                nn::ncm::ApplicationId::GetInvalidId(),
                {0}
            );

            return;
        }

        nn::capsrv::ScreenShotAttribute attribute = {};
        attribute.SetDefault();
        attribute.orientation = request.GetOrienation();
        if(pLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application)
        {
            GetApplicationCopyrightInfoForScreenShot(&attribute);
        }

        capsrv::UserIdList userIdList = {};
        if (request.IsOpenUserIdListRequired())
        {
            GetOpenUserIdList(&userIdList);
        }

        (void)ScreenShotOperation::RequestTakingScreenShot(
            request.GetRequestTick(),
            request.GetApplicationId(),
            pLayer->GetAruid(),
            attribute,
            userIdList,
            nn::TimeSpan::FromMilliSeconds(100)
        );
    }

    void IntegratedDisplayManager::NotifyTakingScreenShotRefused(const ScreenShotRequest& request) NN_NOEXCEPT
    {
        // アプレットモード以外の場合は無視する
        if(m_Mode != Mode_Applet)
        {
            NN_AM_DISPCTRL_LOG_INFO("TakingScreenShotRequest Ignored (invalid-mode)\n");
            return;
        }

        NN_AM_DISPCTRL_LOG_INFO("TakingScreenShotRequest Refused\n");
        (void)ScreenShotOperation::NotifyScreenShotRefused(
            request.GetRequestTick(),
            request.GetApplicationId(),
            { 0 }
        );
    }

    void IntegratedDisplayManager::NotifyTakingScreenShotFailed(const ScreenShotRequest& request) NN_NOEXCEPT
    {
        // アプレットモード以外の場合は無視する
        if(m_Mode != Mode_Applet)
        {
            NN_AM_DISPCTRL_LOG_INFO("TakingScreenShotRequest Ignored (invalid-mode)\n");
            return;
        }

        NN_AM_DISPCTRL_LOG_INFO("TakingScreenShotRequest Failed\n");
        (void)ScreenShotOperation::NotifyScreenShotFailed(
            request.GetRequestTick(),
            request.GetApplicationId(),
            { 0 }
        );
    }
    //-----------------------------------------

    #ifdef NN_AM_DISPCTRL_IS_DUMP_ENABLED
    void IntegratedDisplayManager::Dump() NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_DUMP("IntegratedDisplayManager\n");

    // Mode
        NN_AM_DISPCTRL_LOG_DUMP("  mode=%s\n",
            [](Mode mode) -> const char*{
                switch(mode)
                {
                case Mode_System: return "System";
                case Mode_Applet: return "Applet";
                default: return "N/A";
                }
            }(m_Mode)
        );

    // AppletDisplayLayerPool
        DumpIntegratedLayerListImpl();
        DumpVisibilityManagerListImpl();

    // Current
        NN_AM_DISPCTRL_LOG_DUMP("  ful=%lld / par=%lld / ovl=%lld\n",
            m_AppletPresentationManager.GetCurrentActiveFullscreenLayerHandle(),
            m_AppletPresentationManager.GetCurrentActivePartialLayerHandle(),
            m_AppletPresentationManager.GetCurrentActiveOverlayLayerHandle()
        );

    // OMM
        NN_AM_DISPCTRL_LOG_DUMP("  omm: %s\n", m_IsOperationModeLayerActive ? "active" : "inactive");

    // Copyright
        {
            NN_AM_DISPCTRL_LOG_DUMP("  (c): %s / req=%d / show=%d\n",
                m_ApplicationCopyrightManager.IsLayerCreated() ? "created" : "no-layer",
                m_ApplicationCopyrightManager.GetRequestedTextureIndex(),
                m_ApplicationCopyrightManager.GetPresentedTextureIndex()
            );
            NN_AM_DISPCTRL_LOG_DUMP("       x=%d, y=%d, w=%d, h=%d\n",
                m_ApplicationCopyrightManager.GetImageRegionX(0),
                m_ApplicationCopyrightManager.GetImageRegionY(0),
                m_ApplicationCopyrightManager.GetImageRegionWidth(0),
                m_ApplicationCopyrightManager.GetImageRegionHeight(0)
            );
        }

        //nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    }// NOLINT(impl/function_size)

    void IntegratedDisplayManager::DumpIntegratedLayerListImpl() NN_NOEXCEPT
    {
        int layerCountFree = 0;
        int layerCountNotInitialized = 0;
        int layerCountUndetermined = 0;
        int layerCountDetermined = 0;
        for(int i = 0; i < detail::IntegratedDisplayLayerPool::LayerCount; i++)
        {
            if(auto pLayer = m_AppletDisplayLayerPool.GetLayerAt(i))
            {
                switch(pLayer->GetState())
                {
                case detail::IntegratedDisplayLayerState_NotInitialized:
                    layerCountNotInitialized++;
                    break;
                case detail::IntegratedDisplayLayerState_Undetermined:
                    layerCountUndetermined++;
                    break;
                case detail::IntegratedDisplayLayerState_Determined:
                    layerCountDetermined++;
                    break;
                default:
                    break;
                }
            }
            else
            {
                layerCountFree++;
            }
        }

        NN_AM_DISPCTRL_LOG_DUMP("  D%d U%d E%d F%d\n",
            layerCountDetermined,
            layerCountUndetermined,
            layerCountNotInitialized,
            layerCountFree
        );

        auto getUserClassString = [](detail::IntegratedDisplayLayerUserClass v) -> const char*{
            switch(v)
            {
            case detail::IntegratedDisplayLayerUserClass_Application:      return "app";
            case detail::IntegratedDisplayLayerUserClass_FullscreenApplet: return "ful";
            case detail::IntegratedDisplayLayerUserClass_PartialApplet:    return "par";
            case detail::IntegratedDisplayLayerUserClass_Overlay:          return "ovl";
            case detail::IntegratedDisplayLayerUserClass_NoDisplay:        return "non";
            default: return "n/a";
            }
        };

        auto getBufferOwnerString = [](detail::IntegratedDisplayLayerBufferOwner v) -> const char*{
            switch(v)
            {
            case detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet: return "own";
            case detail::IntegratedDisplayLayerBufferOwner_SystemShared:  return "shr";
            default: return "n/a";
            }
        };

        auto getDestinationLayerString = [](detail::IntegratedDisplayLayerDestinationLayer v) -> const char*{
            switch(v)
            {
            case detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer:  return "dsp";
            case detail::IntegratedDisplayLayerDestinationLayer_IndirectLayer: return "ind";
            default: return "n/a";
            }
        };


        auto makeLayerString = [&](const detail::IntegratedDisplayLayer& layer, int index) -> char*{
            static char buf[1024] = {};
            std::memset(buf, 0, sizeof(buf));

            int remain = static_cast<int>(sizeof(buf));
            int pos = 0;
            int delta = 0;

    #define NN_AM_DISPCTRL_DUMP_ADD_STRING(...) \
        delta = nn::util::SNPrintf(buf + pos, remain, __VA_ARGS__); \
        pos += delta;   \
        remain -= delta;    \
        if(remain <= 0) \
        {   \
            return buf; \
        }


            // パラメータ
            NN_AM_DISPCTRL_DUMP_ADD_STRING("  L%02d: %s,%s,%s | ",
                index,
                getUserClassString(layer.GetUserClass()),
                getBufferOwnerString(layer.GetBufferOwner()),
                getDestinationLayerString(layer.GetDestinationLayer())
            );

            // 状態
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s,z=%d,", layer.IsVisible() ? "vis" : "inv", layer.GetZOrder());
            if(layer.IsCaptureTarget())
            {
                NN_AM_DISPCTRL_DUMP_ADD_STRING("cap,");
            }
            if(layer.IsTransitionLayerRequested())
            {
                NN_AM_DISPCTRL_DUMP_ADD_STRING("tra,");
            }
            if(layer.IsConductorIfForeground())
            {
                NN_AM_DISPCTRL_DUMP_ADD_STRING("ctr,");
            }
            if(layer.IsRecordingLayerStackRequested())
            {
                NN_AM_DISPCTRL_DUMP_ADD_STRING("rec,");
            }
            if(layer.IsDebugLayerStackRequested())
            {
                NN_AM_DISPCTRL_DUMP_ADD_STRING("dbg,");
            }
            if(layer.IsCallerAppletCaptureRequested())
            {
                if(layer.IsCallerAppletCaptureScreenshotPermitted())
                {
                    NN_AM_DISPCTRL_DUMP_ADD_STRING("caY,");
                }
                else
                {
                    NN_AM_DISPCTRL_DUMP_ADD_STRING("caN,");
                }
            }
            if(layer.IsSharedBufferExported())
            {
                NN_AM_DISPCTRL_DUMP_ADD_STRING("exp,");
            }
    #undef NN_AM_DISPCTRL_DUMP_ADD_STRING

            return buf;
        };


        for(int i = 0; i < detail::IntegratedDisplayLayerPool::LayerCount; i++)
        {
            auto pLayer = m_AppletDisplayLayerPool.GetLayerAt(i);
            if(pLayer == nullptr || pLayer->GetState() == detail::IntegratedDisplayLayerState_NotInitialized)
            {
                continue;
            }

            auto str = makeLayerString(*pLayer, i);
            NN_AM_DISPCTRL_LOG_DUMP("%s\n", str);
        }
    }// NOLINT(impl/function_size)

    void IntegratedDisplayManager::DumpVisibilityManagerListImpl() NN_NOEXCEPT
    {
        auto makeLayerString = [&](const detail::DirectLayerVisibilityManager& vis, int index) -> const char*{
            static char buf[1024] = {};
            std::memset(buf, 0, sizeof(buf));

            int remain = static_cast<int>(sizeof(buf));
            int pos = 0;
            int delta = 0;

    #define NN_AM_DISPCTRL_DUMP_ADD_STRING(...) \
        delta = nn::util::SNPrintf(buf + pos, remain, __VA_ARGS__); \
        pos += delta;   \
        remain -= delta;    \
        if(remain <= 0) \
        {   \
            return buf; \
        }

            auto pDumper = vis.GetDumpAccessor();
            if(pDumper->GetLayerId() == 0)
            {
                return "";
            }

            // パラメータ
            NN_AM_DISPCTRL_DUMP_ADD_STRING("  V%02d:%s", index, vis.IsActive() ? "+" : "-");

            // Layer
            NN_AM_DISPCTRL_DUMP_ADD_STRING("|id% 2llu z%d", pDumper->GetLayerId(), pDumper->GetZOrder());

            // Stacks
            NN_AM_DISPCTRL_DUMP_ADD_STRING("[");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", pDumper->GetForegroundStacks().IsAnyOn() ? "f" : "-");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", pDumper->GetInstantCaptureStacks().IsAnyOn() ? "i" : "-");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", pDumper->GetMovieRecordingStacks().IsAnyOn() ? "m" : "-");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", pDumper->GetDebugScreenshotStacks().IsAnyOn() ? "d" : "-");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("]");

            // Current
            NN_AM_DISPCTRL_DUMP_ADD_STRING(" | z%d ", pDumper->GetZOrder());
            // Stacks: Physical は大文字。 Virtual は小文字。
            auto stacks = pDumper->GetCurrentStacks();
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", stacks[nn::vi::LayerStack_Default] ? "D" : "");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", stacks[nn::vi::LayerStack_Lcd] ? "L" : "");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", stacks[nn::vi::LayerStack_Screenshot] ? "s" : "");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", stacks[nn::vi::LayerStack_Recording] ? "m" : "");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", stacks[nn::vi::LayerStack_LastFrame] ? "c" : "");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", stacks[nn::vi::LayerStack_Arbitrary] ? "i" : "");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", stacks[nn::vi::LayerStack_Null] ? "N" : "");
            NN_AM_DISPCTRL_DUMP_ADD_STRING("%s", stacks[nn::vi::LayerStack_ApplicationForDebug] ? "d" : "");


    #undef NN_AM_DISPCTRL_DUMP_ADD_STRING

            return buf;
        };


        for(int i = 0; i < detail::IntegratedDisplayLayerPool::LayerCount; i++)
        {
            auto pVisMgr = m_AppletDisplayLayerPool.GetVisibilityManagerAt(i);
            if(pVisMgr == nullptr || !pVisMgr->IsInitialized())
            {
                continue;
            }

            auto str = makeLayerString(*pVisMgr, i);
            if(str == nullptr || str[0] == 0)
            {
                continue;
            }
            NN_AM_DISPCTRL_LOG_DUMP("%s\n", str);
        }
    }// NOLINT(impl/function_size)
    #else // NN_AM_DISPCTRL_IS_DUMP_ENABLED
    void IntegratedDisplayManager::Dump() NN_NOEXCEPT
    {
    }
    #endif

}}}}
