﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_IntegratedDisplayLayerPool.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace nn{ namespace am{ namespace service{ namespace display{ namespace detail{

    const int IntegratedDisplayLayerPool::LayerCount;

    int IntegratedDisplayLayerPool::GetLayerIndexImpl(IntegratedDisplayLayerHandle h) const NN_NOEXCEPT
    {
        int index = static_cast<int>(h);

        if(index < 0 || index >= LayerCount)
        {
            return -1;
        }

        auto pLayer = &m_List[index].m_Layer;
        if(pLayer->m_State == IntegratedDisplayLayerState_Free)
        {
            return -1;
        }

        return index;
    }

    IntegratedDisplayLayerHandle IntegratedDisplayLayerPool::Acquire() NN_NOEXCEPT
    {
        for(int i = 0; i < LayerCount; i++)
        {
            auto pLayer = &m_List[i].m_Layer;
            if(pLayer->m_State == IntegratedDisplayLayerState_Free)
            {
                pLayer->m_State = IntegratedDisplayLayerState_NotInitialized;
                return i;
            }
        }

        return -1;
    }

    void IntegratedDisplayLayerPool::Release(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
    {
        int index = GetLayerIndexImpl(h);
        if(index < 0)
        {
            return;
        }

        auto pLayer = &m_List[index].m_Layer;
        NN_ABORT_UNLESS_EQUAL(pLayer->m_State, IntegratedDisplayLayerState_NotInitialized);
        NN_ABORT_UNLESS(!m_List[index].m_VisibilityManager.IsInitialized());
        pLayer->m_State = IntegratedDisplayLayerState_Free;
    }

    IntegratedDisplayLayer* IntegratedDisplayLayerPool::GetLayer(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
    {
        int index = GetLayerIndexImpl(h);
        if(index < 0)
        {
            return nullptr;
        }

        return &m_List[index].m_Layer;
    }

    const IntegratedDisplayLayer* IntegratedDisplayLayerPool::GetLayer(IntegratedDisplayLayerHandle h) const NN_NOEXCEPT
    {
        int index = GetLayerIndexImpl(h);
        if(index < 0)
        {
            return nullptr;
        }

        return &m_List[index].m_Layer;
    }

    const IntegratedDisplayLayer* IntegratedDisplayLayerPool::GetLayerAt(int index) const NN_NOEXCEPT
    {
        if(index < 0 || index >= LayerCount)
        {
            return nullptr;
        }

        auto pLayer = &m_List[index].m_Layer;
        if(pLayer->m_State == IntegratedDisplayLayerState_Free)
        {
            return nullptr;
        }

        return pLayer;
    }

    DirectLayerVisibilityManager* IntegratedDisplayLayerPool::GetVisibilityManager(IntegratedDisplayLayerHandle h) NN_NOEXCEPT
    {
        int index = GetLayerIndexImpl(h);
        if(index < 0)
        {
            return nullptr;
        }

        return &m_List[index].m_VisibilityManager;
    }

    DirectLayerVisibilityManager* IntegratedDisplayLayerPool::GetVisibilityManagerAt(int index) NN_NOEXCEPT
    {
        if(index < 0 || index >= LayerCount)
        {
            return nullptr;
        }

        auto pLayer = &m_List[index].m_Layer;
        if(pLayer->m_State == IntegratedDisplayLayerState_Free)
        {
            return nullptr;
        }

        return &m_List[index].m_VisibilityManager;
    }

}}}}}
