﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_DisplayWorker.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_SystemThreadDefinition.h>

namespace nn{ namespace am{ namespace service{ namespace display{

    const uintptr_t DisplayWorker::MultiWaitSlotCount;
    const uintptr_t DisplayWorker::MultiWaitSlotReserved;

    namespace {
        enum ReservedTag : uintptr_t
        {
            ReservedTag_Invalid = 0,
            ReservedTag_Exit,
            ReservedTag_Max,
        };

        NN_STATIC_ASSERT(ReservedTag_Max <= DisplayWorker::MultiWaitSlotReserved);
    }

    void DisplayWorker::Initialize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadFunctionImpl, this, m_ThreadStack, sizeof(m_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(am, DisplayManager)));
        nn::os::SetThreadName(&m_Thread, NN_SYSTEM_THREAD_NAME(am, DisplayManager));

        nn::os::InitializeMultiWait(&m_MultiWait);

        // exit request
        {
            nn::os::InitializeEvent(&m_ExitRequestEvent, false, nn::os::EventClearMode_ManualClear);
            nn::os::InitializeMultiWaitHolder(&m_ExitRequestHolder, &m_ExitRequestEvent);
            m_ExitRequestHolder.userData = ReservedTag_Exit;
            nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_ExitRequestHolder);
        }

        for(auto& e : m_SlotList)
        {
            e = {};
        }
    }

    void DisplayWorker::Finalize() NN_NOEXCEPT
    {
        nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);

        {
            nn::os::FinalizeMultiWaitHolder(&m_ExitRequestHolder);
            nn::os::FinalizeEvent(&m_ExitRequestEvent);
        }

        nn::os::FinalizeMultiWait(&m_MultiWait);
        nn::os::DestroyThread(&m_Thread);
    }

    void DisplayWorker::Start() NN_NOEXCEPT
    {
        nn::os::ClearEvent(&m_ExitRequestEvent);
        nn::os::StartThread(&m_Thread);
    }

    void DisplayWorker::Stop() NN_NOEXCEPT
    {
        nn::os::SignalEvent(&m_ExitRequestEvent);
        nn::os::WaitThread(&m_Thread);
    }

    nn::os::MultiWaitHolderType* DisplayWorker::LinkMultiWait(nn::os::MultiWaitHolderType* pHolder, void (*function)(nn::os::MultiWaitHolderType*, void*), void* userPtr) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);

        MultiWaitSlot* pSlot = nullptr;
        uintptr_t index = 0;
        for(index = 0; index < MultiWaitSlotCount; index++)
        {
            if(m_SlotList[index].pHolder == nullptr)
            {
                pSlot = &m_SlotList[index];
                break;
            }
        }

        if(!pSlot)
        {
            return nullptr;
        }

        auto tag = MultiWaitSlotReserved + index;

        nn::os::LinkMultiWaitHolder(&m_MultiWait, pHolder);
        pHolder->userData = tag;

        pSlot->pHolder  = pHolder;
        pSlot->function = function;
        pSlot->userPtr  = userPtr;
        return pHolder;
    }

    void DisplayWorker::UnlinkMultiWaitImpl(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        if(pHolder == nullptr || pHolder->userData == 0)
        {
            return;
        }

        auto tag = pHolder->userData;
        NN_SDK_REQUIRES_GREATER_EQUAL(tag, MultiWaitSlotReserved);

        auto slot = tag - MultiWaitSlotReserved;
        NN_SDK_REQUIRES_RANGE(slot, 0u, MultiWaitSlotCount);

        auto& e = m_SlotList[slot];
        NN_SDK_ASSERT_EQUAL(e.pHolder, pHolder);

        nn::os::UnlinkMultiWaitHolder(e.pHolder);
        e.pHolder->userData = {};
        e = {};
    }

    void DisplayWorker::UnlinkMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        UnlinkMultiWaitImpl(pHolder);
    }

    void DisplayWorker::ThreadFunctionImpl(void* arg) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<DisplayWorker*>(arg);
        p->ThreadFunction();
    }

    void DisplayWorker::ThreadFunction() NN_NOEXCEPT
    {
        for(;;)
        {
            auto pHolder = nn::os::WaitAny(&m_MultiWait);

            auto tag = pHolder->userData;

            if(tag == ReservedTag_Exit)
            {
                break;
            }
            else if(tag >= MultiWaitSlotReserved && tag < MultiWaitSlotReserved + MultiWaitSlotCount)
            {
                auto index = tag - MultiWaitSlotReserved;

                // Unlink する前にコピー
                auto e = m_SlotList[index];
                NN_SDK_ASSERT_EQUAL(e.pHolder, pHolder);

                // 一旦 Unlink
                UnlinkMultiWaitImpl(pHolder);

                if(e.function)
                {
                    e.function(pHolder, e.userPtr);
                }
            }
            else
            {
                NN_SDK_ASSERT(false, "DisplayWorker: unknown tag(%llu)", tag);
                nn::os::UnlinkMultiWaitHolder(pHolder);
            }
        }
    }

}}}}
