﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_DisplayPrimitiveOperation.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/am/service/display/am_DisplayControlLog.h>
#include <nn/am/service/display/am_MyDisplayManagerService.h>

#define NN_AM_DEFAULT_DISPLAY_NAME "Default"

namespace nn{ namespace am{ namespace service{ namespace display{

    namespace {
        nn::util::optional<MyDisplayManagerService> g_pService;
    }

    void DisplayPrimitiveOperation::Initialize() NN_NOEXCEPT
    {
        g_pService.emplace();
    }

    void DisplayPrimitiveOperation::Finalize() NN_NOEXCEPT
    {
        g_pService = nn::util::nullopt;
    }

    //------------------------------------------------------------

    nn::Result DisplayPrimitiveOperation::OpenDisplay(nn::vi::DisplayId* pOutValue, const char* name) NN_NOEXCEPT
    {
        return g_pService->OpenDisplayWrapper(pOutValue, name);
    }

    void DisplayPrimitiveOperation::CloseDisplay(nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        g_pService->CloseDisplayWrapper(displayId);
    }

    nn::Result DisplayPrimitiveOperation::GetDisplayHotplugState(nn::vi::HotplugStateType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        return g_pService->GetManagerDisplayService()->GetDisplayHotplugState(pOutValue, displayId);
    }

    nn::Result DisplayPrimitiveOperation::GetDisplayHotplugEvent(nn::sf::NativeHandle& outHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        return g_pService->GetManagerDisplayService()->GetDisplayHotplugEvent(&outHandle, displayId);
    }

    nn::Result DisplayPrimitiveOperation::SetDisplayCmuLuma(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT
    {
        return g_pService->GetSystemDisplayService()->SetDisplayCmuLuma(displayId, value);
    }

    nn::Result DisplayPrimitiveOperation::SetDisplayAlpha(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT
    {
        return g_pService->GetManagerDisplayService()->SetDisplayAlpha(displayId, value);
    }

    nn::Result DisplayPrimitiveOperation::SetDisplayPowerState(nn::vi::DisplayId displayId, nn::vi::PowerStateType value) NN_NOEXCEPT
    {
        return g_pService->GetManagerDisplayService()->SetDisplayPowerState(displayId, value);
    }

    nn::Result DisplayPrimitiveOperation::GetDisplayResolution(int64_t* pOutWidth, int64_t* pOutHeight, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        return g_pService->GetManagerDisplayService()->GetDisplayResolution(pOutWidth, pOutHeight, displayId);
    }

    nn::Result DisplayPrimitiveOperation::SetDefaultDisplay(nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        return g_pService->GetManagerDisplayService()->SetDefaultDisplay(displayId);
    }

    //------------------------------------------------------------

    nn::Result DisplayPrimitiveOperation::CreateDisplayLayer(nn::vi::LayerId* pOutValue, nn::vi::LayerSettings settings, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer_CreateDisplayLayer, 60);
        nn::vi::LayerId ret;
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->CreateManagedLayer(&ret, g_pService->GetDisplayId(), settings._storage[0], userAruid));
        // 作成したてのレイヤは Default に入っているため除外する
        nn::vi::LayerConfig config;
        config.stacks.Set<nn::vi::LayerStackFlags::Null>();

        config.mask.Set<nn::vi::LayerStackFlags::Default>();
        config.mask.Set<nn::vi::LayerStackFlags::Null>();

        NN_RESULT_DO(g_pService->GetManagerDisplayService()->SetLayerConfig(ret, config));

        *pOutValue = ret;
        NN_RESULT_SUCCESS;
    }

    void DisplayPrimitiveOperation::DestroyDisplayLayer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer_DestroyDisplayLayer, 60);
        NN_SDK_REQUIRES(g_pService);
        g_pService->GetManagerDisplayService()->DestroyManagedLayer(layerId);
    }

    nn::Result DisplayPrimitiveOperation::AttachDisplayLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->AttachLayerPresentationTracer(layerId));
        NN_RESULT_SUCCESS;
    }

    void DisplayPrimitiveOperation::DetachDisplayLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_pService->GetManagerDisplayService()->DetachLayerPresentationTracer(layerId));
    }

    nn::Result DisplayPrimitiveOperation::StartDisplayLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->StartLayerPresentationRecording(layerId));
        NN_RESULT_SUCCESS;
    }

    nn::Result DisplayPrimitiveOperation::StopDisplayLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->StopLayerPresentationRecording(layerId));
        NN_RESULT_SUCCESS;
    }

    nn::Result DisplayPrimitiveOperation::StartDisplayLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->StartLayerPresentationFenceWait(layerId));
        NN_RESULT_SUCCESS;
    }

    nn::Result DisplayPrimitiveOperation::StopDisplayLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->StopLayerPresentationFenceWait(layerId));
        NN_RESULT_SUCCESS;
    }

    nn::Result DisplayPrimitiveOperation::InitializeDisplayLayerPresentationAllFencesExpiredEvent(nn::os::SystemEventType* pEvent, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        nn::sf::NativeHandle h;
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->GetLayerPresentationAllFencesExpiredEvent(&h, layerId));

        nn::os::AttachReadableHandleToSystemEvent(pEvent, h.GetOsHandle(), h.IsManaged(), nn::os::EventClearMode_ManualClear);
        h.Detach();

        NN_RESULT_SUCCESS;
    }

    void DisplayPrimitiveOperation::FinalizeDisplayLayerPresentationAllFencesExpiredEvent(nn::os::SystemEventType* pEvent, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_UNUSED(layerId);
        nn::os::DestroySystemEvent(pEvent);
    }

    //------------------------------------------------------------

    void DisplayPrimitiveOperation::SetDisplayLayerConfig(nn::vi::LayerId layerId, const nn::vi::LayerConfig& config) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer_SetDisplayLayerVisibility, 60);
        NN_SDK_REQUIRES(g_pService);
        g_pService->GetManagerDisplayService()->SetLayerConfig(layerId, config);
    }

    void DisplayPrimitiveOperation::SetDisplayLayerZOrder(nn::vi::LayerId layerId, int32_t zOrder) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer_SetDisplayLayerVisibility, 60);
        NN_SDK_REQUIRES(g_pService);

        g_pService->GetSystemDisplayService()->SetLayerZ(layerId, zOrder);
    }

    //------------------------------------------------------------

    void DisplayPrimitiveOperation::SetIsConductorLayer(nn::vi::LayerId layerId, bool isConductor) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        g_pService->GetManagerDisplayService()->SetConductorLayer(layerId, isConductor);
    }

    //------------------------------------------------------------

    void DisplayPrimitiveOperation::InitializeLayerTexturePresentingEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        nn::sf::NativeHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->AcquireLayerTexturePresentingEvent(&handle, layerId));

        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_ManualClear);
        handle.Detach();
    }

    void DisplayPrimitiveOperation::FinalizeLayerTexturePresentingEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->ReleaseLayerTexturePresentingEvent(layerId));

        nn::os::DestroySystemEvent(pSystemEvent);
    }

    //------------------------------------------------------------

    nn::Result DisplayPrimitiveOperation::CreateIndirectLayer(nn::vi::IndirectLayerHandleType* pOutHandle) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->CreateIndirectLayer(pOutHandle));
        NN_RESULT_SUCCESS;
    }

    void DisplayPrimitiveOperation::DestroyIndirectLayer(nn::vi::IndirectLayerHandleType hLayer) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_pService->GetManagerDisplayService()->DestroyIndirectLayer(hLayer));
    }

    nn::Result DisplayPrimitiveOperation::CreateIndirectConsumerEndPoint(nn::vi::IndirectConsumerHandleType* pOutValue, nn::vi::IndirectLayerHandleType hLayer, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        NN_AM_SERVICE_LOG(seq, "Try CreateIndirectConsumerEndPoint(L_%lld, A_%llx)\n", hLayer, userAruid.lower);
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->CreateIndirectConsumerEndPoint(pOutValue, hLayer, userAruid));
        NN_AM_SERVICE_LOG(seq, "CreateIndirectConsumerEndPoint(L_%lld, A_%llx) -> P_%lld\n", hLayer, userAruid.lower, *pOutValue);
        NN_RESULT_SUCCESS;
    }

    void DisplayPrimitiveOperation::DestroyIndirectConsumerEndPoint(nn::vi::IndirectLayerHandleType hLayer, nn::vi::IndirectConsumerHandleType hConsumer) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        NN_UNUSED(hConsumer);
        NN_AM_SERVICE_LOG(seq, "DestroyIndirectConsumerEndPoint(P_%lld)\n", hLayer);
        (void)g_pService->GetManagerDisplayService()->DestroyIndirectConsumerEndPoint(hLayer);
    }

    nn::Result DisplayPrimitiveOperation::CreateIndirectProducerEndPoint(nn::vi::IndirectProducerHandleType* pOutValue, nn::vi::IndirectLayerHandleType hLayer, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        NN_AM_SERVICE_LOG(seq, "Try CreateIndirectProducerEndPoint(L_%lld, A_%llx)\n", hLayer, userAruid.lower);
        NN_RESULT_DO(g_pService->GetManagerDisplayService()->CreateIndirectProducerEndPoint(pOutValue, hLayer, userAruid));
        NN_AM_SERVICE_LOG(seq, "CreateIndirectProducerEndPoint(L_%lld, A_%llx) -> P_%lld\n", hLayer, userAruid.lower, *pOutValue);
        NN_RESULT_SUCCESS;
    }

    void DisplayPrimitiveOperation::DestroyIndirectProducerEndPoint(nn::vi::IndirectLayerHandleType hLayer, nn::vi::IndirectProducerHandleType hProducer) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        NN_UNUSED(hProducer);
        NN_AM_SERVICE_LOG(seq, "DestroyIndirectProducerEndPoint(P_%lld)\n", hLayer);
        (void)g_pService->GetManagerDisplayService()->DestroyIndirectProducerEndPoint(hLayer);
    }

    void DisplayPrimitiveOperation::SetIndirectProducerFlipOffset(nn::vi::IndirectLayerHandleType layerHandle, nn::vi::IndirectProducerHandleType hProducer, nn::TimeSpan offset) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        NN_AM_SERVICE_LOG(seq, "Try SetIndirectLayerProducerFlipOffset(L_%lld, P_%llx, O_%lldus)\n", layerHandle, hProducer, offset.GetMicroSeconds());
        g_pService->GetManagerDisplayService()->SetIndirectProducerFlipOffset(layerHandle, hProducer, offset);
        NN_AM_SERVICE_LOG(seq, "CreateIndirectConsumerEndPoint(L_%lld, P_%llx, O_%lldus)\n", layerHandle, hProducer, offset.GetMicroSeconds());
    }

    //------------------------------------------------------------

    nn::Result DisplayPrimitiveOperation::CreateSharedBufferOnStaticStorage(nn::vi::fbshare::SharedBufferHandle* pOutHandle, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        uint64_t key = 0; // 用途を指定するために指定、現状システム共有バッファ以外の使い道がないので 0 。
        return pMgrSrv->CreateSharedBufferStaticStorage(pOutHandle, key, layout);
    }

    nn::Result DisplayPrimitiveOperation::CreateSharedBufferOnProcessHeap(nn::vi::fbshare::SharedBufferHandle* pOutHandle, uint64_t hHeapMemory, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->CreateSharedBufferProcessHeap(pOutHandle, hHeapMemory, layout);
    }

    nn::Result DisplayPrimitiveOperation::CreateSharedBufferOnTransferMemory(nn::vi::fbshare::SharedBufferHandle* pOutHandle, nn::sf::NativeHandle&& hTransferMemory, size_t size, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->CreateSharedBufferTransferMemory(pOutHandle, std::move(hTransferMemory), size, layout);
    }

    void DisplayPrimitiveOperation::DestroySharedBuffer(nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->DestroySharedBuffer(hBuffer));
    }

    void DisplayPrimitiveOperation::RegisterSharedBufferImporter(nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->RegisterSharedBufferImporterAruid(handle, aruid));
    }

    void DisplayPrimitiveOperation::UnregisterSharedBufferImporter(nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->UnregisterSharedBufferImporterAruid(handle, aruid));
    }

    //---------------------

    nn::Result DisplayPrimitiveOperation::ExpandStartupLogoOnSharedFrameBuffer(nn::vi::fbshare::SharedBufferHandle hBuffer, int dstIndex, const nn::vi::fbshare::SharedTextureOption& option) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->ExpandStartupLogoOnSharedFrameBuffer(hBuffer, dstIndex, option);
    }

    nn::Result DisplayPrimitiveOperation::CopyDetachedSharedFrameBufferImage(
        nn::vi::fbshare::SharedBufferHandle hDstBuffer,
        int dstIndex,
        nn::vi::fbshare::SharedBufferHandle hSrcBuffer,
        int srcIndex,
        nn::vi::ImageTransform dstTransform,
        nn::vi::fbshare::SharedTextureAlphaOption alphaOption,
        nn::vi::LayerStackFlagType stacksFilter,
        nn::vi::LayerStackFlagType nullStacks
    ) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);

        // TORIAEZU: 使うオプションだけ設定
        nn::vi::fbshare::SharedTextureOption dstOption = {};
        dstOption.transform = dstTransform;
        dstOption.alphaOption = alphaOption;

        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->CopyDetachedSharedFrameBufferImage(hDstBuffer, dstIndex, hSrcBuffer, srcIndex, dstOption, stacksFilter, nullStacks);
    }

    nn::Result DisplayPrimitiveOperation::SetDetachedSharedFrameBufferImage(
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        int dstIndex,
        const void* srcBuffer,
        size_t srcBufferSize,
        const nn::vi::fbshare::SharedTextureOption& dstOption,
        nn::vi::ImageTransform srcTransform
    ) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->SetDetachedSharedFrameBufferImage(hBuffer, dstIndex, nn::sf::InBuffer(reinterpret_cast<const char*>(srcBuffer), srcBufferSize), dstOption, srcTransform);
    }

    nn::Result DisplayPrimitiveOperation::SetDetachedSharedFrameBufferSubImage(
        nn::vi::fbshare::SharedBufferHandle hBuffer,
        int dstIndex,
        int32_t x,
        int32_t y,
        int32_t w,
        int32_t h,
        uint32_t bgColor,
        const void* srcBuffer,
        size_t srcBufferSize,
        const nn::vi::fbshare::SharedTextureOption& dstOption,
        nn::vi::ImageTransform srcTransform
    ) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->SetDetachedSharedFrameBufferSubImage(hBuffer, dstIndex, x, y, w, h, bgColor, nn::sf::InBuffer(reinterpret_cast<const char*>(srcBuffer), srcBufferSize), dstOption, srcTransform);
    }

    nn::Result DisplayPrimitiveOperation::FillDetachedSharedFrameBufferColor(nn::vi::fbshare::SharedBufferHandle hBuffer, int dstIndex, uint32_t color, const nn::vi::fbshare::SharedTextureOption& option) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->FillDetachedSharedFrameBufferColor(hBuffer, dstIndex, color, option);
    }

    nn::Result DisplayPrimitiveOperation::GetDetachedSharedFrameBufferImage(uint64_t* pOutSize, void* outBuffer, size_t outBufferSize, nn::vi::fbshare::SharedBufferHandle hBuffer, int srcIndex) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->GetDetachedSharedFrameBufferImage(pOutSize, nn::sf::OutBuffer(reinterpret_cast<char*>(outBuffer), outBufferSize), hBuffer, srcIndex);
    }

    nn::Result DisplayPrimitiveOperation::SetDetachedSharedFrameBufferLayerStack(nn::vi::fbshare::SharedBufferHandle hBuffer, int index, nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT
    {
        // TORIAEZU: 同じテクスチャにコピーしようとすると LayerStack だけ書き換わる仕様を使う
        return DisplayPrimitiveOperation::CopyDetachedSharedFrameBufferImage(hBuffer, index, hBuffer, index, nn::vi::ImageTransform_None, nn::vi::fbshare::SharedTextureAlphaOption_None,  0, stacks);
    }

    //---------------------

    nn::Result DisplayPrimitiveOperation::PresentDetachedSharedFrameBufferToLowLevelLayer(nn::vi::fbshare::SharedBufferHandle hBuffer, nn::vi::LayerId lowLayerId, int index) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->PresentDetachedSharedFrameBufferToLowLevelLayer(hBuffer, lowLayerId, index);
    }

    //---------------------

    nn::Result DisplayPrimitiveOperation::CreateSharedLayer(nn::vi::fbshare::SharedLayerHandle* pOutHandle, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        NN_RESULT_DO(pMgrSrv->CreateSharedLayer(pOutHandle, userAruid));
        NN_RESULT_SUCCESS;
    }

    void DisplayPrimitiveOperation::DestroySharedLayer(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->DestroySharedLayer(h));
    }

    void DisplayPrimitiveOperation::InitializeSharedLayerDetachReadyEvent(nn::os::SystemEventType* pOutSystemEvent, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        nn::sf::NativeHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->GetSharedLayerDetachReadyEvent(&handle, h));
        nn::os::AttachReadableHandleToSystemEvent(pOutSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_ManualClear);
        handle.Detach();
    }

    void DisplayPrimitiveOperation::FinalizeSharedLayerDetachReadyEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT
    {
        NN_UNUSED(h);
        nn::os::DestroySystemEvent(pSystemEvent);
    }

    void DisplayPrimitiveOperation::SetSharedLayerVisibility(nn::vi::fbshare::SharedLayerHandle hLayer, bool visible, int32_t /*zOrder*/, bool isCaptureTarget) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        nn::vi::LayerStackFlagType stacks = {};
        if (visible)
        {
            stacks |= (1 << nn::vi::LayerStack_Default);
            if (isCaptureTarget)
            {
                stacks |= (1 << nn::vi::LayerStack_LastFrame);
                stacks |= (1 << nn::vi::LayerStack_Screenshot);
            }
        }
        else
        {
            stacks = (1 << nn::vi::LayerStack_Null);
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->SetSharedLayerLayerStacks(hLayer, stacks));
    }

    nn::Result DisplayPrimitiveOperation::AttachSharedLayer(nn::vi::fbshare::SharedLayerHandle handle, nn::vi::LayerId layerId, const nn::vi::fbshare::SharedLayerTextureIndexList& list) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        NN_RESULT_DO(pMgrSrv->AttachSharedLayerToLowLevelLayer(handle, layerId, list));
        NN_RESULT_SUCCESS;
    }

    void DisplayPrimitiveOperation::DetachSharedLayer(nn::vi::fbshare::SharedLayerHandle handle, nn::os::SystemEventType* pDetachReadyEvent, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->StartDetachSharedLayerFromLowLevelLayer(handle));
        if(nn::os::TimedWaitSystemEvent(pDetachReadyEvent, timeout))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->FinishDetachSharedLayerFromLowLevelLayer(handle));
        }
        else
        {
            NN_AM_DISPCTRL_LOG_WARN("force detach\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->ForceDetachSharedLayerFromLowLevelLayer(handle));
        }
    }

    void DisplayPrimitiveOperation::BlankSharedLowLevelLayer(nn::vi::LayerId layerId, nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->DisconnectSharedLowLevelLayerFromSharedBuffer(layerId));
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->ConnectSharedLowLevelLayerToSharedBuffer(layerId, hBuffer));
    }

    void DisplayPrimitiveOperation::SynchronizeSharedLowLevelLayer(int* pOutIndex, nn::vi::LayerId layerId, nn::os::SystemEventType* pSyncEvent) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        int64_t index = -1;
        nn::os::WaitSystemEvent(pSyncEvent);
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->CheckSharedLowLevelLayerSynchronized(&index, layerId));

        *pOutIndex = static_cast<int>(index);
    }

    nn::Result DisplayPrimitiveOperation::BindSharedLowLevelLayerToDisplayLayer(nn::vi::LayerId layerId, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        return pMgrSrv->BindSharedLowLevelLayerToManagedLayer(layerId, {NN_AM_DEFAULT_DISPLAY_NAME}, aruid);
    }

    void DisplayPrimitiveOperation::UnbindSharedLowLevelLayerFromDisplayLayer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        (void)pMgrSrv->UnbindSharedLowLevelLayer(layerId);
    }

    nn::Result DisplayPrimitiveOperation::ConnectSharedLowLevelLayerToDisplayLayer(nn::vi::LayerId layerId, nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        return pMgrSrv->ConnectSharedLowLevelLayerToSharedBuffer(layerId, hBuffer);
    }

    void DisplayPrimitiveOperation::DisconnectSharedLowLevelLayerFromDisplayLayer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        (void)pMgrSrv->DisconnectSharedLowLevelLayerFromSharedBuffer(layerId);
    }

    nn::Result DisplayPrimitiveOperation::BindSharedLowLevelLayerToIndirectLayer(nn::vi::IndirectProducerHandleType hProducer, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        return pMgrSrv->BindSharedLowLevelLayerToIndirectLayer(hProducer, aruid);
    }

    void DisplayPrimitiveOperation::UnbindSharedLowLevelLayerFromIndirectLayer(nn::vi::IndirectProducerHandleType hProducer) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        (void)pMgrSrv->UnbindSharedLowLevelLayer(static_cast<nn::vi::LayerId>(hProducer));
    }

    nn::Result DisplayPrimitiveOperation::ConnectSharedLowLevelLayerToIndirectLayer(nn::vi::IndirectProducerHandleType hProducer, nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        return pMgrSrv->ConnectSharedLowLevelLayerToSharedBuffer(static_cast<nn::vi::LayerId>(hProducer), hBuffer);
    }

    void DisplayPrimitiveOperation::DisconnectSharedLowLevelLayerFromIndirectLayer(nn::vi::IndirectProducerHandleType hProducer) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        (void)pMgrSrv->DisconnectSharedLowLevelLayerFromSharedBuffer(static_cast<nn::vi::LayerId>(hProducer));
    }

    void DisplayPrimitiveOperation::InitializeSharedLowLevelLayerSynchronizedEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(vi_layer, 60);
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();

        nn::sf::NativeHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->GetSharedLowLevelLayerSynchronizedEvent(&handle, layerId));
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_ManualClear);
        handle.Detach();
    }

    void DisplayPrimitiveOperation::FinalizeSharedLowLevelLayerSynchronizedEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_UNUSED(layerId);
        nn::os::DestroySystemEvent(pSystemEvent);
    }

    //------------------------------------------------------------

    nn::Result DisplayPrimitiveOperation::AllocateDisplayProcessHeapBlock(uint64_t* pOutHandle, uint64_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        return pMgrSrv->AllocateProcessHeapBlock(pOutHandle, size);
    }

    void DisplayPrimitiveOperation::FreeDisplayProcessHeapBlock(uint64_t handle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_pService);
        auto pMgrSrv = g_pService->GetManagerDisplayService();
        NN_ABORT_UNLESS_RESULT_SUCCESS(pMgrSrv->FreeProcessHeapBlock(handle));
    }
}}}}

