﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_DisplayCaptureBufferOperation.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/applet/applet_Result.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/capsrv/capsrv_Result.h>

#include <nn/am/service/display/am_DisplayControlLog.h>
#include <nn/am/service/display/am_CaptureBufferManager.h>
#include <nn/am/service/display/am_DisplayPrimitiveOperation.h>

namespace nn{ namespace am{ namespace service{ namespace display{

    namespace {

        IntegratedDisplayParameter g_Param = {};
        nn::vi::fbshare::SharedBufferHandle g_hSharedBuffer = {};
        nn::util::optional<CaptureBufferManager> g_pCaptureBufferManager;

        const char* GetCaptureBufferNameImpl(CaptureBufferIndex cIdx) NN_NOEXCEPT
        {
            switch(cIdx)
            {
            case CaptureBufferIndex_LastForeground:
                return "LastFg";
            case CaptureBufferIndex_LastApplication:
                return "LastApp";
            case CaptureBufferIndex_CallerApplet:
                return "CalrApt";
            default:
                return "invalid";
            }
        }

        int GetSharedTextureIndexImpl(CaptureBufferIndex cIdx) NN_NOEXCEPT
        {
            switch(cIdx)
            {
            case CaptureBufferIndex_LastForeground:
                return g_Param.lastForegroundBufferIndex;
            case CaptureBufferIndex_LastApplication:
                return g_Param.lastApplicationBufferIndex;
            case CaptureBufferIndex_CallerApplet:
                return g_Param.callerAppletBufferIndex;
            default:
                return -1;
            }
        }

        bool EnsureSharedBufferRegisterd() NN_NOEXCEPT
        {
            static bool s_IsRegisterd = false;
            if(s_IsRegisterd)
            {
                return true;
            }

            auto result = nn::capsrv::AttachSharedBufferToCaptureModule(g_hSharedBuffer);

            if(result.IsFailure())
            {
                NN_AM_DISPCTRL_LOG_WARN("failed to attach shared buffer to capture module(%d-%d)\n", result.GetModule(), result.GetDescription());
                return false;
            }

            s_IsRegisterd = true;
            return true;
        }

        // Texture <- (color)
        void FillTextureImpl(CaptureBufferIndex target, uint32_t color) NN_NOEXCEPT
        {
            NN_AM_DISPCTRL_LOG_CAPOP("  fill texture [0x%08x]->'%s'\n", color, GetCaptureBufferNameImpl(target));
            auto textureIndex = GetSharedTextureIndexImpl(target);
            if(textureIndex < 0)
            {
                NN_AM_DISPCTRL_LOG_WARN("fill-target buffer '%s' does not exist on shared buffer\n", GetCaptureBufferNameImpl(target));
                return;
            }

            nn::vi::fbshare::SharedTextureOption option = {};
            option.colorOption = nn::vi::fbshare::SharedTextureColorOption_None;
            option.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_Opaque;
            option.stacks      = TransitionLayerStackFlags;
            option.transform   = nn::vi::ImageTransform_None;
            (void)DisplayPrimitiveOperation::FillDetachedSharedFrameBufferColor(g_hSharedBuffer, textureIndex, color, option);
        }

        // Texture <- (display)
        void CaptureDisplayImageToTextureImpl(CaptureBufferIndex dst, nn::vi::LayerStack layerStack, nn::TimeSpan timeout) NN_NOEXCEPT
        {
            NN_AM_DISPCTRL_LOG_CAPOP("  capture display [stack%d]->'%s'\n", layerStack, GetCaptureBufferNameImpl(dst));
            auto dstTextureIndex = GetSharedTextureIndexImpl(dst);
            if(dstTextureIndex < 0)
            {
                NN_AM_DISPCTRL_LOG_WARN("capture-destination buffer '%s' does not exist on shared buffer\n", GetCaptureBufferNameImpl(dst));
                return;
            }

            if(!EnsureSharedBufferRegisterd())
            {
                return;
            }

            nn::Result captureResult;
            const int repeatCount = (layerStack == vi::LayerStack_Arbitrary) ? 50 : 1;
            for(int trial = 0; trial < repeatCount; trial++)
            {
                captureResult = nn::capsrv::CaptureRawImageToAttachedSharedBuffer(dstTextureIndex, layerStack, timeout);
                if(nn::capsrv::ResultScreenShotNoTarget::Includes(captureResult))
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(16667 / 2));
                    continue;
                }
                else
                {
                    break;
                }
            }

            // レイヤスタックを設定
            DisplayPrimitiveOperation::SetDetachedSharedFrameBufferLayerStack(g_hSharedBuffer, dstTextureIndex, TransitionLayerStackFlags);

            if(captureResult.IsFailure())
            {
                NN_AM_DISPCTRL_LOG_WARN("failed to capture display to shared buffer '%s' (%d-%d)\n", GetCaptureBufferNameImpl(dst), captureResult.GetModule(), captureResult.GetDescription());
            }
        }

        // Texture <- (external)
        void CaptureTextureImageToTextureImpl(CaptureBufferIndex dst, int srcTextureIndex, nn::TimeSpan timeout) NN_NOEXCEPT
        {
            NN_AM_DISPCTRL_LOG_CAPOP("  capture shared texture [iBuf=%d]->'%s'\n", srcTextureIndex, GetCaptureBufferNameImpl(dst));
            NN_UNUSED(timeout);
            auto dstTextureIndex = GetSharedTextureIndexImpl(dst);
            if(dstTextureIndex < 0)
            {
                NN_AM_DISPCTRL_LOG_WARN("destination buffer '%s' does not exist on shared buffer\n", GetCaptureBufferNameImpl(dst));
                return;
            }

            auto copyResult = DisplayPrimitiveOperation::CopyDetachedSharedFrameBufferImage(
                g_hSharedBuffer,
                dstTextureIndex,
                g_hSharedBuffer,
                srcTextureIndex,
                nn::vi::ImageTransform_None,
                nn::vi::fbshare::SharedTextureAlphaOption_Opaque,
                0, // 強制的に TransitionLayerStack に変更
                TransitionLayerStackFlags
            );

            if(copyResult.IsFailure())
            {
                NN_AM_DISPCTRL_LOG_WARN("failed to copy texture [iBuf=%d]->'%s' (%d-%d)\n", srcTextureIndex, GetCaptureBufferNameImpl(dst), copyResult.GetModule(), copyResult.GetDescription());
            }
        }

        // Texture <- Texture
        void CopyBetweenTextureImpl(CaptureBufferIndex dst, CaptureBufferIndex src, nn::TimeSpan timeout) NN_NOEXCEPT
        {
            NN_AM_DISPCTRL_LOG_CAPOP("  copy texture '%s'->'%s'\n", GetCaptureBufferNameImpl(src), GetCaptureBufferNameImpl(dst));
            NN_UNUSED(timeout);
            auto dstTextureIndex = GetSharedTextureIndexImpl(dst);
            auto srcTextureIndex = GetSharedTextureIndexImpl(src);

            if(dstTextureIndex < 0)
            {
                NN_AM_DISPCTRL_LOG_WARN("destination buffer '%s' does not exist on shared buffer\n", GetCaptureBufferNameImpl(dst));
                return;
            }

            if(srcTextureIndex < 0)
            {
                NN_AM_DISPCTRL_LOG_WARN("source buffer '%s' does not exist on shared buffer\n", GetCaptureBufferNameImpl(src));
                return;
            }

            (void)DisplayPrimitiveOperation::CopyDetachedSharedFrameBufferImage(
                g_hSharedBuffer,
                dstTextureIndex,
                g_hSharedBuffer,
                srcTextureIndex,
                nn::vi::ImageTransform_None,
                nn::vi::fbshare::SharedTextureAlphaOption_Opaque,
                0, // 強制的に TransitionLayerStack に変更
                TransitionLayerStackFlags
            );
        }

        void SetImageAttributeImpl(CaptureBufferIndex index, bool isCapturePermitted, bool isCopyrightRequired) NN_NOEXCEPT
        {
            NN_AM_DISPCTRL_LOG_CAPOP("  set attr '%s' (ss%s,cr%s)\n",
                GetCaptureBufferNameImpl(index),
                isCapturePermitted ? "Y" : "N",
                isCopyrightRequired ? "Y" : "N"
            );
            g_pCaptureBufferManager->SetCaptureBufferState(index, isCapturePermitted, isCopyrightRequired);
        }
    }

    void DisplayCaptureBufferOperation::Initialize(const IntegratedDisplayParameter& param, nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT
    {
        // ログ用関数の未使用警告抑制
        {
            auto v = GetCaptureBufferNameImpl;
            NN_UNUSED(v);
        }

        g_pCaptureBufferManager.emplace();

        g_Param = param;
        g_hSharedBuffer = hBuffer;
    }

    void DisplayCaptureBufferOperation::Finalize() NN_NOEXCEPT
    {
        g_pCaptureBufferManager = nn::util::nullopt;

        g_Param = {};
        g_hSharedBuffer = {};
    }

    //-----------------------------------

    void DisplayCaptureBufferOperation::CaptureDisplay(CaptureBufferIndex dst, nn::vi::LayerStack layerStack, bool isCapturePermitted, bool isCopyrightRequired, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_CAPOP("capture display [stack%d]'->%s'\n", layerStack, GetCaptureBufferNameImpl(dst));

        //
        //   (display)
        //      ↓
        //    Texture
        //
        CaptureDisplayImageToTextureImpl(dst, layerStack, timeout);

        SetImageAttributeImpl(dst, isCapturePermitted, isCopyrightRequired);
    }

    void DisplayCaptureBufferOperation::CaptureSharedTexture(CaptureBufferIndex dst, int srcTextureIndex, bool isScreenshotPermitted, bool isCopyrightRequired, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_CAPOP("capture shared-texture [iBuf=%d]'->%s'\n", srcTextureIndex, GetCaptureBufferNameImpl(dst));

        //
        //  (external)
        //      ↓
        //    Texture
        //
        CaptureTextureImageToTextureImpl(dst, srcTextureIndex, timeout);

        SetImageAttributeImpl(dst, isScreenshotPermitted, isCopyrightRequired);
    }

    void DisplayCaptureBufferOperation::CopyBuffer(CaptureBufferIndex dst, CaptureBufferIndex src, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_CAPOP("copy buffer '%s->%s'\n", GetCaptureBufferNameImpl(src), GetCaptureBufferNameImpl(dst));

        //
        // Texture(src) → Texture(dst)
        //
        CopyBetweenTextureImpl(dst, src, timeout);

        SetImageAttributeImpl(
            dst,
            g_pCaptureBufferManager->IsBufferScreenshotEnabled(src),
            g_pCaptureBufferManager->IsBufferCopyrightRequested(src)
        );
    }

    void DisplayCaptureBufferOperation::FillBuffer(CaptureBufferIndex dst, uint32_t color, bool isScreenshotPermitted, bool isCopyrightRequired) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_CAPOP("fill buffer '->%s'\n", GetCaptureBufferNameImpl(dst));

        //
        // (color) -> Texture
        //
        FillTextureImpl(dst, color);

        SetImageAttributeImpl(dst, isScreenshotPermitted, isCopyrightRequired);
    }

    nn::Result DisplayCaptureBufferOperation::ReadBuffer(void* buffer, size_t bufferSize, CaptureBufferIndex src) NN_NOEXCEPT
    {
        auto srcTextureIndex = GetSharedTextureIndexImpl(src);
        if(srcTextureIndex < 0)
        {
            NN_AM_DISPCTRL_LOG_WARN("source buffer '%s' does not exist on shared buffer\n", GetCaptureBufferNameImpl(src));
            NN_RESULT_THROW(nn::applet::ResultCaptureBufferOperationNotSupported());
        }

        NN_RESULT_THROW_UNLESS(bufferSize >= nn::ae::CaptureBufferSize, nn::applet::ResultOutOfMemory());

        uint64_t readSize = 0;
        NN_RESULT_DO(DisplayPrimitiveOperation::GetDetachedSharedFrameBufferImage(&readSize, buffer, nn::ae::CaptureBufferSize, g_hSharedBuffer, srcTextureIndex));

        if(readSize < bufferSize)
        {
            std::memset(reinterpret_cast<char*>(buffer) + readSize, 0, static_cast<size_t>(bufferSize - readSize));
        }

        NN_RESULT_SUCCESS;
    }

    bool DisplayCaptureBufferOperation::IsBufferScreenshotPermittedForFader(CaptureBufferIndex index) NN_NOEXCEPT
    {
        return g_pCaptureBufferManager->IsBufferScreenshotEnabled(index)
            && !g_pCaptureBufferManager->IsBufferCopyrightRequested(index) // SIGLO-77292: 権利表記を付けているアプリのキャプチャは撮影禁止にする
            ;
    }

}}}}
