﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/core/am_StackableApplet.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/result/result_HandlingUtility.h>
#include <mutex>
#include <utility>

namespace nn { namespace am { namespace service { namespace core {

void StackableApplet::AddChild(std::shared_ptr<Applet> p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    // 既に m_Terminating や m_ExitRequested の場合には通知する
    if (m_Terminating)
    {
        p->Terminate();
    }
    m_Children.push_back(std::move(p));
}

void StackableApplet::CleanJoinableChildren() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_Children.remove_if([](const decltype(m_Children.front())& e)
    {
        return e->TryJoin();
    });
}

// Terminate して Join する
void StackableApplet::CleanAllChildren() NN_NOEXCEPT
{
    for (;;)
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        if (m_Children.empty())
        {
            return;
        }
        auto p = std::move(m_Children.front());
        m_Children.pop_front();
        lk.unlock();
        p->Terminate();
        p->Join();
    }
}

// Terminate だけする
void StackableApplet::TerminateAllChildren() NN_NOEXCEPT
{
    for (;;)
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        if (m_Children.empty())
        {
            return;
        }
        auto p = std::move(m_Children.front());
        m_Children.pop_front();
        lk.unlock();
        p->Terminate();
    }
}

bool StackableApplet::AreAnyChildrenLeft() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    return !m_Children.empty() ? true : false;
}

void StackableApplet::Terminate() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    if (m_Terminating)
    {
        return;
    }
    this->m_Terminating = true;
    for (auto&& e : m_Children)
    {
        e->Terminate();
    }
    this->TerminateOne();
}

bool StackableApplet::TryJoin() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    while (!m_Children.empty())
    {
        if (!m_Children.back()->TryJoin())
        {
            return false;
        }
        m_Children.pop_back();
    }
    return this->TryJoinOne();
}

void StackableApplet::InitializeMultiWaitHolder(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (!m_Children.empty())
    {
        return m_Children.back()->InitializeMultiWaitHolder(pHolder);
    }
    return this->InitializeMultiWaitHolderOne(pHolder);
}

}}}}
