﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/am/detail/am_Log.h>
#include <nn/am/service/am_ErrorReport.h>
#include <nn/am/service/am_SystemReportManager.h>
#include <nn/audioctrl/audioctrl_PlayReport.h>
#include <nn/hid/system/hid_PlayReport.h>
#include <nn/hid/system/hid_RegisteredDevice.h>
#include <nn/os.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/pm/pm_BootModeApi.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_FormatString.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/srepo/srepo_StateNotifier.h>

#include <mutex>

// #define ENABLE_CONTROLLER_REPORT_LOG

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
// Result を返す expression を受けて、失敗の場合には引数なしで return する。
#define NN_AM_SREPO_DO(expression) \
    do \
    { \
        auto _nn_am_srepo_do_result = (expression); \
        if ( _nn_am_srepo_do_result.IsFailure() ) \
        { \
            NN_DETAIL_AM_TRACE("[SystemReportManager] Failed: %s\n  Module: %d\n  Description: %d\n  InnerValue: 0x%08x\n", \
                NN_MACRO_STRINGIZE(expression), _nn_am_srepo_do_result.GetModule(), _nn_am_srepo_do_result.GetDescription(), _nn_am_srepo_do_result.GetInnerValueForDebug()); \
            return; \
        } \
    } while( NN_STATIC_CONDITION(false) )

#endif

namespace nn { namespace am { namespace service {

namespace {

#if defined( ENABLE_CONTROLLER_REPORT_LOG )
    const char* GetControllerDeviceTypeString(hid::system::PlayReportDeviceType type) NN_NOEXCEPT
    {
        switch( type )
        {
        case hid::system::PlayReportDeviceType_JoyConLeft:          return "Joy-Con (L)";
        case hid::system::PlayReportDeviceType_JoyConRight:         return "Joy-Con (R)";
        case hid::system::PlayReportDeviceType_SwitchProController: return "Switch Pro Controller";
        case hid::system::PlayReportDeviceType_UsbController:       return "USB Controller";
        case hid::system::PlayReportDeviceType_Unknown:             return "Unknown";
        default:                                                    return "(unexpected)";
        }
    }

    const char* GetControllerDeviceTypeString(hid::system::DeviceTypeSet type) NN_NOEXCEPT
    {
        switch( hid::system::GetPlayReportDeviceType(type) )
        {
        case hid::system::PlayReportDeviceType_JoyConLeft:          return "Joy-Con (L)";
        case hid::system::PlayReportDeviceType_JoyConRight:         return "Joy-Con (R)";
        case hid::system::PlayReportDeviceType_SwitchProController: return "Switch Pro Controller";
        case hid::system::PlayReportDeviceType_UsbController:       return "USB Controller";
        case hid::system::PlayReportDeviceType_Unknown:             return "Unknown";
        default:                                                    return "(unexpected)";
        }
    }

    const char* GetControllerInterfaceTypeString(hid::system::InterfaceType type) NN_NOEXCEPT
    {
        switch( type )
        {
        case hid::system::InterfaceType_Bluetooth:  return "Bluetooth";
        case hid::system::InterfaceType_Rail:       return "Rail";
        case hid::system::InterfaceType_Usb:        return "USB";
        case hid::system::InterfaceType_Unknown:    return "Unknown";
        default:                                    return "(unexpected)";
        }
    }

    const char* GetControllerStyleString(hid::system::PlayReportPlayStyle style) NN_NOEXCEPT
    {
        switch( style )
        {
        case hid::system::PlayReportPlayStyle_SwitchProController:      return "Switch Pro Controller";
        case hid::system::PlayReportPlayStyle_Handheld:                 return "Handheld";
        case hid::system::PlayReportPlayStyle_JoyConDual:               return "Joy-Con Dual";
        case hid::system::PlayReportPlayStyle_JoyConLeftHorizontal:     return "Joy-Con (L) Horizontal";
        case hid::system::PlayReportPlayStyle_JoyConLeftVertical:       return "Joy-Con (L) Vertical";
        case hid::system::PlayReportPlayStyle_JoyConRightHorizontal:    return "Joy-Con (R) Horizontal";
        case hid::system::PlayReportPlayStyle_JoyConRightVertical:      return "Joy-Con (R) Vertical";
        case hid::system::PlayReportPlayStyle_Unknown:                  return "Unknown";
        default:                                                        return "(unexpected)";
        }
    }

    void PrintPlayReportControllerUsageLog(const hid::system::PlayReportControllerUsage usages[], int count) NN_NOEXCEPT
    {
        NN_DETAIL_AM_TRACE("PlayReportControllerUsage : Count = %d\n", count);
        for( int i = 0; i < count; i++ )
        {
            NN_DETAIL_AM_TRACE("[%d]\n", i);
            NN_DETAIL_AM_TRACE("  Number     : %u\n", usages[i].controllerNumber);
            NN_DETAIL_AM_TRACE("  Style      : %s (%u)\n", GetControllerStyleString(usages[i].style), usages[i].style);
            NN_DETAIL_AM_TRACE("  DeviceType : %s (%u)\n", GetControllerDeviceTypeString(usages[i].deviceType), usages[i].deviceType);
            NN_DETAIL_AM_TRACE("  TypeInfo   : %02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x\n",
                usages[i].raw[7], usages[i].raw[6], usages[i].raw[5], usages[i].raw[4],
                usages[i].raw[3], usages[i].raw[2], usages[i].raw[1], usages[i].raw[0]);
            // 種類別 TypeInfo の解釈
            switch( usages[i].deviceType )
            {
            case hid::system::PlayReportDeviceType_JoyConLeft:
            case hid::system::PlayReportDeviceType_JoyConRight:
            case hid::system::PlayReportDeviceType_SwitchProController:
                {
                    NN_DETAIL_AM_TRACE("  Interface  : %s (%u)\n", GetControllerInterfaceTypeString(usages[i].nxControllerInfo.interfaceType), usages[i].nxControllerInfo.interfaceType);
                    NN_DETAIL_AM_TRACE("  SubType    : %u\n", usages[i].nxControllerInfo.subType);
                    NN_DETAIL_AM_TRACE("  Version    : %x.%x.%x.%x\n",
                        usages[i].nxControllerInfo.version[0], usages[i].nxControllerInfo.version[1],
                        usages[i].nxControllerInfo.version[2], usages[i].nxControllerInfo.version[3]);
                }
                break;
            case hid::system::PlayReportDeviceType_UsbController:
                {
                    NN_DETAIL_AM_TRACE("  VID        : %u\n", (static_cast<uint32_t>(usages[i].usbDevice.vidHigh << 8) || usages[i].usbDevice.vidLow));
                    NN_DETAIL_AM_TRACE("  PID        : %u\n", (static_cast<uint32_t>(usages[i].usbDevice.pidHigh << 8) || usages[i].usbDevice.pidLow));
                }
                break;
            case hid::system::PlayReportDeviceType_Unknown:
            default:
                break;
            }
        }
    }

    void PrintRegisteredDeviceLog(hid::system::RegisteredDevice devices[], int count) NN_NOEXCEPT
    {
        NN_DETAIL_AM_TRACE("RegisteredDevice : Count = %d\n", count);
        for( int i = 0; i < count; i++ )
        {
            NN_DETAIL_AM_TRACE("[%d]\n", i);
            NN_DETAIL_AM_TRACE("  DeviceType : %s (%u)\n", GetControllerDeviceTypeString(devices[i].deviceType), devices[i].deviceType);
            NN_DETAIL_AM_TRACE("  SubType    : %u\n", devices[i].subType);
            NN_DETAIL_AM_TRACE("  Interface  : %s (%u)\n", GetControllerInterfaceTypeString(devices[i].interfaceType), devices[i].interfaceType);
            NN_DETAIL_AM_TRACE("  Address    : %02x%02x%02x%02x%02x%02x\n",
                devices[i].address.address[5], devices[i].address.address[4], devices[i].address.address[3],
                devices[i].address.address[2], devices[i].address.address[1], devices[i].address.address[0]);
            NN_DETAIL_AM_TRACE("  MainColor  : %d.%d.%d.%d\n", devices[i].mainColor.v[3], devices[i].mainColor.v[2], devices[i].mainColor.v[1], devices[i].mainColor.v[0]);
            NN_DETAIL_AM_TRACE("  SubColor   : %d.%d.%d.%d\n", devices[i].subColor.v[3], devices[i].subColor.v[2], devices[i].subColor.v[1], devices[i].subColor.v[0]);
            auto type = hid::system::GetPlayReportDeviceType(devices[i].deviceType);
            if( type == hid::system::PlayReportDeviceType_JoyConLeft || type == hid::system::PlayReportDeviceType_JoyConRight )
            {
                NN_DETAIL_AM_TRACE("  SerialNo.  : %s\n", devices[i].identificationCode);
            }
        }
    }
#endif

    const nn::ApplicationId ControllerReportId{ 0x0100000000001017 };
    const nn::ApplicationId AudioReportId{ 0x0100000000001019 };

    util::optional<ncm::ApplicationId> s_ApplicationId = nullptr;
    os::Mutex s_ApplicationIdMutex(false);
    os::ThreadType  s_Thread;
    os::SystemEvent s_ControllerUsageUpdateEvent;
    os::SystemEvent s_ControllerRegistrationUpdateEvent;
    os::SystemEvent s_AudioVolumeUpdateEvent;
    os::SystemEvent s_AudioOutputDeviceUpdateEvent;
    NN_OS_ALIGNAS_THREAD_STACK char s_ThreadStack[8 * 1024];
    char s_ReportBuffer[1280];

    const nn::TimeSpan ControllerPlayReportTimeSpan = nn::TimeSpan::FromSeconds(60);
    os::TimerEvent g_ControllerPlayReportTimer(os::EventClearMode_ManualClear);

    void ReportControllerUsage() NN_NOEXCEPT
    {
        NN_DETAIL_AM_TRACE("[SystemReportManager] ReportControllerUsage()\n");
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        hid::system::PlayReportControllerUsage usages[hid::system::PlayReportControllerUsageCountMax];
        auto count = hid::system::GetPlayReportControllerUsages(usages, hid::system::PlayReportControllerUsageCountMax);
#if defined( ENABLE_CONTROLLER_REPORT_LOG )
        PrintPlayReportControllerUsageLog(usages, count);
#endif
        if( count == 0 )
        {
            return;
        }

        prepo::SystemPlayReport report("usage");
        report.SetBuffer(s_ReportBuffer, sizeof(s_ReportBuffer));
        NN_AM_SREPO_DO(report.SetApplicationId(ControllerReportId));
        NN_AM_SREPO_DO(report.Add("Count", static_cast<int64_t>(count)));
        for( int i = 0; i < count; i++ )
        {
            char keyBuffer[16];
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "Style%d", i);
            NN_AM_SREPO_DO(report.Add(keyBuffer, static_cast<int64_t>(usages[i].style)));
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "Number%d", i);
            NN_AM_SREPO_DO(report.Add(keyBuffer, static_cast<int64_t>(usages[i].controllerNumber)));
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "Type%d", i);
            auto deviceType = usages[i].deviceType;
            NN_AM_SREPO_DO(report.Add(keyBuffer, static_cast<int64_t>(deviceType)));
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "TypeInfo%d", i);
            NN_AM_SREPO_DO(report.Add(keyBuffer, (static_cast<int64_t>(usages[i].raw[7]) << 56)
                                               | (static_cast<int64_t>(usages[i].raw[6]) << 48)
                                               | (static_cast<int64_t>(usages[i].raw[5]) << 40)
                                               | (static_cast<int64_t>(usages[i].raw[4]) << 32)
                                               | (static_cast<int64_t>(usages[i].raw[3]) << 24)
                                               | (static_cast<int64_t>(usages[i].raw[2]) << 16)
                                               | (static_cast<int64_t>(usages[i].raw[1]) << 8)
                                               | (static_cast<int64_t>(usages[i].raw[0]) << 0)));
        }
        {
            std::lock_guard<decltype(s_ApplicationIdMutex)> lock(s_ApplicationIdMutex);
            if( s_ApplicationId )
            {
                NN_AM_SREPO_DO(report.Add("ApplicationId", prepo::Any64BitId{ (*s_ApplicationId).value }));
            }
        }
        NN_AM_SREPO_DO(report.Save());
#endif
    }

    // コントローラ総数と、レール接続数をsrepoへ通知する(SIGLO-84256)
    void NotifyControllerCount() NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        hid::system::PlayReportControllerUsage usages[hid::system::PlayReportControllerUsageCountMax];
        auto count = hid::system::GetPlayReportControllerUsages(usages, hid::system::PlayReportControllerUsageCountMax);

        // ReportControllerUsage()と異なり count==0 でも実行する

        int railCount = 0;
        for( int i = 0 ; i < count ; i++ ) // レール接続数をカウント
        {
            switch( usages[i].deviceType )
            {
            case hid::system::PlayReportDeviceType_JoyConLeft:
            case hid::system::PlayReportDeviceType_JoyConRight:
            case hid::system::PlayReportDeviceType_SwitchProController:
                if(usages[i].nxControllerInfo.interfaceType == hid::system::InterfaceType_Rail)
                {
                    railCount++;
                }
                break;
            default:
                break;
            }
        }
        nn::srepo::NotifyControllerCountChanged(static_cast<int8_t>(count), static_cast<int8_t>(railCount));
#endif
    }

    void ReportControllerRegistration() NN_NOEXCEPT
    {
        NN_DETAIL_AM_TRACE("[SystemReportManager] ReportControllerRegistration().\n");
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        hid::system::RegisteredDevice devices[hid::system::RegisteredDeviceCountMax];
        auto count = hid::system::GetRegisteredDevices(devices, hid::system::RegisteredDeviceCountMax);
#if defined( ENABLE_CONTROLLER_REPORT_LOG )
        PrintRegisteredDeviceLog(devices, count);
#endif
        if( count == 0 )
        {
            return;
        }

        prepo::SystemPlayReport report("register");
        report.SetBuffer(s_ReportBuffer, sizeof(s_ReportBuffer));
        NN_AM_SREPO_DO(report.SetApplicationId(ControllerReportId));
        NN_AM_SREPO_DO(report.Add("Count", static_cast<int64_t>(count)));
        for( int i = 0; i < count; i++ )
        {
            char keyBuffer[16];
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "Type%d", i);
            NN_AM_SREPO_DO(report.Add(keyBuffer, static_cast<int64_t>(hid::system::GetPlayReportDeviceType(devices[i].deviceType))));
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "SubType%d", i);
            NN_AM_SREPO_DO(report.Add(keyBuffer, static_cast<int64_t>(devices[i].subType)));
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "Address%d", i);
            NN_AM_SREPO_DO(report.Add(keyBuffer, (static_cast<int64_t>(devices[i].address.address[5]) << 40)
                                               | (static_cast<int64_t>(devices[i].address.address[4]) << 32)
                                               | (static_cast<int64_t>(devices[i].address.address[3]) << 24)
                                               | (static_cast<int64_t>(devices[i].address.address[2]) << 16)
                                               | (static_cast<int64_t>(devices[i].address.address[1]) << 8)
                                               | (static_cast<int64_t>(devices[i].address.address[0]) << 0)));
            // シリアル番号を取得するのは Joy-Con のみ。
            if( (devices[i].deviceType == hid::system::DeviceType::JoyConLeft::Mask || devices[i].deviceType == hid::system::DeviceType::JoyConRight::Mask)
              && strlen(devices[i].identificationCode) > 0 )
            {
                util::SNPrintf(keyBuffer, sizeof(keyBuffer), "Serial%d", i);
                NN_AM_SREPO_DO(report.Add(keyBuffer, devices[i].identificationCode));
            }
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "MainColor%d", i);
            NN_AM_SREPO_DO(report.Add(keyBuffer, (static_cast<int64_t>(devices[i].mainColor.v[3]) << 24)
                                               | (static_cast<int64_t>(devices[i].mainColor.v[2]) << 16)
                                               | (static_cast<int64_t>(devices[i].mainColor.v[1]) << 8)
                                               | (static_cast<int64_t>(devices[i].mainColor.v[0]) << 0)));
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "SubColor%d", i);
            NN_AM_SREPO_DO(report.Add(keyBuffer, (static_cast<int64_t>(devices[i].subColor.v[3]) << 24)
                                               | (static_cast<int64_t>(devices[i].subColor.v[2]) << 16)
                                               | (static_cast<int64_t>(devices[i].subColor.v[1]) << 8)
                                               | (static_cast<int64_t>(devices[i].subColor.v[0]) << 0)));
            util::SNPrintf(keyBuffer, sizeof(keyBuffer), "InterfaceType%d", i);
            NN_AM_SREPO_DO(report.Add(keyBuffer, static_cast<int64_t>(devices[i].interfaceType)));
        }
        {
            std::lock_guard<decltype(s_ApplicationIdMutex)> lock(s_ApplicationIdMutex);
            if( s_ApplicationId )
            {
                NN_AM_SREPO_DO(report.Add("ApplicationId", prepo::Any64BitId{ (*s_ApplicationId).value }));
            }
        }
        NN_AM_SREPO_DO(report.Save());
#endif
    }

    void ReportAudioSettings(const char* eventId) NN_NOEXCEPT
    {
        NN_DETAIL_AM_TRACE("[SystemReportManager] ReportAudioSettings(%s).\n", eventId);
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        prepo::SystemPlayReport report(eventId);
        report.SetBuffer(s_ReportBuffer, sizeof(s_ReportBuffer));
        NN_AM_SREPO_DO(report.SetApplicationId(AudioReportId));

        audioctrl::PlayReportAudioVolumeData volData;
        audioctrl::GetAudioVolumeDataForPlayReport(&volData);
        NN_AM_SREPO_DO(report.Add("SpeakerVolume", static_cast<int64_t>(volData.speaker.volume)));
        NN_AM_SREPO_DO(report.Add("SpeakerMute", static_cast<int64_t>(volData.speaker.mute ? 1 : 0)));
        NN_AM_SREPO_DO(report.Add("HeadphoneVolume", static_cast<int64_t>(volData.headphone.volume)));
        NN_AM_SREPO_DO(report.Add("HeadphoneMute", static_cast<int64_t>(volData.headphone.mute ? 1 : 0)));
        NN_AM_SREPO_DO(report.Add("IsHeadphonePowerLimited", static_cast<int64_t>(volData.isHeadphonePowerLimited ? 1 : 0)));
        NN_AM_SREPO_DO(report.Add("UsbOutputDeviceVolume", static_cast<int64_t>(volData.usbOutputDevice.volume)));
        NN_AM_SREPO_DO(report.Add("UsbOutputDeviceMute", static_cast<int64_t>(volData.usbOutputDevice.mute ? 1 : 0)));

        auto outputTarget = audioctrl::GetAudioOutputTargetForPlayReport();
        NN_AM_SREPO_DO(report.Add("OutputTarget", static_cast<int64_t>(outputTarget)));
        {
            std::lock_guard<decltype(s_ApplicationIdMutex)> lock(s_ApplicationIdMutex);
            if( s_ApplicationId )
            {
                NN_AM_SREPO_DO(report.Add("ApplicationId", prepo::Any64BitId{ (*s_ApplicationId).value }));
            }
        }
        NN_AM_SREPO_DO(report.Save());
#else
        NN_UNUSED(eventId);
#endif
    }
}

void StartSystemReportManager() NN_NOEXCEPT
{
    bool enabled = false;
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    size_t size = settings::fwdbg::GetSettingsItemValue(&enabled, sizeof(enabled), "systemreport", "enabled");
    NN_ABORT_UNLESS_EQUAL(size, sizeof(enabled));
#endif

    enabled = (enabled && (pm::GetBootMode() != pm::BootMode_Maintenance));

    NN_DETAIL_AM_TRACE("[SystemReportManager] SystemReport is %s.\n", enabled ? "enabled" : "disabled");

    if( enabled )
    {
        hid::system::BindPlayReportControllerUsageUpdateEvent(s_ControllerUsageUpdateEvent.GetBase(), os::EventClearMode_ManualClear);
        hid::system::BindPlayReportRegisteredDeviceUpdateEvent(s_ControllerRegistrationUpdateEvent.GetBase(), os::EventClearMode_ManualClear);

        audioctrl::BindAudioVolumeUpdateEventForPlayReport(s_AudioVolumeUpdateEvent.GetBase());
        audioctrl::BindAudioOutputDeviceUpdateEventForPlayReport(s_AudioOutputDeviceUpdateEvent.GetBase());

        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&s_Thread, [](void*)
        {
            while( NN_STATIC_CONDITION(true) )
            {
                int eventId = os::WaitAny(
                    s_ControllerUsageUpdateEvent.GetBase(),
                    s_ControllerRegistrationUpdateEvent.GetBase(),
                    s_AudioVolumeUpdateEvent.GetBase(),
                    s_AudioOutputDeviceUpdateEvent.GetBase(),
                    g_ControllerPlayReportTimer.GetBase());

                switch( eventId )
                {
                case 0:
                    {
                        NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_SystemReport_UpdateErrorReportControllerUsage, 120);
                        s_ControllerUsageUpdateEvent.Clear();
                        // エラーレポートに記録するコントローラーの接続状態は即時更新する。
                        UpdateErrorReportControllerUsage();
                        // プレイレポートで送信するコントローラーの接続状態は60秒間安定するのを待つ（60秒以内に再度変化があった場合はタイマーを再トリガー）。
                        g_ControllerPlayReportTimer.StartOneShot(ControllerPlayReportTimeSpan);
                    }
                    break;
                case 1:
                    {
                        NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_SystemReport_ReportControllerRegistration, 120);
                        s_ControllerRegistrationUpdateEvent.Clear();
                        ReportControllerRegistration();
                    }
                    break;
                case 2:
                    {
                        NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_SystemReport_ReportAudioVolumeUpdate, 120);
                        s_AudioVolumeUpdateEvent.Clear();
                        ReportAudioSettings("volume_update");
                    }
                    break;
                case 3:
                    {
                        NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_SystemReport_ReportAudioOutputTargetUpdate, 120);
                        s_AudioOutputDeviceUpdateEvent.Clear();
                        ReportAudioSettings("output_target_update");
                    }
                    break;
                case 4:
                    {
                        NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_SystemReport_ReportControllerUsage, 120);
                        g_ControllerPlayReportTimer.Clear();
                        ReportControllerUsage();
                        NotifyControllerCount();
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }, nullptr, s_ThreadStack, sizeof(s_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(am, SystemReportTask)));
        os::SetThreadNamePointer(&s_Thread, NN_SYSTEM_THREAD_NAME(am, SystemReportTask));
        os::StartThread(&s_Thread);
    }
}

void SetSystemReportApplicationId(const ncm::ApplicationId& id) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_ApplicationIdMutex)> lock(s_ApplicationIdMutex);
    s_ApplicationId = id;
    // アプリ起動前後でコントローラーの接続状態が変更されなかった場合でもそのアプリで使ったコントローラーの状態を記録できるようにタイマーを動かす。
    g_ControllerPlayReportTimer.StartOneShot(ControllerPlayReportTimeSpan);
}

void ClearSystemReportApplicationId() NN_NOEXCEPT
{
    std::lock_guard<decltype(s_ApplicationIdMutex)> lock(s_ApplicationIdMutex);
    s_ApplicationId = nullptr;
}

}}}
