﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_MultipleWait.h>
#include "am_ShellEvent.h"
#include "am_AudioControl.h"
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/applet/applet_Types.h>
#include <nn/omm/omm_Api.h>
#include "../../../Processes/am/am_OperationModeManagerMain.h"


namespace nn { namespace am { namespace service {

//-----------------------------------------------------------------------------
//  コンストラクタ
//
ShellEvent::ShellEvent(AppletSystem* pAppletSystem) NN_NOEXCEPT
      : m_pAppletSystem(pAppletSystem),
        m_ShellEvent(os::EventClearMode_AutoClear, true)
{
    ns::GetApplicationShellEvent(&m_ShellEvent);
    os::InitializeMultiWaitHolder(&m_ShellEventHolder, m_ShellEvent.GetBase());

    m_ShellEventHolder.userData = reinterpret_cast<uintptr_t>(this);
}


//-----------------------------------------------------------------------------
//  デストラクタ
//
ShellEvent::~ShellEvent() NN_NOEXCEPT
{
    os::FinalizeMultiWaitHolder(&m_ShellEventHolder);
}


//-----------------------------------------------------------------------------
//  デバッガでの停止時に対象アプレットの音声を停止する
//
void ShellEvent::SuspendAudioOnDebugging(os::ProcessId pid) NN_NOEXCEPT
{
    am::service::SuspendAudioInProcessForDebug(pid);
}

//-----------------------------------------------------------------------------
//  デバッガでの再実行時に対象アプレットの音声を再開する
//
void ShellEvent::ResumeAudioOnDebugging(os::ProcessId pid) NN_NOEXCEPT
{
    am::service::ResumeAudioInProcessForDebug(pid);
}

//-----------------------------------------------------------------------------
//  ShellEvent のハンドラ関数（多重待ち解除時に呼ばれる）
//
void ShellEvent::ProcessShellEvent() NN_NOEXCEPT
{
    if (!m_ShellEvent.TryWait())
    {
        return;
    }

    ns::ApplicationShellEventInfo infoList[32];
    int listCount = sizeof(infoList) / sizeof(infoList[0]);

    for (;;)
    {
        auto count = ns::PopApplicationShellEventInfo(infoList, listCount);
        for (int i = 0; i < count; ++i)
        {
            auto event = infoList[i].event;
            auto pid   = infoList[i].processId;

            //NN_SDK_LOG("[AM:fromNS] event=%d pid=%16lld\n", event, pid);

            switch (event)
            {
            case ns::ApplicationShellEvent::Exit:
                    /// プロセスの終了を通知
                    NN_AM_SERVICE_LOG(event, NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT " exit\n", NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT_PARAMETER(pid));
                    m_pAppletSystem->GetNsProcessManager()->NotifyExit(pid);
                    break;

            case ns::ApplicationShellEvent::DebugRunning:
                    // TORIAEZU: TODO:
                    NN_AM_SERVICE_LOG(event, NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT " running\n", NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT_PARAMETER(pid));
                    //  デバッガで対象プロセス実行時に Audio を再開させる
                    ResumeAudioOnDebugging(pid);
                    break;

            case ns::ApplicationShellEvent::DebugBreaked:
                    // TORIAEZU: TODO:
                    NN_AM_SERVICE_LOG(event, NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT " debug break\n", NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT_PARAMETER(pid));
                    //  デバッガで対象プロセス停止時に Audio を停止させる
                    SuspendAudioOnDebugging(pid);
                    break;

            case ns::ApplicationShellEvent::RebootRequest:
                    // 再起動
                    NN_AM_SERVICE_LOG(event, NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT " reboot\n", NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT_PARAMETER(pid));
                    m_pAppletSystem->RequestToReboot();
                    break;

            case ns::ApplicationShellEvent::NotifyStartCardUpdate:
                    // カードの自動起動時の CUP 適用処理の開始
                    NN_AM_SERVICE_LOG(event, NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT " start card update\n", NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT_PARAMETER(pid));
                    HideStartupLogoOfOperationModeManager();
                    am::service::ChangeDisplayLayerControlModeToSystem();
                    omm::FadeInDisplay();
                    break;

            case ns::ApplicationShellEvent::ShowCardUpdateProcessingRequest:
                    // カードの自動起動時に CUP 適用処理が動いていることを確認するための表示
                    NN_AM_SERVICE_LOG(event, NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT " show card update processing\n", NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT_PARAMETER(pid));
                    omm::ShowCardUpdateProcessing();
                    break;

            case ns::ApplicationShellEvent::NotifyEndCardUpdate:
                    // カードの自動起動時の CUP 適用処理の終了
                    NN_AM_SERVICE_LOG(event, NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT " end card update\n", NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT_PARAMETER(pid));
                    omm::FadeOutDisplay();
                    break;

            case ns::ApplicationShellEvent::Exception:
                    // プロセスの例外
                    NN_AM_SERVICE_LOG(event, NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT " exception\n", NN_AM_SERVICE_DIAGNOSTICS_PROCESS_ID_STRING_FORMAT_PARAMETER(pid));
                    m_pAppletSystem->GetWindowManager()->NotifyProcessException(pid);
                    break;

            default:
                    break;
            }
        }
        if (count < listCount)
        {
            break;
        }
    }
}


//-----------------------------------------------------------------------------

}}} // namespace nn::am::service

