﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_ServiceConfig.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/util/util_IntUtil.h>
#include <cstring>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/system/settings_SystemApplication.h>

namespace nn { namespace am { namespace service {

namespace {

    // am 関連
    bool g_IsDevelopmentFunctionEnabled = false;
    bool g_AbortsOnSystemAppletLost = true;
    bool g_AbortsOnOverlayAppletLost = true;
    bool g_KillsApplicationOnGpuError = true;
    bool g_DumpsApplicationGpuCoreDumpOnGpuError = false;
    bool g_NegatesTransitionLayerTexture = false;

    // am 以外
    bool g_IsQuestMode = false;
    bool g_IsProdMode = false;
}

void EnableDevelopmentFunction() NN_NOEXCEPT
{
    g_IsDevelopmentFunctionEnabled = true;
}

bool IsDevelopmentFunctionEnabled() NN_NOEXCEPT
{
    return g_IsDevelopmentFunctionEnabled;
}

bool AbortsOnSystemAppletLost() NN_NOEXCEPT
{
    return !g_IsDevelopmentFunctionEnabled || g_AbortsOnSystemAppletLost;
}

bool AbortsOnOverlayAppletLost() NN_NOEXCEPT
{
    return !g_IsDevelopmentFunctionEnabled || g_AbortsOnOverlayAppletLost;
}

bool KillsApplicationOnGpuError() NN_NOEXCEPT
{
    return g_KillsApplicationOnGpuError;
}

bool DumpsApplicationGpuCoreDumpOnGpuError() NN_NOEXCEPT
{
    return g_DumpsApplicationGpuCoreDumpOnGpuError;
}

bool NegatesTransitionLayerTexture() NN_NOEXCEPT
{
    return g_NegatesTransitionLayerTexture;
}

bool IsQuestMode() NN_NOEXCEPT
{
    return g_IsQuestMode;
}

bool IsProdMode() NN_NOEXCEPT
{
    return g_IsProdMode;
}

namespace {

    template <typename T, typename U>
    T ReadFirmwareDebugSettings(const char* name, const char* subName, U&& defaultValue) NN_NOEXCEPT
    {
        T ret;
        auto size = nn::settings::fwdbg::GetSettingsItemValue(&ret, sizeof(ret), name, subName);
        if (!(sizeof(ret) == size))
        {
            NN_AM_SERVICE_LOG(error, "failed: ReadFirmwareDebugSettings(%s/%s) size = %d(expected:%d)\n", name, subName, static_cast<int>(size), static_cast<int>(sizeof(ret)));
            return defaultValue;
        }
        return ret;
    }

    Bit64 ReadFirmwareDebugSettingsBit64ValueAsString(const char* name, const char* subName, Bit64 defaultValue)
    {
        char buffer[32];
        auto n = settings::fwdbg::GetSettingsItemValue(buffer, sizeof(buffer), name, subName);
        if (!(n < sizeof(buffer)))
        {
            return defaultValue;
        }
        buffer[n] = '\0';

        char* endP;
        auto ret = std::strtoull(buffer, &endP, 0);
        if (!(*endP == '\0'))
        {
            return defaultValue;
        }
        return ret;
    }

}

void SetConfigsByFirmwareDebugSettings()
{
    g_IsDevelopmentFunctionEnabled = ReadFirmwareDebugSettings<bool>("am.debug", "dev_function", false);
    g_AbortsOnSystemAppletLost = ReadFirmwareDebugSettings<bool>("am.debug", "abort_on_sa_lost", true);
    g_AbortsOnOverlayAppletLost = ReadFirmwareDebugSettings<bool>("am.debug", "abort_on_oa_lost", true);
    g_KillsApplicationOnGpuError = ReadFirmwareDebugSettings<bool>("am.debug", "gpu_error_kill_app", true);
    g_DumpsApplicationGpuCoreDumpOnGpuError = ReadFirmwareDebugSettings<bool>("gpu_core_dump", "auto_dump", false);
    g_NegatesTransitionLayerTexture = ReadFirmwareDebugSettings<bool>("am.debug", "negative_transition_layer", false);

    g_IsQuestMode = settings::system::GetQuestFlag();
    g_IsProdMode = !nn::settings::fwdbg::IsDebugModeEnabled();
}

TimeSpan GetInitialDefaultHomeLongPressTime() NN_NOEXCEPT
{
    return TimeSpan::FromMilliSeconds(ReadFirmwareDebugSettings<int>("am.debug", "home_long_pressed_time_ms", 500));
}

TimeSpan GetCaptureLongPressTime() NN_NOEXCEPT
{
    return TimeSpan::FromMilliSeconds(ReadFirmwareDebugSettings<int>("am.debug", "capture_long_pressed_time_ms", 384));
}

uint32_t GetDefaultContinuousRecordingVideoBitRate() NN_NOEXCEPT
{
    auto value = ReadFirmwareDebugSettings<int>("am.debug", "continuous_recording_video_bit_rate", 5000000);
    return util::IsIntValueRepresentable<uint32_t>(value) ? static_cast<uint32_t>(value) : 5000000;
}

TimeSpan GetDefaultContinuousRecordingMaxTime() NN_NOEXCEPT
{
    return TimeSpan::FromMilliSeconds(ReadFirmwareDebugSettings<int>("am.debug", "continuous_recording_max_time_ms", 30000));
}

TimeSpan GetDefaultContinuousRecordingMinTime() NN_NOEXCEPT
{
    return TimeSpan::FromMilliSeconds(ReadFirmwareDebugSettings<int>("am.debug", "continuous_recording_min_time_ms", 3500));
}

uint8_t GetDefaultContinuousRecordingFps() NN_NOEXCEPT
{
    auto fps = ReadFirmwareDebugSettings<int>("am.debug", "continuous_recording_fps", 30);
    switch (fps)
    {
        case 30: return 30;
        case 60: return 60;
        default: return 30;
    }
}

uint16_t GetDefaultContinuousRecordingFrameCountBetweenIdr() NN_NOEXCEPT
{
    auto n = ReadFirmwareDebugSettings<int>("am.debug", "continuous_recording_key_frame_count", 15);
    if (!(util::IsIntValueRepresentable<uint16_t>(n) && 1 <= n))
    {
        n = 15;
    }
    return static_cast<uint16_t>(n);
}

TimeSpan GetDefaultContinuousRecordingAdditionalTimeOnDockInOut() NN_NOEXCEPT
{
    return TimeSpan::FromMilliSeconds(ReadFirmwareDebugSettings<int>("am.debug", "continuous_recording_add_time_on_dock_inout_ms", 1000));
}

bool IsContinuousRecordingDebugMode() NN_NOEXCEPT
{
    return g_IsDevelopmentFunctionEnabled && ReadFirmwareDebugSettings<bool>("am.debug", "continuous_recording_debug_mode", false);
}

TimeSpan GetContinuousRecordingMaxTimeOnDebugMode() NN_NOEXCEPT
{
    return TimeSpan::FromMilliSeconds(ReadFirmwareDebugSettings<int>("am.debug", "continuous_recording_max_time_ms_on_debug_mode", 30000));
}

size_t GetContinuousRecordingExtraMemorySize(Bit64 applicationId) NN_NOEXCEPT
{
    auto size = ReadFirmwareDebugSettings<uint32_t>("am.debug", "continuous_recording_extra_memory_size", 0);
    if (size == 0)
    {
        return 0;
    }

    auto target = ReadFirmwareDebugSettingsBit64ValueAsString("am.debug", "continuous_recording_extra_memory_target_application_id", 0);
    if (!(target == 0 || target == applicationId))
    {
        return 0;
    }

    return util::align_down(size, 2 * 1024 * 1024);
}

bool IsContinuousRecordingForceDisabled() NN_NOEXCEPT
{
    return ReadFirmwareDebugSettings<bool>("am.debug", "force_disable_continuous_recording", false);
}

bool SavesMovieAutomaticallyOnException() NN_NOEXCEPT
{
    return ReadFirmwareDebugSettings<bool>("am.debug", "saves_movie_automatically_on_exception", false);
}

util::optional<Bit64> GetApplicationIdByContentActionNameByDebugConfig(const char* name) NN_NOEXCEPT
{
    if (!g_IsDevelopmentFunctionEnabled)
    {
        return util::nullopt;
    }
    char buffer[65];
    auto n = nn::settings::fwdbg::GetSettingsItemValue(buffer, sizeof(buffer), "am.debug", "content_action_name_for_debug_0");
    if (!(n < sizeof(buffer)))
    {
        return util::nullopt;
    }
    buffer[n] = 0;
    if (!(std::strncmp(name, buffer, sizeof(buffer) - 1) == 0))
    {
        return util::nullopt;
    }
    auto id = ReadFirmwareDebugSettingsBit64ValueAsString("am.debug", "content_action_application_id_for_debug_0", 0);
    if (id == 0)
    {
        return util::nullopt;
    }
    return id;
}

}}}
