﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/omm/omm_Api.h>
#include <nn/am/service/am_AppletSystem.h>

namespace nn { namespace am { namespace service {

//-----------------------------------------------------------------------------
//  Omm 監視用クラス
//  TORIAEZU:
//      単一スレッドでの利用を前提としているため、
//      m_State 変数の排他などは行なわない。
//
class OmmEvent
{
public:
    OmmEvent() NN_NOEXCEPT;
    ~OmmEvent() NN_NOEXCEPT;

    void SetupOmmEventHolder() NN_NOEXCEPT;
    void SetAppletSystem(AppletSystem* pAppletSystem) NN_NOEXCEPT
    {
        m_pAppletSystem = pAppletSystem;
    }

    void LinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        os::LinkMultiWaitHolder(p, &m_OmmEventHolder);
        os::LinkMultiWaitHolder(p, &m_StartupLogoTimerHolder);
    }
    void UnlinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        NN_UNUSED(p);
        os::UnlinkMultiWaitHolder(&m_OmmEventHolder);
    }
    bool HandleEventIfHolderOwner(os::MultiWaitHolderType* holder)
    {
        if (holder == &m_OmmEventHolder)
        {
            OmmEventHandler();
            return true;
        }
        if (holder == &m_StartupLogoTimerHolder)
        {
            StartupLogoTimerHandler();
            return true;
        }
        return false;
    }

    void StartStartupLogoTimer(const TimeSpan& timespan) NN_NOEXCEPT
    {
        m_StartupLogoTimer.StartOneShot(timespan);
    }

private:
    void OmmEventHandler() NN_NOEXCEPT;
    void StartupLogoTimerHandler() NN_NOEXCEPT;

    void NotifyOperationModeChanged() NN_NOEXCEPT
    {
        NN_AM_SERVICE_LOG(event, "operation mode changed\n");
        m_pAppletSystem->GetWindowManager()->NotifyOperationModeChanged();
        m_pAppletSystem->GetWindowManager()->NotifyIdlePolicyChanged();
    }

private:
    AppletSystem*               m_pAppletSystem;

    os::SystemEvent             m_OmmEvent;
    os::TimerEvent m_StartupLogoTimer{os::EventClearMode_AutoClear};

    os::MultiWaitHolderType     m_OmmEventHolder;
    os::MultiWaitHolderType m_StartupLogoTimerHolder;

    nn::omm::OperationMode      m_Mode;
};

OmmEvent& GetOmmEvent() NN_NOEXCEPT;

//-----------------------------------------------------------------------------

}}} // namespace nn::am::service

