﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/am/service/am_PlayDataManager.h>
#include <nn/am/service/am_StuckChecker.h>

#include "am_OmmEvent.h"
#include "../../../Processes/am/am_OperationModeManagerMain.h"

namespace nn { namespace am { namespace service {

OmmEvent& GetOmmEvent() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(OmmEvent, g_OmmEvent);
    return g_OmmEvent;
}

//-----------------------------------------------------------------------------
//  コンストラクタ
//
OmmEvent::OmmEvent() NN_NOEXCEPT
{
    m_StartupLogoTimer.Clear();
    os::InitializeMultiWaitHolder( &m_StartupLogoTimerHolder, m_StartupLogoTimer.GetBase() );
    m_StartupLogoTimerHolder.userData = reinterpret_cast<uintptr_t>(this);

    // omm の初期化が終わるまで何もできないので、
    // EventHandler から SetupOmmEventHolder() を別途呼んでもらう
}

void OmmEvent::SetupOmmEventHolder() NN_NOEXCEPT
{
    // SIGLO-79632
    // EventHandler 起動時は既に登録済みの動作モードを初期値とみなす。
    // 初呼出時は omm から取得したものを受け取る。
    m_Mode = m_pAppletSystem->GetOperationMode();
    NN_AM_SERVICE_STUCK_CHECKED(omm_Get, 60, nn::omm::GetOperationModeChangeEvent( &m_OmmEvent ));
    NotifyOperationModeChangeEvent(static_cast<oe::OperationMode>(m_Mode));

    os::InitializeMultiWaitHolder( &m_OmmEventHolder, m_OmmEvent.GetBase() );
    m_OmmEventHolder.userData = reinterpret_cast<uintptr_t>(this);
}

//-----------------------------------------------------------------------------
//  デストラクタ
//
OmmEvent::~OmmEvent() NN_NOEXCEPT
{
    os::FinalizeMultiWaitHolder( &m_OmmEventHolder );
    os::FinalizeMultiWaitHolder( &m_StartupLogoTimerHolder );
}


//-----------------------------------------------------------------------------
//  OmmEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void OmmEvent::OmmEventHandler() NN_NOEXCEPT
{
    if (!m_OmmEvent.TryWait())
    {
        return;
    }

    auto mode = NN_AM_SERVICE_STUCK_CHECKED(omm_Get, 60, nn::omm::GetOperationMode());
    if ( mode != m_Mode )
    {
        NotifyOperationModeChangeEvent( static_cast<oe::OperationMode>(mode) );
        m_Mode = mode;
        m_pAppletSystem->SetOperationMode(m_Mode);
        NotifyOperationModeChanged();
    }
}

void OmmEvent::StartupLogoTimerHandler() NN_NOEXCEPT
{
    if (!m_StartupLogoTimer.TryWait())
    {
        return;
    }

    os::UnlinkMultiWaitHolder(&m_StartupLogoTimerHolder);
    HideStartupLogoOfOperationModeManager();
}


//-----------------------------------------------------------------------------

}}} // namespace nn::am::service
