﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_Lock.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <utility>
#include <mutex>
#include <limits>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/am/am_Result.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>

namespace nn { namespace am { namespace service {

namespace {

class LockAccessorImpl
{
public:

    explicit LockAccessorImpl(std::shared_ptr<LockAccessorBody> pBody) NN_NOEXCEPT
        : m_Body(std::move(pBody))
    {
    }

    Result GetEvent(sf::Out<sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        *pOut = sf::NativeHandle(os::GetReadableHandleOfSystemEvent(m_Body->GetEventForLockable()), false);
        NN_RESULT_SUCCESS;
    }

    Result TryLock(sf::Out<bool> pOut, sf::Out<sf::NativeHandle> pOutEventHandle, bool needsEvent) NN_NOEXCEPT
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        NN_RESULT_THROW_UNLESS(m_Count < std::numeric_limits<decltype(m_Count)>::max(), am::ResultInvalidCall());
        auto locked = m_Count > 0 || m_Body->TryLock();
        if (locked)
        {
            ++this->m_Count;
            lk.unlock();
            *pOut = true;
            NN_RESULT_SUCCESS;
        }
        else
        {
            lk.unlock();
            if (needsEvent)
            {
                NN_RESULT_DO(this->GetEvent(pOutEventHandle));
            }
            *pOut = false;
            NN_RESULT_SUCCESS;
        }
    }

    Result Unlock() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        NN_RESULT_THROW_UNLESS(m_Count > 0, am::ResultInvalidCall());
        --this->m_Count;
        if (m_Count == 0)
        {
            m_Body->Unlock();
        }
        NN_RESULT_SUCCESS;
    }

    ~LockAccessorImpl() NN_NOEXCEPT
    {
        if (m_Count > 0)
        {
            m_Body->Unlock();
        }
    }

private:

    const std::shared_ptr<LockAccessorBody> m_Body;
    os::Mutex m_Mutex{false};
    uint32_t m_Count{0};

};

}

sf::SharedPointer<ILockAccessor> MakeSfLock(std::shared_ptr<LockAccessorBody> pBody) NN_NOEXCEPT
{
    return SfObjectFactory::CreateSharedEmplaced<ILockAccessor, LockAccessorImpl>(std::move(pBody));
}

ReaderWriterLock::ReaderWriterLock() NN_NOEXCEPT
{
    m_ReaderLockable.Signal();
    m_WriterLockable.Signal();
}

class ReaderWriterLock::ReaderLock
    : public LockAccessorBody
{
private:

    const std::shared_ptr<ReaderWriterLock> m_Parent;

public:

    explicit ReaderLock(std::shared_ptr<ReaderWriterLock> parent) NN_NOEXCEPT
        : m_Parent(std::move(parent))
    {
    }

    virtual bool TryLock() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Parent->m_Mutex)> lk(m_Parent->m_Mutex);
        auto& state = m_Parent->m_LockState;
        if (state >= 0)
        {
            NN_SDK_ASSERT(m_Parent->m_ReaderLockable.TryWait());
            // reader が取れた
            if (state == 0)
            {
                // unlocked 状態からの場合、writer 側のイベントを clear
                m_Parent->m_WriterLockable.Clear();
            }
            ++state;
            NN_SDK_ASSERT(m_Parent->m_ReaderLockable.TryWait());
            NN_SDK_ASSERT(!m_Parent->m_WriterLockable.TryWait());
            return true;
        }
        else
        {
            NN_SDK_ASSERT(!m_Parent->m_ReaderLockable.TryWait());
            NN_SDK_ASSERT(!m_Parent->m_WriterLockable.TryWait());
            return false;
        }
    }

    virtual void Unlock() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Parent->m_Mutex)> lk(m_Parent->m_Mutex);
        auto& state = m_Parent->m_LockState;
        NN_SDK_ASSERT(state > 0);
        --state;
        if (state == 0)
        {
            // unlocked になった場合、writer 側のイベントを signal
            m_Parent->m_WriterLockable.Signal();
        }
        NN_SDK_ASSERT(m_Parent->m_ReaderLockable.TryWait());
    }

    virtual os::SystemEventType* GetEventForLockable() NN_NOEXCEPT
    {
        return m_Parent->m_ReaderLockable.GetBase();
    }

};

std::shared_ptr<LockAccessorBody> ReaderWriterLock::GetReaderLockAccessor() NN_NOEXCEPT
{
    return MakeShared<ReaderLock>(this->shared_from_this());
}

class ReaderWriterLock::WriterLock
    : public LockAccessorBody
{
private:

    const std::shared_ptr<ReaderWriterLock> m_Parent;

public:

    explicit WriterLock(std::shared_ptr<ReaderWriterLock> parent) NN_NOEXCEPT
        : m_Parent(std::move(parent))
    {
    }

    virtual bool TryLock() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Parent->m_Mutex)> lk(m_Parent->m_Mutex);
        auto& state = m_Parent->m_LockState;
        if (state == 0)
        {
            NN_SDK_ASSERT(m_Parent->m_WriterLockable.TryWait());
            state = -1;
            m_Parent->m_WriterLockable.Clear();
            m_Parent->m_ReaderLockable.Clear();
            NN_SDK_ASSERT(!m_Parent->m_ReaderLockable.TryWait());
            NN_SDK_ASSERT(!m_Parent->m_WriterLockable.TryWait());
            return true;
        }
        else
        {
            NN_SDK_ASSERT(!m_Parent->m_ReaderLockable.TryWait());
            NN_SDK_ASSERT(!m_Parent->m_WriterLockable.TryWait());
            return false;
        }
    }

    virtual void Unlock() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Parent->m_Mutex)> lk(m_Parent->m_Mutex);
        auto& state = m_Parent->m_LockState;
        NN_SDK_ASSERT(state == -1);
        state = 0;
        m_Parent->m_WriterLockable.Signal();
        m_Parent->m_ReaderLockable.Signal();
        NN_SDK_ASSERT(m_Parent->m_ReaderLockable.TryWait());
        NN_SDK_ASSERT(m_Parent->m_WriterLockable.TryWait());
    }

    virtual os::SystemEventType* GetEventForLockable() NN_NOEXCEPT
    {
        return m_Parent->m_WriterLockable.GetBase();
    }

};

std::shared_ptr<LockAccessorBody> ReaderWriterLock::GetWriterLockAccessor() NN_NOEXCEPT
{
    return MakeShared<WriterLock>(this->shared_from_this());
}

}}}
