﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_IntegratedApplication.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_UniqueLock.h>
#include <utility>
#include <memory>
#include <mutex>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/am/service/am_ErrorReport.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/am/am_Result.h>
#include <nn/applet/applet_Result.h>
#include <nn/applet/applet_Types.h>
#include <nn/ae/ae_Types.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/am/service/am_ApplicationFunctions.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/arp/arp_Api.h>
#include <nn/am/service/am_PlayDataManager.h>
#include <nn/am/service/am_SystemReportManager.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/am/service/am_ContinuousRecording.h>
#include <nn/am/service/am_RightsEnvironment.h>
#include <nn/oe/oe_ApplicationControlTypes.h>
#include <nn/oe/oe_DebugApis.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ApplicationSaveDataManagement.h>
#include <nn/os/os_MultipleWaitUtility.h>
#include <nn/os/os_TimerEvent.h>
#include "am_ApplicationTimer.h"
#include "am_GrcControl.h"
#include "am_CommonUtility.h"

namespace nn { namespace am { namespace service {

ApplicationErrorReportInfo IntegratedApplication::GetApplicationErrorReportInfo() const NN_NOEXCEPT
{
    return {m_ControlProperty, *m_pApplicationLaunchInfo};
}

void IntegratedApplication::InitializeImplForConstructors() NN_NOEXCEPT
{
    for (auto&& e: m_Channels)
    {
        e = MakeShared<StorageChannel>(true);
    }
    ns::GameCardAttachmentInfo info;
    ns::GetGameCardAttachmentInfo(&info);
    this->m_InitialGameCardSequenceNumber = info.sequence;

    // LastApplication のキャプチャバッファを黒塗りでクリアしておく
    // アプリ起動時なので基本的には 1 回で成功する。
    for (int i=0; i<20; ++i)
    {
        NN_AM_SERVICE_APPLET_LOG(seq, this, "Invoke ClearCaptureImage(index=%d)", am::service::CaptureBufferIndex_LastApplication);
        auto result = am::service::ClearCaptureBufferImage(am::service::CaptureBufferIndex_LastApplication, am::service::CaptureBufferColorBlack, true);
        if (result.IsSuccess())
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    }

    if (m_AppletIdentityInfo.IsApplication())
    {
        auto pRightsEnvironment = MakeShared<service::rightsManagement::ApplicationRightsEnvironment>();
        auto result = pRightsEnvironment->Initialize(m_AppletIdentityInfo.applicationId);
        if (result.IsSuccess())
        {
            // ホスト起動など、アプリケーション記録がないような場合には失敗するので、成功した場合のみ登録
            OverrideRightsEnvironment(std::move(pRightsEnvironment));
        }
    }
}

arp::ApplicationLaunchProperty IntegratedApplication::MakeArpApplicationLaunchProperty() const NN_NOEXCEPT
{
    arp::ApplicationLaunchProperty ret;
    ret.id = m_pApplicationLaunchInfo->id;
    ret.patchStorageId = m_pApplicationLaunchInfo->patchStorageId;
    ret.programIndex = static_cast<uint8_t>(m_SubProgramJumpInfo.GetCurrentProgramIndex());
    ret.storageId = m_pApplicationLaunchInfo->applicationStorageId;
    ret.version = m_pApplicationLaunchInfo->version;
    return ret;
}

Result IntegratedApplication::EnsureApplicationProperties() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_ApplicationPropertiesMutex)> lk(m_ApplicationPropertiesMutex);
    if (!m_pApplicationPropertiesResult)
    {
        this->m_pApplicationPropertiesResult = [&]() -> Result
        {
            if (m_AppletIdentityInfo.IsApplication())
            {
                auto pNs = GetAppletSystem()->GetNsProcessManager();
                if (!m_pApplicationLaunchInfo)
                {
                    ns::ApplicationLaunchInfo launchInfo;
                    NN_RESULT_DO(pNs->GetApplicationLaunchInfo(&launchInfo, m_AppletIdentityInfo.applicationId));
                    this->m_pApplicationLaunchInfo = launchInfo;
                }
                uint8_t programIndex;
                NN_RESULT_DO(pNs->GetApplicationMainProgramIndex(&programIndex, *m_pApplicationLaunchInfo));
                NN_RESULT_DO(pNs->GetApplicationControlProperty(&this->m_ControlProperty, *m_pApplicationLaunchInfo, programIndex));
                m_SubProgramJumpInfo.ChangeNextProgramIndex(programIndex);
            }
            else
            {
                m_pApplicationLaunchInfo.emplace();
            }
            NN_RESULT_SUCCESS;
        }();
    }
    return *m_pApplicationPropertiesResult;
}

IntegratedApplication::IntegratedApplication(AppletSystem* pAppletSystem, os::ProcessId processId, const ns::ApplicationLaunchInfo& launchInfo) NN_NOEXCEPT
    : Derived(pAppletSystem)
    , m_pApplicationLaunchInfo(launchInfo)
    , m_pProcessIdForFloating(processId)
{
    // AppletIdentityInfo の構築
    this->m_AppletIdentityInfo = applet::AppletIdentityInfo::Make(launchInfo.id);
    EnsureApplicationProperties();

    auto programIndex = m_SubProgramJumpInfo.GetNext(m_AppletIdentityInfo.applicationId).second;
    m_SubProgramJumpInfo.OnExecuteProgram(programIndex);

    InitializeImplForConstructors();
}

IntegratedApplication::IntegratedApplication(AppletSystem* pAppletSystem, const applet::AppletIdentityInfo& appletIdentityInfo) NN_NOEXCEPT
    : Derived(pAppletSystem)
    , m_AppletIdentityInfo(appletIdentityInfo)
{
    InitializeImplForConstructors();
}

IntegratedApplication::IntegratedApplication(AppletSystem* pAppletSystem, ncm::ApplicationId applicationId) NN_NOEXCEPT
    : IntegratedApplication(pAppletSystem, applet::AppletIdentityInfo::Make(applicationId))
{
}

IntegratedApplication::~IntegratedApplication() NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, this, "~IntegratedApplication()");

    if (m_AppletIdentityInfo.IsApplication())
    {
        // 一時ストレージの Unmount 処理が完了していない可能性があるので最大 10 秒までリトライ
        for (int i = 0; i < 20; ++i)
        {
            auto result = fs::CleanUpTemporaryStorage();
            if (result.IsSuccess())
            {
                break;
            }
            else if (fs::ResultTargetLocked::Includes(result))
            {
                // retry
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }
    }
}

inline int IntegratedApplication::GetIndexByLaunchParameterKind(applet::LaunchParameterKind kind) NN_NOEXCEPT
{
    switch (kind)
    {
        case applet::LaunchParameterKind_User: return 0;
        case applet::LaunchParameterKind_Account: return 1;
        case applet::LaunchParameterKind_News: return 2;
        default: return -1;
    }
}

std::shared_ptr<StorageChannel> IntegratedApplication::GetInChannel(applet::LaunchParameterKind kind) NN_NOEXCEPT
{
    auto i = GetIndexByLaunchParameterKind(kind);
    if (!(i >= 0))
    {
        return nullptr;
    }
    return m_Channels[i];
}

Result IntegratedApplication::PushToInChannel(applet::LaunchParameterKind kind, sf::SharedPointer<IStorage> storage) NN_NOEXCEPT
{
    if (storage)
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        auto pChannel = this->GetInChannel(kind);
        NN_RESULT_THROW_UNLESS(pChannel, am::ResultInvalidCall());
        pChannel->Push(std::move(other));
    }
    NN_RESULT_SUCCESS;
}

applet::AppletIdentityInfo IntegratedApplication::GetAppletIdentityInfo() NN_NOEXCEPT
{
    return m_AppletIdentityInfo;
}

Result IntegratedApplication::CreateProcessImpl(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT
{
    switch (m_AppletIdentityInfo.appletId)
    {
        case applet::AppletId_Application:
        {
            auto p = m_SubProgramJumpInfo.GetNext(m_AppletIdentityInfo.applicationId);
            auto applicationId = p.first;
            auto programIndex = p.second;
            m_SubProgramJumpInfo.OnExecuteProgram(programIndex);
            return CreateApplicationProcessImpl(pOut, applicationId, programIndex);
        }
        case applet::AppletId_SystemApplication: return CreateSystemApplicationProcessImpl(pOut, m_AppletIdentityInfo.systemApplicationId);
        default: NN_UNEXPECTED_DEFAULT;
    }
}

void IntegratedApplication::BeforeActivateProxy(os::ProcessId processId) NN_NOEXCEPT
{
    if (m_AppletIdentityInfo.IsApplication())
    {
        if (!EnsureApplicationProperties().IsSuccess())
        {
            return;
        }
        NN_RESULT_ABORTING_BLOCK
        {
            arp::Registrar arpRegistrar;
            NN_RESULT_DO(arp::AcquireRegistrar(&arpRegistrar));
            NN_RESULT_DO(arpRegistrar.SetApplicationControlProperty(m_ControlProperty));
            NN_RESULT_DO(arpRegistrar.SetApplicationLaunchProperty(MakeArpApplicationLaunchProperty()));
            NN_RESULT_DO(arpRegistrar.Issue(processId));
            NN_RESULT_SUCCESS;
        };
    }
}

void IntegratedApplication::AfterDeactivateProxy(os::ProcessId processId) NN_NOEXCEPT
{
    if (m_AppletIdentityInfo.IsApplication())
    {
        if (!EnsureApplicationProperties().IsSuccess())
        {
            return;
        }
        arp::DeleteProperties(processId);
    }
}

std::shared_ptr<process::NsProcess> IntegratedApplication::MakeAttachedProcess(os::ProcessId processId) NN_NOEXCEPT
{
    switch (m_AppletIdentityInfo.appletId)
    {
        case applet::AppletId_Application: return GetAppletSystem()->GetNsProcessManager()->MakeApplicationProcess(processId);
        case applet::AppletId_SystemApplication: return GetAppletSystem()->GetNsProcessManager()->MakeLibraryAppletProcess(processId);
        default: NN_UNEXPECTED_DEFAULT;
    }
}

void IntegratedApplication::OnGetForeground() NN_NOEXCEPT
{
    if (GetHdcpPolicy() == ns::Hdcp::Required)
    {
        NN_AM_SERVICE_STUCK_CHECKED(omm_NotifyHdcpApplicationDrawingStarted, 60, omm::NotifyHdcpApplicationDrawingStarted());
    }
}

void IntegratedApplication::SendPlayLog( pdm::AppletEventType eventType ) NN_NOEXCEPT
{
    auto info = GetAppletIdentityInfo();
    ncm::ProgramId programId;
    programId.value = info.IsApplication() ? info.applicationId.value : info.systemApplicationId.value;
    NotifyAppletEvent( eventType,  programId, GetApplicationVersion(), info.appletId, GetLaunchedStorageId(), GetPlayLogPolicy() );
}

Result IntegratedApplication::TerminateAndSetResult(Result terminateResult) NN_NOEXCEPT
{
    am::service::SetTerminateResult(GetAppletIdentityInfo().applicationId, terminateResult, GetApplicationErrorReportInfo());
    this->TerminateWithResult( terminateResult );
    NN_RESULT_SUCCESS;
}

class IntegratedApplication::ApplicationAccessorImpl
    : public MainAppletAccessorImpl
{
private:

    IntegratedApplication* GetApplet() const NN_NOEXCEPT
    {
        return static_cast<IntegratedApplication*>(AppletAccessorImpl::GetApplet());
    }

public:

    explicit ApplicationAccessorImpl(std::shared_ptr<IntegratedApplication> p) NN_NOEXCEPT
        : MainAppletAccessorImpl(std::move(p))
    {
    }

    Result RequestForApplicationToGetForeground() NN_NOEXCEPT
    {
        return GetApplet()->MoveWindowToTop();
    }

    Result TerminateAllLibraryApplets() NN_NOEXCEPT
    {
        // Terminate 時に対象アプレットがキャプチャバッファを MapTransferMemory
        // していないことを保証するため、キャプチャバッファセマフォを取得する。
        // ResultCaptureBufferBusy 時は shim 側でリトライする。
        if (!am::service::TryAcquireCaptureBufferSemaphore())
        {
            return applet::ResultCaptureBufferBusy();
        }
        NN_UTIL_SCOPE_EXIT
        {
            am::service::ReleaseCaptureBufferSemaphore();
        };

        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "TerminateAllLibraryApplets()");
        this->GetApplet()->TerminateAllChildren();
        NN_RESULT_SUCCESS;
    }

    Result AreAnyLibraryAppletsLeft(sf::Out<bool> pOut) NN_NOEXCEPT
    {
        auto isLeft = this->GetApplet()->AreAnyChildrenLeft();
        NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "AreAnyLibraryAppletsLeft() = %s", (isLeft ? "true" : "false"));
        *pOut = isLeft;
        NN_RESULT_SUCCESS;
    }

    Result GetCurrentLibraryApplet(sf::Out<sf::SharedPointer<IAppletAccessor>> pOut) NN_NOEXCEPT
    {
        auto p = GetApplet()->GetLastLibraryApplet();
        if (!p)
        {
            *pOut = nullptr;
            NN_RESULT_SUCCESS;
        }
        else
        {
            *pOut = sf::CreateSharedObjectEmplaced<IAppletAccessor, AppletAccessorImpl>(std::move(p));
            NN_RESULT_SUCCESS;
        }
    }

    Result GetApplicationId(sf::Out<ncm::ApplicationId> pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(GetApplet()->m_AppletIdentityInfo.IsApplication(), am::ResultInvalidCall());
        *pOut = GetApplet()->m_AppletIdentityInfo.applicationId;
        NN_RESULT_SUCCESS;
    }

    Result PushLaunchParameter(Bit32 launchParameterKind, sf::SharedPointer<IStorage> storage) NN_NOEXCEPT
    {
        NN_RESULT_DO(GetApplet()->PushToInChannel(static_cast<applet::LaunchParameterKind>(launchParameterKind), std::move(storage)));
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s(%u)", NN_CURRENT_FUNCTION_NAME, launchParameterKind);
        NN_RESULT_SUCCESS;
    }

    Result GetApplicationControlProperty(const sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(buffer.GetSize() >= sizeof(ns::ApplicationControlProperty), am::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(!GetApplet()->GetAppletIdentityInfo().IsSystemApplication(), am::ResultApplicationHasNoProperty());

        NN_RESULT_THROW_UNLESS(GetApplet()->EnsureApplicationProperties().IsSuccess(), am::ResultApplicationPropertyNotReady());
        std::memcpy(buffer.GetPointerUnsafe(), &GetApplet()->m_ControlProperty, sizeof(ns::ApplicationControlProperty));
        NN_RESULT_SUCCESS;
    }

    Result GetApplicationLaunchProperty(const sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(buffer.GetSize() >= sizeof(arp::ApplicationLaunchProperty), am::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(!GetApplet()->GetAppletIdentityInfo().IsSystemApplication(), am::ResultApplicationHasNoProperty());

        NN_RESULT_THROW_UNLESS(GetApplet()->EnsureApplicationProperties().IsSuccess(), am::ResultApplicationPropertyNotReady());
        auto launchProperty = GetApplet()->MakeArpApplicationLaunchProperty();
        std::memcpy(buffer.GetPointerUnsafe(), &launchProperty, sizeof(arp::ApplicationLaunchProperty));
        NN_RESULT_SUCCESS;
    }

    Result GetResult() NN_NOEXCEPT
    {
        NN_RESULT_DO(GetApplet()->m_ApplicationResult);
        return MainAppletAccessorImpl::GetResult();
    }

    Result GetApplicationLaunchRequestInfo(sf::Out<applet::ApplicationLaunchRequestInfo> pOutInfo) NN_NOEXCEPT
    {
        *pOutInfo = GetApplet()->m_ApplicationLaunchRequestInfo;
        NN_RESULT_SUCCESS;
    }
};

void IntegratedApplication::SetApplicationLaunchRequestInfo(const applet::ApplicationLaunchRequestInfo& info) NN_NOEXCEPT
{
    this->m_ApplicationLaunchRequestInfo = info;
}

sf::SharedPointer<IApplicationAccessor> IntegratedApplication::CreateAccessor() NN_NOEXCEPT
{
    return sf::CreateSharedObjectEmplaced<IApplicationAccessor, ApplicationAccessorImpl>(SharedFromThis());
}

Result IntegratedApplication::GetDesiredLanguageImpl(settings::LanguageCode* pOut) NN_NOEXCEPT
{
    return am::service::GetDesiredLanguage(pOut, GetSupportedLanguageFlag());
}

class IntegratedApplication::ApplicationSelfProxyImpl
    : public MainAppletSelfProxyImpl
{
public:

    explicit ApplicationSelfProxyImpl(std::shared_ptr<IntegratedApplication> p, os::ProcessId processId) NN_NOEXCEPT
        : MainAppletSelfProxyImpl(p, processId)
    {
        this->m_IsNotifiedApplicationRunningToOverlayApplet = false;
        this->SetHandlesHomeButtonShortPressed(HomeButtonShortPressedHandleType_NotifyBlockingHomeButton);
        {
            this->m_LogoHandling = GetApplet()->GetLogoHandlingPolicy();
            GetApplet()->UpdateApplicationScreenShotPermission(GetApplet()->GetScreenShotPolicy() == ns::Screenshot::Allow);

            // NX Addon 3.x 系で "Allow" 指定のものは  Manual  相当
            // NX Addon 4.1 以前の環境だとデフォルトは Disable 相当
            // NX Addon 4.2 以降の環境だとデフォルトは Enabled 相当
            auto videoCapturePolicy = GetApplet()->GetVideoCapturePolicy();
            GetApplet()->SetVideoCaptureValue(videoCapturePolicy);
            if (videoCapturePolicy == ns::VideoCapture::Manual && IsInBlackListOfContinuousRecordingEarlyAdopters(GetApplet()->GetApplicationId()))
            {
                GetApplet()->SetGamePlayRecordingForbiddenByBlackList(true);
            }

            if (GetApplet()->GetHdcpPolicy() == ns::Hdcp::Required)
            {
                NN_AM_SERVICE_STUCK_CHECKED(omm_NotifyHdcpApplicationExecutionStarted, 60, omm::NotifyHdcpApplicationExecutionStarted());
            }
        }
        {
            RegisterRunningApplication(GetApplet()->GetApplicationErrorReportInfo());
            SetSystemReportApplicationId(GetApplet()->GetApplicationId());
            GetApplet()->CheckGameCardState();
        }
        if (m_LogoHandling == ns::LogoHandling::Auto)
        {
            NotifyApplicationRunningToOverlayApplet();
        }
        this->StartExitTimerForQuest();
    }

    ~ApplicationSelfProxyImpl() NN_NOEXCEPT
    {
        service::FinalizeApplicationCopyrightSharedBuffer();

        GetApplet()->UpdateApplicationScreenShotPermission(true);
        if (GetApplet()->GetHdcpPolicy() == ns::Hdcp::Required)
        {
            NN_AM_SERVICE_STUCK_CHECKED(omm_NotifyHdcpApplicationExecutionFinished, 60, omm::NotifyHdcpApplicationExecutionFinished());
        }
        this->StopExitTimerForQuest();

        UnregisterRunningApplication();
        ClearSystemReportApplicationId();

        if (m_pJitProcess)
        {
            m_pJitProcess->Terminate();
            m_pJitProcess->Join();
            m_pJitProcess.reset();
        }
    }

    Result PopLaunchParameter(sf::Out<sf::SharedPointer<IStorage>> pOut, Bit32 launchParameterKind) NN_NOEXCEPT
    {
        auto pChannel = GetApplet()->GetInChannel(static_cast<applet::LaunchParameterKind>(launchParameterKind));
        NN_RESULT_THROW_UNLESS(pChannel, am::ResultInvalidCall());
        auto ret = pChannel->Pop();
        NN_RESULT_THROW_UNLESS(ret, ResultNoStorage());
        *pOut = std::move(ret);
        NN_AM_SERVICE_APPLET_LOG(call, GetApplet(), "%s(%u)", NN_CURRENT_FUNCTION_NAME, launchParameterKind);
        NN_RESULT_SUCCESS;
    }

    Result CreateApplicationAndPushAndRequestToStartForQuest(ncm::ApplicationId applicationId, sf::SharedPointer<IStorage> storage, uint32_t playableTime, uint32_t idleDetectionTime) NN_NOEXCEPT
    {
        if (applicationId == ncm::ApplicationId::GetInvalidId())
        {
            NN_RESULT_THROW_UNLESS(GetApplet()->m_AppletIdentityInfo.IsApplication(), am::ResultInvalidCall());
            applicationId = GetApplet()->m_AppletIdentityInfo.applicationId;
        }

        // アプリ起動要求元の情報
        applet::ApplicationLaunchRequestInfo launchRequestInfo = {};
        launchRequestInfo.requestingAppletId     = applet::AppletId_Application;
        launchRequestInfo.requestingMainAppletId = applet::AppletId_Application;
        launchRequestInfo.applicationId          = GetApplet()->GetApplicationId();

        std::shared_ptr<IntegratedApplication> pApplication;
        NN_RESULT_DO(GetApplet()->GetAppletSystem()->CreateApplicationForReserved(&pApplication, applicationId, launchRequestInfo));
        NN_RESULT_DO(pApplication->PushToInChannel(applet::LaunchParameterKind_User, std::move(storage)));
        pApplication->SetExitTimerForQuest(playableTime, idleDetectionTime);
        GetApplet()->GetAppletSystem()->PushLaunchRequestedApplication(pApplication);
        NN_RESULT_SUCCESS;
    }

    Result CreateApplicationAndRequestToStartForQuest(ncm::ApplicationId applicationId, uint32_t playableTime, uint32_t idleDetectionTime) NN_NOEXCEPT
    {
        return CreateApplicationAndPushAndRequestToStartForQuest(applicationId, nullptr, playableTime, idleDetectionTime);
    }

    Result CreateApplicationAndPushAndRequestToStart(ncm::ApplicationId applicationId, sf::SharedPointer<IStorage> storage) NN_NOEXCEPT
    {
        return CreateApplicationAndPushAndRequestToStartForQuest(applicationId, std::move(storage), 0, 0);
    }

    Result CreateApplicationAndRequestToStart(ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        return CreateApplicationAndPushAndRequestToStartForQuest(applicationId, nullptr, 0, 0);
    }

    Result ExecuteProgram(ProgramSpecifyKind kind, Bit64 id) NN_NOEXCEPT
    {
        InvalidateDisplayLayer();
        switch (kind)
        {
            case ProgramSpecifyKind::SubProgramIndex:
            {
                NN_RESULT_THROW_UNLESS(util::IsIntValueRepresentable<uint8_t>(id), am::ResultInvalidCall());
                GetApplet()->m_SubProgramJumpInfo.ChangeNextProgramIndex(static_cast<uint8_t>(id));
                break;
            }
            case ProgramSpecifyKind::ApplicationId:
            {
                NN_RESULT_THROW_UNLESS(!am::service::IsProdMode(), am::ResultDevelopmentFunctionCalled());
                GetApplet()->m_SubProgramJumpInfo.ChangeNextApplicationId({id});
                break;
            }
            case ProgramSpecifyKind::Same:
            {
                // nop
                break;
            }
            default: NN_RESULT_THROW(am::ResultInvalidCall());
        }
        return GetApplet()->ExecuteNextProgram();
    }

    Result ClearUserChannel() NN_NOEXCEPT
    {
        GetApplet()->GetInChannel(applet::LaunchParameterKind_User)->Clear();
        NN_RESULT_SUCCESS;
    }

    Result UnpopToUserChannel(sf::SharedPointer<IStorage> storage) NN_NOEXCEPT
    {
        decltype(storage) other;
        NN_RESULT_DO(storage->GetAndInvalidate(&other));
        GetApplet()->GetInChannel(applet::LaunchParameterKind_User)->Unpop(std::move(other));
        NN_RESULT_SUCCESS;
    }

    Result GetPreviousProgramIndex(sf::Out<int32_t> pOut) NN_NOEXCEPT
    {
        *pOut = GetApplet()->m_SubProgramJumpInfo.GetPreviousProgramIndex();
        NN_RESULT_SUCCESS;
    }

    Result EnsureSaveData(sf::Out<int64_t> pOut, const account::Uid& uid) NN_NOEXCEPT
    {
        return am::service::EnsureSaveData(pOut.GetPointer(), GetApplet()->GetAppletIdentityInfo().applicationId, uid, GetApplet()->RefControlProperty());
    }

    Result CreateCacheStorage(sf::Out<int64_t> pOut, sf::Out<int32_t> pOutTargetMedia, uint16_t index, int64_t cacheStorageSize, int64_t cacheStorageJournalSize) NN_NOEXCEPT
    {
        return am::service::CreateCacheStorage(pOut.GetPointer(), pOutTargetMedia.GetPointer(), GetApplet()->GetAppletIdentityInfo().applicationId, index, cacheStorageSize, cacheStorageJournalSize, GetApplet()->RefControlProperty());
    }

    Result GetDesiredLanguage(sf::Out<settings::LanguageCode> pOut) NN_NOEXCEPT
    {
        return GetApplet()->GetDesiredLanguageImpl(pOut.GetPointer());
    }

    Result SetTerminateResult(uint32_t result) NN_NOEXCEPT
    {
        return am::service::SetTerminateResult(GetApplet()->GetAppletIdentityInfo().applicationId, result::detail::ConstructResult(result), GetApplet()->GetApplicationErrorReportInfo());
    }

    Result GetDisplayVersion(sf::Out<oe::DisplayVersion> pOut) NN_NOEXCEPT
    {
        return am::service::GetDisplayVersion(pOut.GetPointer(), GetApplet()->GetDisplayVersion());
    }

    Result GetLaunchStorageInfoForDebug(sf::Out<ncm::StorageId> pOutLaunchStorage, sf::Out<ncm::StorageId> pOutPatchStorage) NN_NOEXCEPT
    {
        *pOutLaunchStorage = GetApplet()->GetLaunchedStorageId();
        *pOutPatchStorage  = GetApplet()->GetPatchStorageId();
        NN_RESULT_SUCCESS;
    }

    Result ExtendSaveData(nn::sf::Out<std::int64_t> pOut, std::uint8_t saveDataType, const nn::account::Uid& uid, std::int64_t saveDataSize, std::int64_t saveDataJournalSize) NN_NOEXCEPT
    {
        return am::service::ExtendSaveData(pOut.GetPointer(), GetApplet()->GetAppletIdentityInfo().applicationId, saveDataType, uid, saveDataSize, saveDataJournalSize, GetApplet()->RefControlProperty());
    }

    Result GetSaveDataSize(nn::sf::Out<std::int64_t> pOutSize, nn::sf::Out<std::int64_t> pOutJournalSize, std::uint8_t saveDataType, const nn::account::Uid& uid) NN_NOEXCEPT
    {
        return am::service::GetSaveDataSize(pOutSize.GetPointer(), pOutJournalSize.GetPointer(), GetApplet()->GetAppletIdentityInfo().applicationId, saveDataType, uid);
    }

    Result BeginBlockingHomeButtonShortAndLongPressed(int64_t unused)
    {
        NN_UNUSED(unused);
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonShortPressed = true;
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonLongPressed = true;
        NN_RESULT_SUCCESS;
    }

    Result EndBlockingHomeButtonShortAndLongPressed() NN_NOEXCEPT
    {
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonShortPressed = false;
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonLongPressed = false;
        NN_RESULT_SUCCESS;
    }

    Result BeginBlockingHomeButton(int64_t longPressTimeInNanoSeconds) NN_NOEXCEPT
    {
        auto longPressTime = TimeSpan::FromNanoSeconds(longPressTimeInNanoSeconds);
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).homeButtonLongPressingTimeToSet = longPressTime;
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonShortPressed = true;
        NN_RESULT_SUCCESS;
    }

    Result EndBlockingHomeButton() NN_NOEXCEPT
    {
        auto updater = GetApplet()->GetAppletSystem()->GetWindowManager()->BeginUpdate();
        updater.RefWindowProperty(GetApplet()->GetWindow()).homeButtonLongPressingTimeToSet = 0;
        updater.RefWindowProperty(GetApplet()->GetWindow()).handlesHomeButtonShortPressed = false;
        NN_RESULT_SUCCESS;
    }

    Result NotifyRunning(sf::Out<bool> pOut) NN_NOEXCEPT
    {
        if (m_LogoHandling == ns::LogoHandling::Manual)
        {
            NotifyApplicationRunningToOverlayApplet();
            *pOut = true;
            NN_RESULT_SUCCESS;
        }
        else
        {
            *pOut = false;
            NN_RESULT_SUCCESS;
        }
    }

    Result GetPseudoDeviceId(sf::Out<util::Uuid> pOutId) NN_NOEXCEPT
    {
        return am::service::GetPseudoDeviceId(pOutId.GetPointer(), GetApplet()->GetSeedForPseudoDeviceId() );
    }

    std::shared_ptr<ApplicationSelfProxyImpl> GetApplicationFunctions() NN_NOEXCEPT
    {
        return std::static_pointer_cast<ApplicationSelfProxyImpl>(shared_from_this());
    }

    // アプリケーションからは遮断
    virtual Result PushToGeneralChannel(sf::SharedPointer<am::service::IStorage> pStorage) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(GetApplet()->m_AppletIdentityInfo.IsSystemApplication(), am::ResultInvalidAppletProxyCall());
        return AppletSelfProxyImpl::PushToGeneralChannel(std::move(pStorage));
    }

    virtual Result GetApplicationIdByContentActionName(sf::Out<Bit64> pOut, const sf::InArray<char>& name) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(GetApplet()->m_AppletIdentityInfo.IsSystemApplication(), am::ResultInvalidAppletProxyCall());
        return AppletSelfProxyImpl::GetApplicationIdByContentActionName(pOut, name);
    }

    // アプリの自発終了
    virtual Result Exit() NN_NOEXCEPT NN_OVERRIDE
    {
        return GetApplet()->TerminateAndSetResult( ResultApplicationExitedVoluntarily() );
    }

    // アプリの終了（試遊台環境専用、SA にメッセージ通知後に SA から終了）
    Result ExitAndGoBackQuestMenu() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(IsQuestMode(), ResultNotQuestEnvironment());
        GetApplet()->GetAppletSystem()->NotifyGoBackQuestMenu();
        NN_RESULT_SUCCESS;
    }

    // アプリ向けのメディア再生中設定（nn::applet 用）
    virtual Result SetMediaPlaybackState(bool isInMediaPlayback) NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard<os::Mutex> lk(m_MediaPlaybackStateMutex);
        m_IsInMediaPlaybackSectionForApplet = isInMediaPlayback;
        return AppletSelfProxyImpl::SetMediaPlaybackState( m_IsInMediaPlaybackSectionForApplet || m_IsInMediaPlaybackSectionForApplication );
    }

    // アプリ向けのメディア再生中設定（nn::oe 用）
    Result SetMediaPlaybackStateForApplication(bool isInMediaPlayback) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lk(m_MediaPlaybackStateMutex);
        m_IsInMediaPlaybackSectionForApplication = isInMediaPlayback;
        return AppletSelfProxyImpl::SetMediaPlaybackState( m_IsInMediaPlaybackSectionForApplet || m_IsInMediaPlaybackSectionForApplication );
    }

    Result RequestToShutdown() NN_NOEXCEPT
    {
        return GetApplet()->GetAppletSystem()->RequestToShutdown();
    }

    Result RequestToReboot() NN_NOEXCEPT
    {
        return GetApplet()->GetAppletSystem()->RequestToReboot();
    }

    // アプリでの例外発生時にクラッシュレポートを収集するかを設定（nn::oe 用）
    Result EnableApplicationCrashReport(bool isEnabled) NN_NOEXCEPT
    {
        NN_AM_SERVICE_STUCK_CHECKED(ns_EnableApplicationCrashReport, 30, ns::EnableApplicationCrashReport(isEnabled));
        NN_RESULT_SUCCESS;
    }

    // アプリクラッシュ時に全スレッドのログを出すか設定（nn::oe 用）
    Result EnableApplicationAllThreadDumpOnCrash(bool isEnabled) NN_NOEXCEPT
    {
        NN_AM_SERVICE_STUCK_CHECKED(ns_EnableApplicationAllThreadDumpOnCrash, 30, ns::EnableApplicationAllThreadDumpOnCrash(isEnabled));
        NN_RESULT_SUCCESS;
    }

    Result QueryApplicationPlayStatistics(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::pdm::ApplicationPlayStatistics>& outList, const nn::sf::InArray<nn::ncm::ApplicationId>& applicationIdList) NN_NOEXCEPT
    {
        return am::service::QueryApplicationPlayStatistics(
            outCount.GetPointer(), outList.GetData(), applicationIdList.GetData(), static_cast<int>(outList.GetLength()),
            GetApplet()->GetPlayLogQueryPolicy(), GetApplet()->GetApplicationId());
    }

private:
    os::Mutex m_MediaPlaybackStateMutex{false};
    bool m_IsInMediaPlaybackSectionForApplet{false};
    bool m_IsInMediaPlaybackSectionForApplication{false};

    IntegratedApplication* GetApplet() NN_NOEXCEPT
    {
        return static_cast<IntegratedApplication*>(AppletSelfProxyImpl::GetApplet());
    }

    ns::LogoHandling m_LogoHandling;
    std::atomic_bool m_IsNotifiedApplicationRunningToOverlayApplet;

public:
    void NotifyApplicationRunningToOverlayApplet() NN_NOEXCEPT
    {
        if (GetApplet()->m_AppletIdentityInfo.IsApplication())
        {
            if (!m_IsNotifiedApplicationRunningToOverlayApplet.exchange(true))
            {
                GetApplet()->GetAppletSystem()->NotifyApplicationRunning(GetApplet()->m_AppletIdentityInfo.applicationId);
            }
        }
    }

private:
    void StartExitTimerForQuest() NN_NOEXCEPT
    {
        // アプリプレイタイマーの制御（試遊台向け）
        auto playableTime = GetApplet()->GetPlayableTimeForQuest();
        if (playableTime > 0)
        {
            GetApplicationTimer().StartPlayableTimer( TimeSpan::FromSeconds(playableTime) );
            NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Apply application playable timer=%u(sec)\n", playableTime);
        }
        // アプリ無操作時間検出の制御（試遊台向け）
        auto idleDetectionTime = GetApplet()->GetIdleDetectionTimeForQuest();
        if (idleDetectionTime > 0)
        {
            GetApplicationTimer().StartIdleDetectionTimer( TimeSpan::FromSeconds(idleDetectionTime) );
            NN_AM_SERVICE_APPLET_LOG(seq, GetApplet(), "Apply application idle detection timer=%u(sec)\n", idleDetectionTime);
        }
    }

    void StopExitTimerForQuest() NN_NOEXCEPT
    {
        GetApplicationTimer().StopPlayableTimer();
        GetApplicationTimer().StopIdleDetectionTimer();
    }

public:

    Result CreateMovieMaker(sf::Out<sf::SharedPointer<IMovieMaker>> pOut, sf::NativeHandle transferMemoryHandle, uint64_t transferMemorySize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(util::IsIntValueRepresentable<size_t>(transferMemorySize), am::ResultInvalidCall());
        applet::AppletResourceUserId aruid;
        NN_RESULT_DO(GetAppletResourceUserId(&aruid));
        class Mixin
            : public MovieMakerMixin
        {
        private:
            std::shared_ptr<ApplicationSelfProxyImpl> m_pParent;
        public:
            void SetParent(std::shared_ptr<ApplicationSelfProxyImpl> pParent) NN_NOEXCEPT
            {
                this->m_pParent = std::move(pParent);
            }
            ~Mixin() NN_NOEXCEPT
            {
                if (m_pParent)
                {
                    m_pParent->OnGamePlayRecordingStateChangedImpl();
                }
            };
        };
        auto pMixin = service::MakeShared<Mixin>();
        sf::SharedPointer<IMovieMaker> p;
        NN_RESULT_DO(service::CreateMovieMaker(&p, aruid, GetApplet()->GetApplicationId(), std::move(transferMemoryHandle), static_cast<size_t>(transferMemorySize), pMixin));
        this->OnGamePlayRecordingStateChangedImpl();
        pMixin->SetParent(std::static_pointer_cast<ApplicationSelfProxyImpl>(this->shared_from_this()));
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }

    // 権利表記
    Result InitializeApplicationCopyrightFrameBuffer(sf::NativeHandle sfHandle, uint64_t size, int width, int height) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(size % (256u * 1024u) == 0, ResultInvalidParameter());
        NN_RESULT_THROW_UNLESS(width > 0 && width <= ApplicationCopyrightTextureWidthMax, ResultInvalidParameter());
        NN_RESULT_THROW_UNLESS(height > 0 && height <= ApplicationCopyrightTextureHeightMax, ResultInvalidParameter());

        return service::InitializeApplicationCopyrightSharedBuffer( sfHandle, static_cast<size_t>(size), width, height );
    }

    Result SetApplicationCopyrightImage(sf::InBuffer buffer, int x, int y, int width, int height, int originMode)
    {
        NN_RESULT_THROW_UNLESS(x >= 0 && y >= 0 && width > 0 && height > 0, ResultInvalidParameter());

        NN_RESULT_DO( service::SetApplicationCopyrightTexture(x, y, width, height, buffer.GetPointerUnsafe(), buffer.GetSize(), originMode) );
        NN_RESULT_SUCCESS;
    }

    Result SetApplicationCopyrightVisibility(bool isVisible) NN_NOEXCEPT
    {
        service::SetApplicationCopyrightVisibility(isVisible);
        NN_RESULT_SUCCESS;
    }

private:

    std::shared_ptr<process::NsProcess> m_pJitProcess;

public:

    Result PrepareForJit() NN_NOEXCEPT
    {
        if (m_pJitProcess)
        {
            NN_RESULT_SUCCESS;
        }
        decltype(m_pJitProcess) pJitProcess;
        NN_RESULT_DO(GetApplet()->CreateSystemApplicationProcessImpl(&pJitProcess, {0x010000000000003B}));
        this->m_pJitProcess = std::move(pJitProcess);
        NN_RESULT_SUCCESS;
    }

};

void IntegratedApplication::OnLostForeground() NN_NOEXCEPT
{
    if (GetHdcpPolicy() == ns::Hdcp::Required)
    {
        NN_AM_SERVICE_STUCK_CHECKED(omm_NotifyHdcpApplicationDrawingFinished, 60, omm::NotifyHdcpApplicationDrawingFinished());
    }

    auto pSelfProxy = GetSelfProxy();
    if (pSelfProxy)
    {
        // SIGLO-71052:
        //  InFocus 状態でなくなった場合は起動ロゴ制御が Manual の場合でも
        //  規定時間が経過したら起動ロゴを消去するように OA に通知する
        static_cast<IntegratedApplication::ApplicationSelfProxyImpl*>(pSelfProxy.get())->NotifyApplicationRunningToOverlayApplet();
    }
}

bool IntegratedApplication::IsScreenShotEnabled() NN_NOEXCEPT
{
    return GetAppletSystem()->IsApplicationScreenShotEnabled();
}

void IntegratedApplication::UpdateApplicationScreenShotPermission(bool isPermitted) NN_NOEXCEPT
{
    GetAppletSystem()->SetApplicationScreenShotPermission(isPermitted);
    this->OnGamePlayRecordingStateChanged();
}

Result IntegratedApplication::SetScreenShotPermissionImpl(applet::ScreenShotPermission permission) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(seq, this, "SetScreenShotPermission(meta=%s, %s)", GetScreenShotPolicy() == ns::Screenshot::Allow ? "Allow" : "Deny", permission == applet::ScreenShotPermission_Forbid ? "false" : "true");
    NN_RESULT_THROW_UNLESS(GetScreenShotPolicy() == ns::Screenshot::Allow, am::ResultLackOfCapability());

    switch (permission)
    {
    case applet::ScreenShotPermission_Permit:
            this->UpdateApplicationScreenShotPermission(true);
            break;

    case applet::ScreenShotPermission_Forbid:
            this->UpdateApplicationScreenShotPermission(false);
            break;

    case applet::ScreenShotPermission_Inherit:
            this->UpdateApplicationScreenShotPermission(true);
            break;

    default:
            NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

IntegratedApplication::AppletProxyInfo IntegratedApplication::DoCreateAppletProxy(os::ProcessId processId) NN_NOEXCEPT
{
    return MakeAppletProxyInfo<ApplicationProxy>(MakeShared<ApplicationSelfProxyImpl>(SharedFromThis(), processId));
}

namespace {

os::SdkMutexType g_AccountDeclarationLock = NN_OS_SDK_MUTEX_INITIALIZER();

class AccountDeclaration
{
public:
    AccountDeclaration() NN_NOEXCEPT = default;
    AccountDeclaration(os::SdkMutexType& mutex, ncm::ApplicationId applicationId) NN_NOEXCEPT
        : m_Lock(mutex)
        , m_AccountDeclaration(account::EnableInterprogramOpenUserRetention({applicationId.value}))
    {
    }
private:
    std::unique_lock<os::SdkMutexType> m_Lock;
    account::Declaration m_AccountDeclaration;
};
AccountDeclaration g_AccountDeclaration;

}

Result IntegratedApplication::Before() NN_NOEXCEPT
{
    NN_RESULT_DO(EnsureApplicationProperties());
    NN_RESULT_DO( Derived::Before() );
    if (m_AppletIdentityInfo.IsApplication())
    {
        GetAppletSystem()->NotifyApplicationStarted(m_AppletIdentityInfo.applicationId);
        g_AccountDeclaration = AccountDeclaration{g_AccountDeclarationLock, m_AppletIdentityInfo.applicationId};
    }
    NN_RESULT_SUCCESS;
}

void IntegratedApplication::Cleanup() NN_NOEXCEPT
{
    g_AccountDeclaration = {};

    if (m_AppletIdentityInfo.IsApplication())
    {

        // 先行で削除試行
        (void)fs::CleanUpTemporaryStorage();

        GetAppletSystem()->NotifyApplicationExited(m_AppletIdentityInfo.applicationId);
    }

    // SystemApplet への通知
    GetAppletSystem()->GetApplicationObserver()->NotifyApplicationExit();
    if (!IsGameCardStateValidForThis())
    {
        this->m_ApplicationResult = ResultAppletTerminatedByGameCardLost();
    }
    Derived::Cleanup();
}

bool IntegratedApplication::IsGameCardStateValidForThis() NN_NOEXCEPT
{
    switch (GetLaunchedStorageId())
    {
        case ncm::StorageId::Card:
        {
            ns::GameCardAttachmentInfo info;
            ns::GetGameCardAttachmentInfo(&info);
            std::lock_guard<decltype(m_GameCardSequenceMutex)> lk(m_GameCardSequenceMutex);
            return m_InitialGameCardSequenceNumber == info.sequence && info.isAttached;
        }
        default:
        {
            return true;
        }
    }
}

void IntegratedApplication::CheckGameCardState() NN_NOEXCEPT
{
    if (!IsGameCardStateValidForThis())
    {
        NN_AM_SERVICE_APPLET_LOG(error, this, "application terminated because of game card ejection");
        this->KillProcess();
        this->RequestExit();
    }
}

void IntegratedApplication::OnGameCardStateChanged() NN_NOEXCEPT
{
    CheckGameCardState();
}

Result IntegratedApplication::SubmitApplicationInfo() NN_NOEXCEPT
{
    auto identityInfo = GetAppletIdentityInfo();
    if( identityInfo.IsApplication() )
    {
        return am::service::SubmitApplicationInfo(GetApplicationErrorReportInfo());
    }
    else if( identityInfo.IsSystemApplication() )
    {
        return am::service::SubmitApplicationInfo(identityInfo.systemApplicationId.value);
    }
    else
    {
        NN_RESULT_THROW(am::ResultNotSupported());
    }
}

std::pair<ncm::ApplicationId, uint8_t> IntegratedApplication::SubProgramJumpInfo::GetNext(ncm::ApplicationId defaultApplicationId) NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    auto applicationId = m_pNextApplicationId.value_or(defaultApplicationId);
    NN_ABORT_UNLESS(m_pNextProgramIndex);
    return {applicationId, *m_pNextProgramIndex};
}

void IntegratedApplication::SubProgramJumpInfo::OnExecuteProgram(uint8_t newProgramIndex) NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    this->m_PreviousProgramIndex = m_CurrentProgramIndex;
    this->m_CurrentProgramIndex = newProgramIndex;
}

int32_t IntegratedApplication::SubProgramJumpInfo::GetCurrentProgramIndex() const NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    return m_CurrentProgramIndex;
}

int32_t IntegratedApplication::SubProgramJumpInfo::GetPreviousProgramIndex() const NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    return m_PreviousProgramIndex;
}

void IntegratedApplication::SubProgramJumpInfo::ChangeNextApplicationId(ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    this->m_pNextApplicationId = applicationId;
    this->m_pNextProgramIndex = {};
}

void IntegratedApplication::SubProgramJumpInfo::ChangeNextProgramIndex(uint8_t programIndex) NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    this->m_pNextProgramIndex = programIndex;
}

Result IntegratedApplication::ExecuteNextProgram() NN_NOEXCEPT
{
    ReserveToWindAndUnwind();
    UnlockExit(); // see SIGLO-39618
    KillProcess();
    NN_RESULT_SUCCESS;
}

void IntegratedApplication::OnRightsActivated(os::Tick expirationTick) NN_NOEXCEPT
{
    // TODO
    NN_UNUSED(expirationTick);
}

void IntegratedApplication::OnRightsDeactivated() NN_NOEXCEPT
{
    GetAppletSystem()->NotifyApplicationSuspendedByRightsError();
}

void IntegratedApplication::OnRightsAvailableTimeChanged(os::Tick expirationTick) NN_NOEXCEPT
{
    // TODO: 残り時間によって延長要求・OA へ通知
    // - 残り時間を時間帯ごとに抽象化
    // - 時間帯におけるエッジを抽出
    NN_UNUSED(expirationTick);
}

}}}
