﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "am_GrcControl.h"

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/am/am_Result.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/grc/grc_Api.h>
#include <nn/grc/grc_Result.h>
#include <utility>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_Exchange.h>
#include <nn/os/os_SdkMutex.h>
#include <mutex>

namespace nn { namespace am { namespace service {

namespace {

os::SdkMutex g_Mutex;
bool g_IsGrcAvailable = true;

using GrcMovieMaker = grcsrv::IMovieMaker;

class MovieMakerImpl
    : public sf::ISharedObject
{
private:

    bool m_Initialized = false;
    std::shared_ptr<MovieMakerMixin> m_pMovieMakerMixin;
    GrcOccupier m_GrcOccupier;
    sf::SharedPointer<grcsrv::IOffscreenRecorder> m_pOffScreenRecorder;
    sf::SharedPointer<GrcMovieMaker> m_pGrcMovieMaker;
    uint64_t m_LayerHandle;

    class GrcMovieMakerImpl
    {
        NN_SF_MARK_THIS_AS_SMART_POINTER;
    private:

        sf::SharedPointer<MovieMakerImpl> m_pParent;

    public:

        NN_IMPLICIT GrcMovieMakerImpl(MovieMakerImpl* pParent) NN_NOEXCEPT
            : m_pParent{pParent, true}
        {
        }

        GrcMovieMaker* operator->() const NN_NOEXCEPT
        {
            return m_pParent->m_pGrcMovieMaker.operator->();
        }

    };

public:

    Result Initialize(GrcOccupier grcOccupier, applet::AppletResourceUserId aruid, ncm::ApplicationId applicationId, sf::NativeHandle transferMemoryHandle, size_t transferMemorySize, std::shared_ptr<MovieMakerMixin> pMovieMakerMixin) NN_NOEXCEPT
    {
        auto success = false;

        decltype(m_pOffScreenRecorder) pOffScreenRecorder;
        NN_RESULT_DO(grc::CreateOffscreenRecorder(&pOffScreenRecorder, std::move(transferMemoryHandle), transferMemorySize));

        decltype(m_LayerHandle) layerHandle;
        NN_RESULT_DO(pOffScreenRecorder->CreateOffscreenLayer(&layerHandle, aruid));
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(pOffScreenRecorder->DestroyOffscreenLayer(layerHandle));
            }
        };

        decltype(m_pGrcMovieMaker) pMovieMaker;
        NN_RESULT_DO(grc::CreateMovieMakerForApplication(&pMovieMaker, aruid, applicationId));

        success = true;
        this->m_GrcOccupier = std::move(grcOccupier);
        this->m_pMovieMakerMixin = std::move(pMovieMakerMixin);
        this->m_pOffScreenRecorder = std::move(pOffScreenRecorder);
        this->m_pGrcMovieMaker = std::move(pMovieMaker);
        this->m_LayerHandle = layerHandle;
        this->m_Initialized = true;
        NN_RESULT_SUCCESS;
    }

    ~MovieMakerImpl() NN_NOEXCEPT
    {
        if (!m_Initialized)
        {
            return;
        }
        m_pOffScreenRecorder->DestroyOffscreenLayer(m_LayerHandle);
    }

    Result GetGrcMovieMaker(sf::Out<sf::SharedPointer<GrcMovieMaker>> pOut) NN_NOEXCEPT
    {
        *pOut = sf::CreateSharedObject<GrcMovieMaker>(GrcMovieMakerImpl{this});
        NN_RESULT_SUCCESS;
    }

    Result GetLayerHandle(sf::Out<uint64_t> pOut) NN_NOEXCEPT
    {
        *pOut = m_LayerHandle;
        NN_RESULT_SUCCESS;
    }

};

}

void GrcOccupier::Occupy() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_HasOccupation);
    std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
    this->m_HasOccupation = std::exchange(g_IsGrcAvailable, false);
}

void GrcOccupier::Unoccupy() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_HasOccupation);
    std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
    g_IsGrcAvailable = std::exchange(this->m_HasOccupation, false);
}

bool IsGrcAvailable() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
    return g_IsGrcAvailable;
}

Result CreateMovieMaker(sf::Out<sf::SharedPointer<IMovieMaker>> pOut, applet::AppletResourceUserId aruid, ncm::ApplicationId applicationId, sf::NativeHandle transferMemoryHandle, size_t transferMemorySize, std::shared_ptr<MovieMakerMixin> pMovieMakerMixin) NN_NOEXCEPT
{
    GrcOccupier grcOccupier{true};
    NN_RESULT_THROW_UNLESS(grcOccupier.HasOccupation(), grc::ResultGrcBusy());
    auto p = sf::CreateSharedObjectEmplaced<IMovieMaker, MovieMakerImpl>();
    NN_RESULT_DO(p.GetImpl().Initialize(std::move(grcOccupier), aruid, applicationId, std::move(transferMemoryHandle), transferMemorySize, std::move(pMovieMakerMixin)));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

namespace {

class ManagedGameMovieTrimmer
{
    NN_SF_MARK_THIS_AS_SMART_POINTER;
private:

    GrcOccupier m_GrcOccupier;
    sf::SharedPointer<grcsrv::IGameMovieTrimmer> m_pParent;

public:

    NN_IMPLICIT ManagedGameMovieTrimmer(sf::SharedPointer<grcsrv::IGameMovieTrimmer> pParent) NN_NOEXCEPT
        : m_pParent{std::move(pParent)}
    {
    }

    void SetGrcOccupier(GrcOccupier grcOccupier) NN_NOEXCEPT
    {
        this->m_GrcOccupier = std::move(grcOccupier);
    }

    grcsrv::IGameMovieTrimmer* operator->() const NN_NOEXCEPT
    {
        return m_pParent.Get();
    }

};

}

Result CreateGameMovieTrimmer(sf::Out<sf::SharedPointer<grcsrv::IGameMovieTrimmer>> pOut, GrcOccupier* pGrcOccupier, sf::NativeHandle transferMemoryHandle, uint64_t transferMemorySize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pGrcOccupier->HasOccupation(), am::ResultInvalidCall());
    sf::SharedPointer<grcsrv::IGameMovieTrimmer> p;
    NN_RESULT_DO(grc::CreateGameMovieTrimmer(&p, std::move(transferMemoryHandle), transferMemorySize));
    auto ret = sf::CreateSharedObjectEmplaced<grcsrv::IGameMovieTrimmer, ManagedGameMovieTrimmer>(std::move(p));
    NN_RESULT_THROW_UNLESS(ret, am::ResultOutOfServiceMemory());
    ret.GetImpl().SetGrcOccupier(std::move(*pGrcOccupier));
    *pOut = std::move(ret);
    NN_RESULT_SUCCESS;
}

}}}
