﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_GpuErrorControl.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/am/am_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <mutex>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/am/service/am_AppletSystem.h>
#include <nn/am/service/am_StuckChecker.h>

#include <nv_GemControl.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace nn { namespace am { namespace service {

namespace {

    inline Result ConvertToAmResult(::nv::gem::Result gemResult) NN_NOEXCEPT
    {
        switch (gemResult)
        {
            case ::nv::gem::Result_Success:
            {
                NN_RESULT_SUCCESS;
            }
            default:
            {
                NN_RESULT_THROW( am::ResultNvGemResultUnknown() );
            }
        }
    }

    #define ABORT_UNLESS_GEM_SUCCESS(r) NN_ABORT_UNLESS_RESULT_SUCCESS( ConvertToAmResult(r) )

    class GpuErrorController
    {
    private:

        bool m_IsHandlingEnabled{false};

        nv::gemcontrol::Client m_GemControlClient;
        os::SystemEventType m_Event;

        struct Entry
        {
            applet::AppletResourceUserId aruid = {};
        };

        os::Mutex m_Mutex{false};
        Entry m_Entries[16];

        std::shared_ptr<process::NsProcess> m_pCoreDumperProcess;
        applet::AppletResourceUserId m_AruidToBeDump;

        static std::shared_ptr<process::NsProcess> LaunchGpuCoreDumperImpl(AppletSystem* pAppletSystem) NN_NOEXCEPT
        {
            // TORIAEZU: ns::LaunchLibraryApplet で起動
            std::shared_ptr<process::NsProcess> pProcess;
            auto result = pAppletSystem->GetNsProcessManager()->MakeLibraryAppletProcess(&pProcess, {0x010000000000209B});
            if (!result.IsSuccess())
            {
                NN_AM_SERVICE_LOG(error, "failed to launch GpuCoreDumper: result = %08x\n", result.GetInnerValueForDebug());
                return nullptr;
            }
            return pProcess;
        }

        void OnEventImpl(AppletSystem* pAppletSystem) NN_NOEXCEPT
        {
            uint64_t aruid64;
            bool hasCoreDump;
            auto result = NN_AM_SERVICE_STUCK_CHECKED(gpu_error, 60, m_GemControlClient.GetAruid(&aruid64, &hasCoreDump));
            ABORT_UNLESS_GEM_SUCCESS(result);
            if (aruid64 == 0)
            {
                return;
            }
            NN_SDK_ASSERT(!m_pCoreDumperProcess);
            NN_UTIL_SCOPE_EXIT
            {
                if (!m_pCoreDumperProcess)
                {
                    NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_error, 60);
                    ABORT_UNLESS_GEM_SUCCESS(m_GemControlClient.Reset());
                }
            };
            applet::AppletResourceUserId aruid = { aruid64 };
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            for (auto&& e: m_Entries)
            {
                if (e.aruid == aruid)
                {
                    NN_AM_SERVICE_LOG(error, "GPU error detected: ARUID = %llu\n", aruid.lower);
                    if (hasCoreDump && DumpsApplicationGpuCoreDumpOnGpuError())
                    {
                        auto pCoreDumperProcess = LaunchGpuCoreDumperImpl(pAppletSystem);
                        if (pCoreDumperProcess)
                        {
                            // プロセスの完了検知を同一スレッドで行っているため、
                            // ダンププロセスの完了をここで行うとデッドロックする。
                            // ShellEvent の通知時に OnMaybeProcessTerminated() を呼び、
                            // m_pCoreDumperProcess の TryWait を行い、Join, GPU reset, プロセスの強制終了を行う。
                            this->m_pCoreDumperProcess = std::move(pCoreDumperProcess);
                            this->m_AruidToBeDump = aruid;
                            return;
                        }
                    }
                    if (KillsApplicationOnGpuError())
                    {
                        TerminateApplicationProcessImpl({aruid.lower});
                    }
                    return;
                }
            }
        }

        void TerminateApplicationProcessImpl(os::ProcessId processId) NN_NOEXCEPT
        {
            NN_AM_SERVICE_LOG(error, "kill process %llu\n", processId.value);
            auto result = NN_AM_SERVICE_STUCK_CHECKED(ns_TerminateApplication, 60, ns::TerminateApplication(processId));
            NN_UNUSED(result);
        }

    public:

        void Initialize() NN_NOEXCEPT
        {
            NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_error, 60);
            this->m_IsHandlingEnabled = KillsApplicationOnGpuError() || DumpsApplicationGpuCoreDumpOnGpuError(); // TORIAEZU
            if (m_IsHandlingEnabled)
            {
                ABORT_UNLESS_GEM_SUCCESS(m_GemControlClient.Initialize());
                ABORT_UNLESS_GEM_SUCCESS(m_GemControlClient.ControlNotification(true));
                ABORT_UNLESS_GEM_SUCCESS(m_GemControlClient.GetEventHandle(&m_Event));
            }
            else
            {
                os::CreateSystemEvent(&m_Event, os::EventClearMode_ManualClear, false);
            }
        }

        os::SystemEventType* GetEvent() NN_NOEXCEPT
        {
            return &m_Event;
        }

        void OnEvent(AppletSystem* pAppletSystem) NN_NOEXCEPT
        {
            if (!m_IsHandlingEnabled)
            {
                return;
            }
            OnEventImpl(pAppletSystem);
        }

        void OnMaybeProcessTerminated() NN_NOEXCEPT
        {
            if (m_pCoreDumperProcess && os::TryWaitAny(m_pCoreDumperProcess.get()) == 0)
            {
                m_pCoreDumperProcess->Join();
                this->m_pCoreDumperProcess.reset();
                if (KillsApplicationOnGpuError())
                {
                    TerminateApplicationProcessImpl({m_AruidToBeDump.lower});
                    this->m_AruidToBeDump = {};
                }
                NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_error, 60);
                ABORT_UNLESS_GEM_SUCCESS(m_GemControlClient.Reset());
            }
        }

        void Register(applet::AppletResourceUserId aruid) NN_NOEXCEPT
        {
            NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_error, 60);
            NN_SDK_ASSERT(aruid != applet::AppletResourceUserId::GetInvalidId());
            if (!m_IsHandlingEnabled)
            {
                return;
            }
            std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
            for (auto&& e: m_Entries)
            {
                if (e.aruid == applet::AppletResourceUserId::GetInvalidId())
                {
                    e.aruid = aruid;
                    lk.unlock();
                    ABORT_UNLESS_GEM_SUCCESS(m_GemControlClient.SetNotificationPerm(aruid.lower, true));
                    if (DumpsApplicationGpuCoreDumpOnGpuError())
                    {
                        ABORT_UNLESS_GEM_SUCCESS(m_GemControlClient.SetCoreDumpPerm(aruid.lower, true));
                    }
                    return;
                }
            }
            NN_AM_SERVICE_LOG(error, "warning: GpuErrorController::Register failed: no entry\n");
        }

        void Unregister(applet::AppletResourceUserId aruid) NN_NOEXCEPT
        {
            NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_error, 60);
            NN_SDK_ASSERT(aruid != applet::AppletResourceUserId::GetInvalidId());
            if (!m_IsHandlingEnabled)
            {
                return;
            }
            std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
            for (auto&& e: m_Entries)
            {
                if (e.aruid == aruid)
                {
                    e.aruid = applet::AppletResourceUserId::GetInvalidId();
                    lk.unlock();
                    if (DumpsApplicationGpuCoreDumpOnGpuError())
                    {
                        ABORT_UNLESS_GEM_SUCCESS(m_GemControlClient.SetCoreDumpPerm(aruid.lower, false));
                    }
                    ABORT_UNLESS_GEM_SUCCESS(m_GemControlClient.SetNotificationPerm(aruid.lower, false));
                    return;
                }
            }
            NN_AM_SERVICE_LOG(error, "warning: GpuErrorController::Unregister failed: unknown ARUID\n");
        }

    };

    GpuErrorController g_GpuErrorController;

}

void InitializeGpuErrorControl() NN_NOEXCEPT
{
    g_GpuErrorController.Initialize();
}

void RegisterToGpuErrorControl(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    g_GpuErrorController.Register(aruid);
}

void UnregisterToGpuErrorControl(applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    g_GpuErrorController.Unregister(aruid);
}

GpuErrorEvent::GpuErrorEvent() NN_NOEXCEPT
{
    os::InitializeMultiWaitHolder(&this->m_Holder,g_GpuErrorController.GetEvent());
    this->m_Holder.userData = reinterpret_cast<uintptr_t>(this);
}

GpuErrorEvent::~GpuErrorEvent() NN_NOEXCEPT
{
    os::FinalizeMultiWaitHolder(&m_Holder);
}

void GpuErrorEvent::Handle(AppletSystem* pAppletSystem) NN_NOEXCEPT
{
    g_GpuErrorController.OnEvent(pAppletSystem);
    os::ClearSystemEvent(g_GpuErrorController.GetEvent());
}

void GpuErrorEvent::OnMaybeProcessTerminated() NN_NOEXCEPT
{
    g_GpuErrorController.OnMaybeProcessTerminated();
}

}}}
