﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_ApplicationFunctions.h>

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/am/service/am_ErrorReport.h>
#include <nn/am/service/am_IntegratedApplet.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/applet/applet_Types.h>
#include <nn/erpt.h>
#include <nn/err/err_SystemApi.h>
#include <nn/fs/fs_ErrorReport.h>
#include <nn/ns/ns_TerminateResultApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <cstring>
#include <movie/private/MultimediaTelemetryReport.h>

namespace nn { namespace am { namespace service {

namespace {

    os::Mutex g_RunningAppletContextMutex(false);
    const int RunningAppletCountMax = 10;
    int g_RunningAppletCount = 0;
    Bit64 g_RunningApplet[RunningAppletCountMax] = {};

    const int ForegroundAppletHistoryCountMax = 10;
    Bit64 g_ForegroundAppletHistory[ForegroundAppletHistoryCountMax] = {};

    Bit8 g_ErrorContextBuffer[1024];

    // エラー履歴での表示優先度。呼び出しスタック上で最も優先度が高いものをエラー履歴向けに保存する（同じ優先度であれば子供が優先）。
    int GetErrorHistoryDisplayPriority(applet::AppletId appletId) NN_NOEXCEPT
    {
        switch( appletId )
        {
        case applet::AppletId::AppletId_Application:
        case applet::AppletId::AppletId_SystemApplication:
            return 3;
        case applet::AppletId::AppletId_LibraryAppletShop:
        case applet::AppletId::AppletId_LibraryAppletPhotoViewer:
        case applet::AppletId::AppletId_LibraryAppletController:
        case applet::AppletId::AppletId_LibraryAppletMyPage:
            return 2;
        case applet::AppletId::AppletId_SystemAppletMenu:
            return 1;
        default:
            return 0;
        }
    }

    am::service::IntegratedApplet* SelectApplicationInfoReportApplet(am::service::IntegratedApplet* pCallerApplet) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pCallerApplet);

        am::service::IntegratedApplet* pReportApplet = pCallerApplet;
        int reportAppletPriority = GetErrorHistoryDisplayPriority(pReportApplet->GetAppletIdentityInfo().appletId);

        for( auto pApplet = pReportApplet->GetParent(); pApplet != nullptr; pApplet = pApplet->GetParent() )
        {
            auto p = GetErrorHistoryDisplayPriority(pApplet->GetAppletIdentityInfo().appletId);
            if( p > reportAppletPriority )
            {
                pReportApplet = pApplet;
                reportAppletPriority = p;
            }
        }
        return pReportApplet;
    }

    const char* GetStorageIdStr(const ncm::StorageId& storageId) NN_NOEXCEPT
    {
        switch( storageId )
        {
        case ncm::StorageId::BuildInSystem:
            return "BuildInSystem";
        case ncm::StorageId::BuildInUser:
            return "BuildInUser";
        case ncm::StorageId::Card:
            return "Card";
        case ncm::StorageId::Host:
            return "Host";
        case ncm::StorageId::None:
            return "None";
        case ncm::StorageId::SdCard:
            return "SdCard";
        default:
            return "(unknown)";
        }
    }

    Result AddApplicationIdField(erpt::Context* pContext, Bit64 id) NN_NOEXCEPT
    {
        char applicationIdString[17];
        const size_t ApplicationIdStringSize = sizeof(applicationIdString);
        util::SNPrintf(applicationIdString, ApplicationIdStringSize, "%016llx", id);
        return pContext->Add(erpt::FieldId::ApplicationID, applicationIdString, ApplicationIdStringSize);
    }

    Result SubmitRunningApplicationInfo(const ApplicationErrorReportInfo& info) NN_NOEXCEPT
    {
        auto& controlProperty = info.controlProperty;
        auto& launchProperty = info.launchProperty;

        erpt::Context context(erpt::RunningApplicationInfo);

        char applicationIdString[17];
        const size_t applicationIdStringSize = sizeof(applicationIdString);
        util::SNPrintf(applicationIdString, sizeof(applicationIdString), "%016llx", launchProperty.id.value);
        NN_RESULT_DO(context.Add(erpt::FieldId::RunningApplicationId, applicationIdString, applicationIdStringSize));

        auto title = controlProperty.GetDefaultTitle();
        NN_RESULT_DO(context.Add(erpt::FieldId::RunningApplicationTitle, title.name, static_cast<uint32_t>(std::strlen(title.name))));

        const char* displayVersion = controlProperty.displayVersion;
        NN_RESULT_DO(context.Add(erpt::FieldId::RunningApplicationVersion, displayVersion, static_cast<uint32_t>(std::strlen(displayVersion))));

        const char* storage = GetStorageIdStr(launchProperty.applicationStorageId);
        NN_RESULT_DO(context.Add(erpt::FieldId::RunningApplicationStorageLocation, storage, static_cast<uint32_t>(std::strlen(storage))));

        storage = GetStorageIdStr(launchProperty.patchStorageId);
        NN_RESULT_DO(context.Add(erpt::FieldId::RunningApplicationPatchStorageLocation, storage, static_cast<uint32_t>(std::strlen(storage))));

        NN_RESULT_DO(context.Add(erpt::FieldId::RunningApplicationVersionNumber, launchProperty.version));
        NN_RESULT_DO(context.SubmitContext());

        context.ResetContext(erpt::CategoryId::AcpGeneralSettingsInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpStartupUserAccount, static_cast<uint8_t>(controlProperty.startupUserAccount)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpAttributeFlag, controlProperty.attributeFlag));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpSupportedLanguageFlag, controlProperty.supportedLanguageFlag));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpParentalControlFlag, controlProperty.parentalControlFlag));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpScreenShot, static_cast<uint8_t>(controlProperty.screenShot)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpVideoCapture, static_cast<uint8_t>(controlProperty.videoCapture)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpDataLossConfirmation, static_cast<uint8_t>(controlProperty.dataLossConfirmation)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpPresenceGroupId, controlProperty.presenceGroupId.value));
        int l = util::Strnlen(controlProperty.applicationErrorCodeCategory.value, NN_ARRAY_SIZE(controlProperty.applicationErrorCodeCategory.value));
        if( l > 0 )
        {
            NN_RESULT_DO(context.Add(erpt::FieldId::AcpApplicationErrorCodeCategory, controlProperty.applicationErrorCodeCategory.value, static_cast<uint32_t>(l)));
        }
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpLocalCommunicationId, controlProperty.localCommunicationId, NN_ARRAY_SIZE(controlProperty.localCommunicationId)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpLogoType, static_cast<uint8_t>(controlProperty.logoType)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpLogoHandling, static_cast<uint8_t>(controlProperty.logoHandling)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpCrashReport, static_cast<uint8_t>(controlProperty.crashReport)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpHdcp, static_cast<uint8_t>(controlProperty.hdcp)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpSeedForPseudoDeviceId, controlProperty.seedForPseudoDeviceId));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpRepairFlag, controlProperty.repairFlag));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpRequiredNetworkServiceLicenseOnLaunchFlag, controlProperty.requiredNetworkServiceLicenseOnLaunchFlag));
        NN_RESULT_DO(context.SubmitContext());

        context.ResetContext(erpt::CategoryId::AcpAocSettingsInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpAocRegistrationType, static_cast<uint8_t>(controlProperty.addOnContentRegistrationType)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpAocBaseId, controlProperty.addOnContentBaseId));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpRuntimeAocInstall, static_cast<uint8_t>(controlProperty.runtimeAddOnContentInstall)));
        NN_RESULT_DO(context.SubmitContext());

        context.ResetContext(erpt::CategoryId::AcpBcatSettingsInfo);
        /* パスフレーズは保存しない
        l = util::Strnlen(controlProperty.bcatPassphrase, NN_ARRAY_SIZE(controlProperty.bcatPassphrase));
        if( l > 0 )
        {
            NN_RESULT_DO(context.Add(erpt::FieldId::AcpBcatPassphrase, controlProperty.bcatPassphrase, static_cast<uint32_t>(l)));
        }
        */
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpBcatDeliveryCacheStorageSize, controlProperty.bcatDeliveryCacheStorageSize));
        NN_RESULT_DO(context.SubmitContext());

        context.ResetContext(erpt::CategoryId::AcpStorageSettingsInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpUserAccountSaveDataSize, controlProperty.userAccountSaveDataSize));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpUserAccountSaveDataJournalSize, controlProperty.userAccountSaveDataJournalSize));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpDeviceSaveDataSize, controlProperty.deviceSaveDataSize));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpDeviceSaveDataJournalSize, controlProperty.deviceSaveDataJournalSize));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpUserAccountSaveDataSizeMax, controlProperty.userAccountSaveDataJournalSizeMax));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpUserAccountSaveDataJournalSizeMax, controlProperty.userAccountSaveDataJournalSizeMax));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpDeviceSaveDataSizeMax, controlProperty.deviceSaveDataSizeMax));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpDeviceSaveDataJournalSizeMax, controlProperty.deviceSaveDataJournalSizeMax));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpTemporaryStorageSize, controlProperty.temporaryStorageSize));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpCacheStorageSize, controlProperty.cacheStorageSize));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpCacheStorageJournalSize, controlProperty.cacheStorageJournalSize));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpCacheStorageDataAndJournalSizeMax, controlProperty.cacheStorageDataAndJournalSizeMax));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpCacheStorageIndexMax, controlProperty.cacheStorageIndexMax));
        NN_RESULT_DO(context.SubmitContext());

        context.ResetContext(erpt::CategoryId::AcpPlayLogSettingsInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpPlayLogPolicy, static_cast<uint8_t>(controlProperty.playLogPolicy)));
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpPlayLogQueryCapability, static_cast<uint8_t>(controlProperty.playLogQueryCapability)));
        uint32_t count = 0;
        for( ; count < NN_ARRAY_SIZE(controlProperty.playLogQueryableApplicationId); count++ )
        {
            if( controlProperty.playLogQueryableApplicationId[count] == ncm::ApplicationId::GetInvalidId() )
            {
                break;
            }
        }
        if( count > 0 )
        {
            NN_RESULT_DO(context.Add(erpt::FieldId::AcpPlayLogQueryableApplicationId,
                reinterpret_cast<const Bit64*>(controlProperty.playLogQueryableApplicationId), count));
        }
        NN_RESULT_DO(context.SubmitContext());

        context.ResetContext(erpt::CategoryId::AcpRatingSettingsInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::AcpRatingAge, controlProperty.ratingAge, NN_ARRAY_SIZE(controlProperty.ratingAge)));
        NN_RESULT_DO(context.SubmitContext());

        NN_RESULT_SUCCESS;
    }

    void SetErrorReportFileSystemInfo() NN_NOEXCEPT
    {
        nn::fs::SetErrorReportNsInfo();
        nn::fs::SetErrorReportFileSystemInfo();
    }

    Result SubmitMultimediaInfo(const movie::MultimediaTelemetryReport mmTelemetryReport) NN_NOEXCEPT
    {
        auto context = erpt::Context(erpt::CategoryId::MultimediaInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::VideoCodecTypeEnum, mmTelemetryReport.VideoCodecTypeEnum));
        NN_RESULT_DO(context.Add(erpt::FieldId::VideoBitRate, mmTelemetryReport.VideoBitRate));
        NN_RESULT_DO(context.Add(erpt::FieldId::VideoFrameRate, mmTelemetryReport.VideoFrameRate));
        NN_RESULT_DO(context.Add(erpt::FieldId::VideoWidth, mmTelemetryReport.VideoWidth));
        NN_RESULT_DO(context.Add(erpt::FieldId::VideoHeight, mmTelemetryReport.VideoHeight));
        NN_RESULT_DO(context.Add(erpt::FieldId::AudioCodecTypeEnum, mmTelemetryReport.AudioCodecTypeEnum));
        NN_RESULT_DO(context.Add(erpt::FieldId::AudioSampleRate, mmTelemetryReport.AudioSampleRate));
        NN_RESULT_DO(context.Add(erpt::FieldId::AudioBitRate, mmTelemetryReport.AudioBitRate));
        NN_RESULT_DO(context.Add(erpt::FieldId::MultimediaContainerType, mmTelemetryReport.ContainerType));
        NN_RESULT_DO(context.Add(erpt::FieldId::MultimediaProfileType, mmTelemetryReport.ProfileType));
        NN_RESULT_DO(context.Add(erpt::FieldId::MultimediaLevelType, mmTelemetryReport.LevelType));
        NN_RESULT_DO(context.Add(erpt::FieldId::MultimediaCacheSizeEnum, mmTelemetryReport.CacheSizeEnum));
        NN_RESULT_DO(context.Add(erpt::FieldId::MultimediaErrorStatusEnum, mmTelemetryReport.ErrorStatusEnum));
        NN_RESULT_DO(context.Add(erpt::FieldId::MultimediaErrorLog, mmTelemetryReport.ErrorLog, sizeof(mmTelemetryReport.ErrorLog) / sizeof(mmTelemetryReport.ErrorLog[0])));
        NN_RESULT_DO(context.SubmitContext());
        NN_RESULT_SUCCESS;
    }

    void ClearErptContext(erpt::CategoryId category) NN_NOEXCEPT
    {
        auto context = erpt::Context(category);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    }

    Result ClearErrorReportApplicationInfo()
    {
        ClearErptContext(erpt::ApplicationInfo);
        NN_RESULT_SUCCESS;
    }

    void ClearErrorReportFileSystemInfo() NN_NOEXCEPT
    {
        nn::fs::ClearErrorReportNsInfo();
        nn::fs::ClearErrorReportFileSystemInfo();
    }

    Result MakeErrorReport(Result result, bool isApplicationAbort, bool isCreateProcessFailure, erpt::ReportType reportType = erpt::ReportType_Invisible) NN_NOEXCEPT
    {
        SetErrorReportFileSystemInfo();

        NN_UTIL_SCOPE_EXIT
        {
            ClearErrorReportApplicationInfo();
            ClearErrorReportFileSystemInfo();
        };

        auto errorCode = err::ConvertResultToErrorCode(result);
        char buffer[err::ErrorCode::StringLengthMax];
        err::GetErrorCodeString(buffer, sizeof(buffer), errorCode);

        erpt::Context context(erpt::ErrorInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::ErrorCode, buffer, static_cast<uint32_t>(std::strlen(buffer))));
        NN_RESULT_DO(context.Add(erpt::FieldId::AbortFlag, isApplicationAbort));
        if( isApplicationAbort )
        {
            // 必須項目ではないので true 時のみ付与。
            NN_RESULT_DO(context.Add(erpt::FieldId::ApplicationAbortFlag, isApplicationAbort));
        }
        NN_RESULT_DO(context.Add(erpt::FieldId::CreateProcessFailureFlag, isCreateProcessFailure));
        NN_RESULT_DO(context.CreateReport(reportType));
        NN_RESULT_SUCCESS;
    }

    Result MakeVisibleErrorReportImpl(err::ErrorCode errorCode) NN_NOEXCEPT
    {
        SetErrorReportFileSystemInfo();

        NN_UTIL_SCOPE_EXIT
        {
            ClearErrorReportApplicationInfo();
            ClearErrorReportFileSystemInfo();
        };

        char buffer[err::ErrorCode::StringLengthMax];
        err::GetErrorCodeString(buffer, sizeof(buffer), errorCode);
        erpt::Context context(erpt::ErrorInfo);
        NN_RESULT_DO(context.Add(erpt::FieldId::ErrorCode, buffer, static_cast<uint32_t>(std::strlen(buffer))));
        NN_RESULT_DO(context.CreateReport(erpt::ReportType_Visible));
        NN_RESULT_SUCCESS;
    }

    Result MakeVisibleErrorReportImpl(err::ErrorCode errorCode, const err::ErrorContext& errorContext) NN_NOEXCEPT
    {
        SetErrorReportFileSystemInfo();

        NN_UTIL_SCOPE_EXIT
        {
            ClearErrorReportApplicationInfo();
            ClearErrorReportFileSystemInfo();
        };

        char buffer[err::ErrorCode::StringLengthMax];
        err::GetErrorCodeString(buffer, sizeof(buffer), errorCode);
        erpt::Context context(erpt::ErrorInfo, g_ErrorContextBuffer, static_cast<uint32_t>(sizeof(g_ErrorContextBuffer)));
        NN_RESULT_DO(context.AddErrorContext((errorContext)));
        NN_RESULT_DO(context.Add(erpt::FieldId::ErrorCode, buffer, static_cast<uint32_t>(std::strlen(buffer))));
        NN_RESULT_DO(context.CreateReport(erpt::ReportType_Visible));
        NN_RESULT_SUCCESS;
    }
}

Result MakeApplicationAbortErrorReport(Result result, const ApplicationErrorReportInfo& info) NN_NOEXCEPT
{
    NN_RESULT_DO(SubmitApplicationInfo(info));

    Result r;
    Result terminateResult;
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(ns_GetApplicationTerminateResult, 60);
        r = ns::GetApplicationTerminateResult(&terminateResult, info.launchProperty.id);
    }
    if( r.IsFailure() )
    {
        NN_DETAIL_AM_WARN("MakeApplicationAbortErrorReport: Failed to get application (0x%016llx) terminate result : %03d-%04d (0x%08lx)\n",
            info.launchProperty.id.value, r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
    }
    // GetApplicationTerminateResult で取得できる = 外部に公開できる Result でのアボートのみ Visible にする。
    auto reportVisibility = (r.IsSuccess() && terminateResult.IsFailure()) ? erpt::ReportType_Visible : erpt::ReportType_Invisible;
    return MakeErrorReport(result, true, false, reportVisibility);
}

Result MakeCreateProcessFailureErrorReport(Result result, Bit64 id) NN_NOEXCEPT
{
    erpt::Context context(erpt::ApplicationInfo);

    char applicationIdString[17];
    const size_t applicationIdStringSize = sizeof(applicationIdString);
    util::SNPrintf(applicationIdString, sizeof(applicationIdString), "%016llx", id);
    NN_RESULT_DO(context.Add(erpt::FieldId::ApplicationID, applicationIdString, applicationIdStringSize));
    NN_RESULT_DO(context.SubmitContext());

    return MakeErrorReport(result, false, true);
}

Result MakeVisibleErrorReport(err::ErrorCode errorCode, am::service::IntegratedApplet* pCallerApplet) NN_NOEXCEPT
{
    NN_RESULT_DO(SelectApplicationInfoReportApplet(pCallerApplet)->SubmitApplicationInfo());
    NN_RESULT_DO(MakeVisibleErrorReportImpl(errorCode));
    NN_RESULT_SUCCESS;
}

Result MakeVisibleErrorReport(err::ErrorCode errorCode, const err::ErrorContext& errorContext, am::service::IntegratedApplet* pCallerApplet) NN_NOEXCEPT
{
    NN_RESULT_DO(SelectApplicationInfoReportApplet(pCallerApplet)->SubmitApplicationInfo());
    NN_RESULT_DO(MakeVisibleErrorReportImpl(errorCode, errorContext));
    NN_RESULT_SUCCESS;
}

Result MakeMultimediaErrorReportIfNecessary(Bit32 resultValue, am::service::IntegratedApplet* pCallerApplet, const char* pMultimediaTelemetry, size_t multimediaTelemetrySize) NN_NOEXCEPT
{
    const nn::TimeSpan MinimumReportInterval = nn::TimeSpan::FromMinutes(1);
    NN_FUNCTION_LOCAL_STATIC(os::Tick, s_LastMultimediaErrorReportTick, = (os::Tick(0) - os::ConvertToTick(MinimumReportInterval)) );

    // 一定時間以内にこの API でエラーレポートが作成されていた場合にはスキップする。
    // Multimedia ライブラリがアプリなどから想定外の使われ方をされた場合に、大量のレポートが作成されてしまう可能性を排除するため。
    if( (os::GetSystemTick() - s_LastMultimediaErrorReportTick).ToTimeSpan() < MinimumReportInterval )
    {
        NN_DETAIL_AM_TRACE("MakeMultimediaErrorReportIfNecessary : Skip making an error report. Not enough time has passed since the last one.\n");
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW_UNLESS(multimediaTelemetrySize == sizeof(movie::MultimediaTelemetryReport), am::ResultInvalidParameter());
    NN_RESULT_DO(pCallerApplet->SubmitApplicationInfo()); // Invisible なので呼び出しスタック関係なく呼び出し元を保存する。
    NN_RESULT_DO(SubmitMultimediaInfo(*reinterpret_cast<const movie::MultimediaTelemetryReport*>(pMultimediaTelemetry)));
    NN_UTIL_SCOPE_EXIT{ ClearErptContext(erpt::CategoryId::MultimediaInfo); };
    NN_RESULT_DO(MakeErrorReport(result::detail::ConstructResult(resultValue), false, false));
    s_LastMultimediaErrorReportTick = os::GetSystemTick();
    NN_RESULT_SUCCESS;
}

void RegisterRunningApplication(const ApplicationErrorReportInfo& info) NN_NOEXCEPT
{
    auto result = SubmitRunningApplicationInfo(info);
    if( result.IsFailure() )
    {
        NN_DETAIL_AM_WARN("SubmitRunningApplicationInfo failed: %03d-%04d (0x%08lx)\n",
            result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
    }
}

void UnregisterRunningApplication() NN_NOEXCEPT
{
    ClearErptContext(erpt::CategoryId::RunningApplicationInfo);
    ClearErptContext(erpt::CategoryId::AcpGeneralSettingsInfo);
    ClearErptContext(erpt::CategoryId::AcpPlayLogSettingsInfo);
    ClearErptContext(erpt::CategoryId::AcpAocSettingsInfo);
    ClearErptContext(erpt::CategoryId::AcpBcatSettingsInfo);
    ClearErptContext(erpt::CategoryId::AcpStorageSettingsInfo);
    ClearErptContext(erpt::CategoryId::AcpRatingSettingsInfo);
}

void RegisterRunningApplet(ncm::ProgramId programId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_LESS(g_RunningAppletCount, RunningAppletCountMax);

    std::lock_guard<decltype(g_RunningAppletContextMutex)> lock(g_RunningAppletContextMutex);

    // 起動したアプレットを配列の先頭に配置する。[0, g_RunningAppletCount - 1] を [1, g_RunningAppletCount] に移動して 0 に追加。
    std::memmove(&g_RunningApplet[1], &g_RunningApplet[0], sizeof(g_RunningApplet[0]) * g_RunningAppletCount);
    g_RunningApplet[0] = programId.value;
    g_RunningAppletCount += 1;

    auto context = erpt::Context(erpt::CategoryId::RunningAppletInfo);
    context.Add(erpt::FieldId::RunningAppletList, g_RunningApplet, g_RunningAppletCount);
    context.SubmitContext();
}

void UnregisterRunningApplet(ncm::ProgramId programId) NN_NOEXCEPT
{
    for( int i = 0; i < RunningAppletCountMax; i++ )
    {
        if( g_RunningApplet[i] == programId.value )
        {
            std::lock_guard<decltype(g_RunningAppletContextMutex)> lock(g_RunningAppletContextMutex);

            // 取り除いたアプレットの分を詰めて末尾を無効化。
            std::memmove(&g_RunningApplet[i], &g_RunningApplet[i + 1], sizeof(g_RunningApplet[0]) * (g_RunningAppletCount - 1 - i));
            g_RunningApplet[g_RunningAppletCount - 1] = ncm::ProgramId::GetInvalidId().value;
            g_RunningAppletCount -= 1;

            auto context = erpt::Context(erpt::CategoryId::RunningAppletInfo);
            context.Add(erpt::FieldId::RunningAppletList, g_RunningApplet, g_RunningAppletCount);
            context.SubmitContext();
            return;
        }
    }
    NN_SDK_ASSERT(false, "ProgramId 0x%016llx is not found.\n", programId.value);
}

void UpdateFocusedAppletHistory(ncm::ProgramId programId) NN_NOEXCEPT
{
    static os::Mutex s_Mutex(false);
    std::lock_guard<decltype(s_Mutex)> lock(s_Mutex);

    // フォーカスを取得したアプレットを配列の先頭に配置する。[0, MAX - 2] を [1, MAX - 1] に移動して 0 に追加。
    std::memmove(&g_ForegroundAppletHistory[1], &g_ForegroundAppletHistory[0], sizeof(g_ForegroundAppletHistory[0]) * (ForegroundAppletHistoryCountMax - 1));
    g_ForegroundAppletHistory[0] = programId.value;

    auto context = erpt::Context(erpt::CategoryId::FocusedAppletHistoryInfo);
    context.Add(erpt::FieldId::FocusedAppletHistory, g_ForegroundAppletHistory, ForegroundAppletHistoryCountMax);
    context.SubmitContext();
}

Result SubmitApplicationInfo(const ApplicationErrorReportInfo& info) NN_NOEXCEPT
{
    auto& controlProperty = info.controlProperty;
    auto& launchProperty = info.launchProperty;

    erpt::Context context(erpt::ApplicationInfo);

    NN_RESULT_DO(AddApplicationIdField(&context, launchProperty.id.value));

    auto title = controlProperty.GetDefaultTitle();
    NN_RESULT_DO(context.Add(erpt::FieldId::ApplicationTitle, title.name, static_cast<uint32_t>(std::strlen(title.name))));

    const char* displayVersion = controlProperty.displayVersion;
    NN_RESULT_DO(context.Add(erpt::FieldId::ApplicationVersion, displayVersion, static_cast<uint32_t>(std::strlen(displayVersion))));

    const char* storage = GetStorageIdStr(launchProperty.applicationStorageId);
    NN_RESULT_DO(context.Add(erpt::FieldId::ApplicationStorageLocation, storage, static_cast<uint32_t>(std::strlen(storage))));

    NN_RESULT_DO(context.SubmitContext());

    NN_RESULT_SUCCESS;
}

Result SubmitApplicationInfo(Bit64 idValue) NN_NOEXCEPT
{
    erpt::Context context(erpt::ApplicationInfo);
    NN_RESULT_DO(AddApplicationIdField(&context, idValue));
    NN_RESULT_DO(context.SubmitContext());
    NN_RESULT_SUCCESS;
}

void UpdateErrorReportControllerUsage() NN_NOEXCEPT
{
    NN_DETAIL_AM_TRACE("UpdateErrorReportControllerUsage()\n");
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    hid::system::PlayReportControllerUsage usages[hid::system::PlayReportControllerUsageCountMax];
    auto count = hid::system::GetPlayReportControllerUsages(usages, hid::system::PlayReportControllerUsageCountMax);

    uint8_t unknownControllerCount = 0;
    uint8_t attachedControllerCount = 0;
    uint8_t bluetoothControllerCount = 0;
    uint8_t usbControllerCount = 0;

    uint8_t controllerTypes[hid::system::PlayReportControllerUsageCountMax];
    uint8_t controllerStyles[hid::system::PlayReportControllerUsageCountMax];
    uint8_t controllerInterfaces[hid::system::PlayReportControllerUsageCountMax];

    for( int i = 0; i < count; i++ )
    {
        auto deviceType = usages[i].deviceType;

        controllerTypes[i] = deviceType;
        controllerStyles[i] = usages[i].style;

        switch( deviceType )
        {
        case hid::system::PlayReportDeviceType_JoyConLeft:
        case hid::system::PlayReportDeviceType_JoyConRight:
        case hid::system::PlayReportDeviceType_SwitchProController:
            {
                controllerInterfaces[i] = usages[i].nxControllerInfo.interfaceType;
                switch( usages[i].nxControllerInfo.interfaceType )
                {
                case hid::system::InterfaceType_Rail:
                    {
                        attachedControllerCount++;
                    }
                    break;
                case hid::system::InterfaceType_Bluetooth:
                    {
                        bluetoothControllerCount++;
                    }
                    break;
                case hid::system::InterfaceType_Usb:
                    {
                        usbControllerCount++;
                    }
                    break;
                case hid::system::InterfaceType_Unknown:
                default:
                    {
                        unknownControllerCount++;
                    }
                    break;
                }
            }
            break;
        case hid::system::PlayReportDeviceType_UsbController:
            {
                controllerInterfaces[i] = hid::system::InterfaceType_Usb;
                usbControllerCount++;
            }
            break;
        case hid::system::PlayReportDeviceType_Unknown:
        default:
            {
                controllerInterfaces[i] = hid::system::InterfaceType_Unknown;
                unknownControllerCount++;
            }
            break;
        }
    }

    erpt::Context context(erpt::CategoryId::ConnectedControllerInfo);
    context.Add(erpt::FieldId::UnknownControllerCount, unknownControllerCount);
    context.Add(erpt::FieldId::AttachedControllerCount, attachedControllerCount);
    context.Add(erpt::FieldId::BluetoothControllerCount, bluetoothControllerCount);
    context.Add(erpt::FieldId::UsbControllerCount, usbControllerCount);
    if( count > 0 )
    {
        context.Add(erpt::FieldId::ControllerTypeList, controllerTypes, static_cast<uint32_t>(count));
        context.Add(erpt::FieldId::ControllerStyleList, controllerStyles, static_cast<uint32_t>(count));
        context.Add(erpt::FieldId::ControllerInterfaceList, controllerInterfaces, static_cast<uint32_t>(count));
    }
    context.SubmitContext();
#endif
}

}}}
