﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Types.h>
#include <nn/applet/applet_Types.h>

namespace nn { namespace am { namespace service {

class AudioControl
{
private:

    static const int AppletCountMax = 16;

    class AppletInfoForAudioControl
    {
    private:
        applet::AppletResourceUserId m_Aruid = applet::AppletResourceUserId::GetInvalidId();
        int64_t m_SuspendCount{0};
        bool m_SuspendAudioForDebugEnabled{false};
        bool m_PreviousNeedSuspend{false};

    public:
        void UpdateAudioSuspendState() NN_NOEXCEPT;

        int64_t IncrementSuspendCount() NN_NOEXCEPT
        {
            ++m_SuspendCount;
            return m_SuspendCount;
        }

        int64_t DecrementSuspendCount() NN_NOEXCEPT
        {
            if (m_SuspendCount > 0)
            {
                --m_SuspendCount;
            }
            return m_SuspendCount;
        }

        void SetSuspendAudioForDebugEnabled(bool enabled) NN_NOEXCEPT
        {
            m_SuspendAudioForDebugEnabled = enabled;
        }

        void Set(applet::AppletResourceUserId aruid) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(aruid != applet::AppletResourceUserId::GetInvalidId());
            m_Aruid = aruid;
            m_SuspendCount = 0;
            m_SuspendAudioForDebugEnabled = false;
        }
        void Clear() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Aruid != applet::AppletResourceUserId::GetInvalidId());
            m_Aruid = applet::AppletResourceUserId::GetInvalidId();
            m_SuspendCount = 0;
            m_SuspendAudioForDebugEnabled = false;
            m_PreviousNeedSuspend = false;
        }
        bool IsEmpty() const NN_NOEXCEPT
        {
            return m_Aruid == applet::AppletResourceUserId::GetInvalidId();
        }
        bool IsEqual(applet::AppletResourceUserId aruid) const NN_NOEXCEPT
        {
            return m_Aruid == aruid;
        }
    };

    AppletInfoForAudioControl   m_List[AppletCountMax];
    nn::os::Mutex               m_Mutex{false};

    AppletInfoForAudioControl* FindAruidAndRegisterUnsafe(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    AppletInfoForAudioControl* FindAruidUnsafe(applet::AppletResourceUserId aruid) NN_NOEXCEPT;

public:
    bool Register(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    bool Unregister(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    void SuspendAudio(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    void ResumeAudio(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    void SuspendAudioForDebug(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    void ResumeAudioForDebug(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
};

void RegisterAruidForAudioControl(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
void UnregisterAruidForAudioControl(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
void SuspendAudioInProcess(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
void ResumeAudioInProcess(applet::AppletResourceUserId aruid) NN_NOEXCEPT;
void SuspendAudioInProcessForDebug(os::ProcessId pid) NN_NOEXCEPT;
void ResumeAudioInProcessForDebug(os::ProcessId pid) NN_NOEXCEPT;

}}}
