﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_Tick.h>
#include <nn/am/service/am_AppletSystem.h>
#include <nn/hid/system/hid_InputDetection.h>

namespace nn { namespace am { namespace service {

//-----------------------------------------------------------------------------
//  ApplicationTimer 監視用クラス
//
//  アプリケーションに対して以下の時間を計測する。
//  - アプリが起動してからの総経過時間（フォーカス状態に依存しない）
//  - アプリが無操作状態を継続している時間
//
class ApplicationTimer
{
public:
    void StartPlayableTimer(TimeSpan playableTime) NN_NOEXCEPT;
    void StopPlayableTimer() NN_NOEXCEPT;
    void StartIdleDetectionTimer(TimeSpan idleDetectionTime) NN_NOEXCEPT;
    void StopIdleDetectionTimer() NN_NOEXCEPT;

public:
    static TimeSpan GetPeriodicTimeForIdleDetection() NN_NOEXCEPT
    {
        // 1 秒おきにアプリ無操作時間のチェックを行う
        return TimeSpan::FromMilliSeconds(1000);
    }

public:
    ApplicationTimer() NN_NOEXCEPT;
    ~ApplicationTimer() NN_NOEXCEPT;

    void SetAppletSystem(AppletSystem* pAppletSystem) NN_NOEXCEPT
    {
        m_pAppletSystem = pAppletSystem;
    }

    void LinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        os::LinkMultiWaitHolder(p, &m_IdleDetectionTimerEventHolder);
        os::LinkMultiWaitHolder(p, &m_PlayableTimerEventHolder);
    }
    void UnlinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        NN_UNUSED(p);
        os::UnlinkMultiWaitHolder(&m_IdleDetectionTimerEventHolder);
        os::UnlinkMultiWaitHolder(&m_PlayableTimerEventHolder);
    }
    bool HandleEventIfHolderOwner(os::MultiWaitHolderType* holder)
    {
        if (holder == &m_IdleDetectionTimerEventHolder)
        {
            IdleDetectionPeriodicTimerHandler();
            return true;
        }
        else if (holder == &m_PlayableTimerEventHolder)
        {
            PlayableTimerHandler();
            return true;
        }
        return false;
    }

private:
    os::Tick GetCurrentIdlingTickCount() NN_NOEXCEPT;
    void IdleDetectionPeriodicTimerHandler() NN_NOEXCEPT;
    void PlayableTimerHandler() NN_NOEXCEPT;

    void RequestToGoBackQuestMenu() NN_NOEXCEPT
    {
        if (m_pAppletSystem)
        {
            m_pAppletSystem->NotifyGoBackQuestMenu();
        }
    }

private:
    AppletSystem* m_pAppletSystem;

    os::TimerEvent m_PlayableTimer{os::EventClearMode_AutoClear};
    os::TimerEvent m_IdleDetectionTimer{os::EventClearMode_AutoClear};

    os::MultiWaitHolderType m_IdleDetectionTimerEventHolder;
    os::MultiWaitHolderType m_PlayableTimerEventHolder;

    os::Tick m_LastInputDetectedTick{0};
    os::Tick m_IdleDetectionTotalTick{0};

    os::Mutex m_Mutex{true};

    bool m_IsIdleDetectionTimerEnabled{false};
    bool m_IsPlayableTimerEnabled{false};
};

ApplicationTimer& GetApplicationTimer() NN_NOEXCEPT;

//-----------------------------------------------------------------------------

}}} // namespace nn::am::service

