﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include "am_ApplicationTimer.h"
#include <mutex>
#include <algorithm>

namespace nn { namespace am { namespace service {

ApplicationTimer& GetApplicationTimer() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(ApplicationTimer, g_ApplicationTimer);
    return g_ApplicationTimer;
}

//-----------------------------------------------------------------------------
//  コンストラクタ
//
ApplicationTimer::ApplicationTimer() NN_NOEXCEPT
{
    m_IdleDetectionTimer.Clear();
    m_PlayableTimer.Clear();

    os::InitializeMultiWaitHolder(&m_IdleDetectionTimerEventHolder, m_IdleDetectionTimer.GetBase());
    os::InitializeMultiWaitHolder(&m_PlayableTimerEventHolder, m_PlayableTimer.GetBase());

    m_IdleDetectionTimerEventHolder.userData = reinterpret_cast<uintptr_t>(this);
    m_PlayableTimerEventHolder.userData = reinterpret_cast<uintptr_t>(this);
}


//-----------------------------------------------------------------------------
//  デストラクタ
//
ApplicationTimer::~ApplicationTimer() NN_NOEXCEPT
{
    os::FinalizeMultiWaitHolder(&m_IdleDetectionTimerEventHolder);
    os::FinalizeMultiWaitHolder(&m_PlayableTimerEventHolder);
}

//-----------------------------------------------------------------------------
//  無操作時間検出用のハンドラ関数（多重待ち解除時に呼ばれる）
//
void ApplicationTimer::StartIdleDetectionTimer(TimeSpan idleDetectionTime) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);

    m_IdleDetectionTimer.Stop();
    m_IdleDetectionTimer.Clear();

    m_LastInputDetectedTick = os::GetSystemTick();
    m_IdleDetectionTotalTick = os::ConvertToTick( idleDetectionTime );

    auto interval = this->GetPeriodicTimeForIdleDetection();
    m_IdleDetectionTimer.StartPeriodic( interval, interval );
    m_IsIdleDetectionTimerEnabled = true;
}

void ApplicationTimer::StopIdleDetectionTimer() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);

    m_IsIdleDetectionTimerEnabled = false;
    m_IdleDetectionTimer.Stop();
    m_IdleDetectionTimer.Clear();
}

os::Tick ApplicationTimer::GetCurrentIdlingTickCount() NN_NOEXCEPT
{
    // 最後に入力を検知したときの Tick を取得（アナログスティック以外が対象）
    hid::system::InputSourceState inputState;
    auto inputSourceIdSet = hid::system::InputSourceIdSet().Set();
    inputSourceIdSet.Reset(hid::system::InputSourceId::AnalogStick::Index);
    hid::system::GetInputSourceState(&inputState, inputSourceIdSet);

    auto lastInputDetectedTick = os::Tick(inputState.timestamp);
    if (lastInputDetectedTick > m_LastInputDetectedTick)
    {
        m_LastInputDetectedTick = lastInputDetectedTick;
    }

    // 現在の Tick との差分を返す
    return os::GetSystemTick() - m_LastInputDetectedTick;
}

void ApplicationTimer::IdleDetectionPeriodicTimerHandler() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);

    if (!m_IdleDetectionTimer.TryWait())
    {
        return;
    }
    if (!m_IsIdleDetectionTimerEnabled)
    {
        return;
    }

    // 無操作が継続している時間を取得
    auto currentIdlingTick = this->GetCurrentIdlingTickCount();
    if (currentIdlingTick > m_IdleDetectionTotalTick)
    {
        this->StopIdleDetectionTimer();
        NN_AM_SERVICE_LOG(event, "Request to go back Quest Menu because idle detected(%lld(msec) > %lld(msec))\n", currentIdlingTick.ToTimeSpan().GetMilliSeconds(), m_IdleDetectionTotalTick.ToTimeSpan().GetMilliSeconds());
        this->RequestToGoBackQuestMenu();
    }
}


//-----------------------------------------------------------------------------
//  プレイ時間計測用のハンドラ関数（多重待ち解除時に呼ばれる）
//
void ApplicationTimer::StartPlayableTimer(TimeSpan playableTime) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);

    m_IsPlayableTimerEnabled = true;
    m_PlayableTimer.Stop();
    m_PlayableTimer.Clear();
    m_PlayableTimer.StartOneShot( playableTime );
}

void ApplicationTimer::StopPlayableTimer() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);

    m_IsPlayableTimerEnabled = false;
    m_PlayableTimer.Stop();
    m_PlayableTimer.Clear();
}

void ApplicationTimer::PlayableTimerHandler() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);

    if (!m_PlayableTimer.TryWait())
    {
        return;
    }
    if (!m_IsPlayableTimerEnabled)
    {
        return;
    }

    // 指定されたプレイ時間を経過した
    this->StopPlayableTimer();
    NN_AM_SERVICE_LOG(event, "Request to go back Quest Menu because elapsed playable time\n");
    this->RequestToGoBackQuestMenu();
}


//-----------------------------------------------------------------------------

}}} // namespace nn::am::service

