﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_AppletSystem.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Abort.h>
#include <mutex>
#include <utility>
#include <nn/nn_SdkAssert.h>
#include <nn/am/am_ResultPrivate.h>

#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/am/service/am_IntegratedSystemApplet.h>
#include <nn/am/service/am_IntegratedOverlayApplet.h>
#include <nn/am/service/am_IntegratedLibraryApplet.h>
#include <nn/am/service/am_IntegratedApplication.h>

#include <nn/am/service/am_Storage.h>
#include <nn/am/service/am_ServiceConfig.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/am/service/am_GpuResourceControl.h>
#include <nn/am/service/am_NsWrapper.h>
#include <nn/lbl/lbl_VrMode.h>

#include <nn/os/os_SdkUnsafeMemory.h>
#include <nn/ns/ns_Result.h>

namespace nn { namespace am { namespace service {

namespace detail {

Result AppletProxyManager::GetAppletProxy(std::shared_ptr<AppletProxy>* pOut, nn::Bit64 processId, sf::NativeHandle processHandle, const ProxyOption& proxyOption, const AppletAttribute* pAttribute) NN_NOEXCEPT
{
    if ( processHandle.GetOsHandle() == 0 )
    {
        *pOut = nullptr;
        NN_RESULT_THROW(ResultNoAppletProxy());
    }

    std::lock_guard<decltype(m_MapMutex)> lk(m_MapMutex);
    auto it = m_Map.find(processId);
    if (it == m_Map.end())
    {
        *pOut = nullptr;
        NN_RESULT_THROW(ResultNoAppletProxy());
    }
    if (it->second.p == nullptr)
    {
        // アプリのプロキシ生成待ちの場合は nullptr で SUCCESS を返す
        *pOut = nullptr;
        NN_RESULT_SUCCESS;
    }
    std::shared_ptr<AppletProxy> ret;
    NN_RESULT_DO(it->second.p->CreateAppletProxy(&ret, {processId}));
    it->second.p->SetProcessHandle(std::move(processHandle));
    if (pAttribute)
    {
        it->second.p->SetAppletAttribute(pAttribute);
    }
    it->second.p->ActivateWindow(proxyOption);
    *pOut = ret;
    NN_RESULT_SUCCESS;
}

void AppletProxyManager::RegisterAppletProxy(nn::Bit64 processId, IntegratedApplet* p) NN_NOEXCEPT
{
#if 1
    // TORIAEZU: TODO:
    //  本来は IntegratedApplet::OnProcessBegin でレジスト処理を抑制すべきだが
    //  超暫定で以下の方法で回避しておく。
    if (processId == 0)
    {
        return;
    }
#endif
    NN_SDK_ASSERT(processId != 0);
    // FloatingApplication の初登録時は p = nullptr で呼ばれる
    std::lock_guard<decltype(m_MapMutex)> lk(m_MapMutex);
    m_Map[processId].p = std::move(p);
}

void AppletProxyManager::UnregisterAppletProxy(nn::Bit64 processId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(processId != 0);
    std::lock_guard<decltype(m_MapMutex)> lk(m_MapMutex);
    auto it = m_Map.find(processId);
    NN_ABORT_UNLESS(it != m_Map.end());
    m_Map.erase(it);
}

inline void AppletCleaner::RegisterApplet(std::shared_ptr<core::Applet> pApplet) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_AM_SERVICE_APPLET_LOG(call, pApplet, "registered to applet cleaner");
    m_Applets.push_back(std::move(pApplet));
}

void AppletCleaner::CleanApplets() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    for (;;)
    {
        auto cleaned = false;
        m_Applets.remove_if([&] (decltype(m_Applets.front())& e)
        {
            if (!(e.unique() && e->TryJoin()))
            {
                return false;
            }
            NN_AM_SERVICE_APPLET_LOG(call, e.get(), "unregistered from applet cleaner");
            cleaned = true;
            return true;
        });
        if (!cleaned)
        {
            break;
        }
    }
}

void FloatingApplicationList::PushApplication(std::shared_ptr<IntegratedApplication> pApplication) NN_NOEXCEPT
{
    NN_AM_SERVICE_APPLET_LOG(call, pApplication, "added to floating list");
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_List.push_back(pApplication);
    {
        auto p = m_pObserver.lock();
        if (p)
        {
            p->NotifyFloatingApplicationExistence();
        }
    }
}

sf::SharedPointer<am::service::IApplicationAccessor> FloatingApplicationList::PopApplication() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    for (;;)
    {
        if (m_List.empty())
        {
            return nullptr;
        }
        auto p = m_List.front().lock();
        m_List.pop_front();
        if (p)
        {
            lk.unlock();
            NN_AM_SERVICE_APPLET_LOG(call, p, "removed from floating");
            return p->CreateAccessor();
        }
        else
        {
            NN_AM_SERVICE_LOG(call, "floating application expired\n");
        }
    }
}

void FloatingApplicationList::CleanApplications() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_List.remove_if([] (decltype(m_List.front())& e)
    {
        auto ret = e.expired();
        if (ret)
        {
            NN_AM_SERVICE_LOG(call, "floating application expired\n");
        }
        return ret;
    });
}

void FloatingApplicationList::ClearAllImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
    if (!m_List.empty())
    {
        NN_AM_SERVICE_LOG(call, "clear %d floating applications\n", static_cast<int>(m_List.size()));
        m_List.clear();
    }
}

void FloatingApplicationList::RegisterObserver(std::shared_ptr<IntegratedSystemApplet> pObserver) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    this->m_pObserver = pObserver;
}

void ApplicationObserver::RegisterObserver(std::shared_ptr<IntegratedSystemApplet> pObserver) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    this->m_pObserver = pObserver;
}

void ApplicationObserver::NotifyApplicationExit() NN_NOEXCEPT
{
    auto p = m_pObserver.lock();
    if (p)
    {
        p->NotifyApplicationExit();
    }
}

void GlobalStateManagerForAppletSystem::OnFinishedSleepSequence() NN_NOEXCEPT
{
    auto pApplet = m_pApplet.lock();
    if (pApplet)
    {
        pApplet->PushMessageToProcess(applet::Message_FinishedSleepSequence);
    }
    m_pApplet.reset();
}

ResourceReservationWindowManager::ResourceReservationWindowManager(window::WindowManager* pWindowManager) NN_NOEXCEPT
    : m_pWindowManager(pWindowManager)
{
    window::WindowProperty prop = {};
    prop.globalOrder = window::WindowGlobalOrder::ResourceReservation;
    prop.pGpuResourceGroupId = GpuResourceGroupId_Overlay;
    prop.gpuTimeSliceBoost = GetGpuTimeSliceOf(GpuTimeKind_Oa);
    this->m_pWindow = pWindowManager->CreateWindow(nullptr, window::GetNullTransiter(), prop);
    auto updater = m_pWindowManager->BeginUpdate();
    updater.ActivateWindow(m_pWindow);
}

ResourceReservationWindowManager::~ResourceReservationWindowManager() NN_NOEXCEPT
{
    {
        auto updater = m_pWindowManager->BeginUpdate();
        updater.DeactivateWindow(m_pWindow);
    }
    m_pWindowManager->RemoveWindow(m_pWindow);
}

BackgroundWindowManager::BackgroundWindowManager(AppletSystem* pAppletSystem, window::WindowManager* pWindowManager) NN_NOEXCEPT
    : m_pAppletSystem(pAppletSystem)
    , m_pWindowManager(pWindowManager)
{
    window::WindowProperty prop = {};
    prop.globalOrder = window::WindowGlobalOrder::Background;
    prop.controlsGpuResource = true;
    prop.defaultHomeButtonLongPressingTimeToSet = GetInitialDefaultHomeLongPressTime();
    prop.setsHomeButtonLongPressingTime = true;
    prop.handlesHomeButtonLongPressed = true;
    prop.handlesCaptureButtonShortPressed = true;
    prop.handlesCaptureButtonLongPressed = true;
    prop.handlesRequestToVrModeCurtain = true;
    prop.pWirelessPriorityModeRequest = WirelessPriorityMode_OptimizedForBluetooth;
    this->m_pWindow = pWindowManager->CreateWindow(nullptr, this, prop);
    auto updater = m_pWindowManager->BeginUpdate();
    updater.ActivateWindow(m_pWindow);
}

BackgroundWindowManager::~BackgroundWindowManager() NN_NOEXCEPT
{
    {
        auto updater = m_pWindowManager->BeginUpdate();
        updater.DeactivateWindow(m_pWindow);
    }
    m_pWindowManager->RemoveWindow(m_pWindow);
}

void BackgroundWindowManager::OnHomeButtonLongPressed() NN_NOEXCEPT
{
    if (auto p = this->m_pAppletSystem->GetTheOverlayApplet())
    {
        p->OnHomeButtonLongPressed();
    }
}

void BackgroundWindowManager::OnCaptureButtonShortPressed() NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(call, "OnCaptureButtonShortPressed() but failed\n");
    auto request = service::display::ScreenShotRequest::GetNotifyingFailureRequest({}, nn::ncm::ApplicationId::GetInvalidId(), nn::os::GetSystemTick());
    service::RequestScreenShotTask(request);
}

void BackgroundWindowManager::OnCaptureButtonLongPressed() NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(call, "OnCaptureButtonLongPressed\n");
    // 何もしない
}

void BackgroundWindowManager::SetDefaultHomeButtonLongPressTime(TimeSpan time) NN_NOEXCEPT
{
    auto updater = m_pWindowManager->BeginUpdate();
    updater.RefWindowProperty(m_pWindow).defaultHomeButtonLongPressingTimeToSet = time;
}

void BackgroundWindowManager::OnVrModeCurtainRequired() NN_NOEXCEPT
{
    if (auto p = this->m_pAppletSystem->GetTheOverlayApplet())
    {
        p->NotifyVrModeCurtainRequired();
    }
    if (auto p = this->m_pAppletSystem->GetTheSystemApplet())
    {
        p->NotifyVrModeCurtainRequired();
    }
}

} // detail

template <typename Applet, typename... Args>
inline std::shared_ptr<Applet> AppletSystem::MakeAppletImpl(Args&&... args) NN_NOEXCEPT
{
    m_AppletCleaner.CleanApplets();
    m_FloatingApplicationList.CleanApplications();
    auto p = MakeShared<Applet>(std::forward<Args>(args)...);
    {
        // TORIAEZU: SA が存在しないとき、自動 move to top を有効にする
        //   DevMenu からの ns 起動時に誰も move to top してくれないため
        auto pTheSystemApplet = m_pTheSystemApplet.lock();
        if (!(pTheSystemApplet && !static_cast<core::Applet*>(pTheSystemApplet.get())->TryJoin()))
        {
            p->SetAutoMoveToTop();
        }
    }
    m_AppletCleaner.RegisterApplet(p);
    return p;
}

inline void AppletSystem::RegisterSystemApplet(std::shared_ptr<IntegratedSystemApplet> pObserver) NN_NOEXCEPT
{
    m_pTheSystemApplet = pObserver;
    m_FloatingApplicationList.RegisterObserver(pObserver);
    m_ApplicationObserver.RegisterObserver(pObserver);
}


void AppletSystem::BeginProcessCreation() NN_NOEXCEPT
{
    ++m_ProcessCreationCount;
}

void AppletSystem::EndProcessCreation() NN_NOEXCEPT
{
    --m_ProcessCreationCount;
}

void AppletSystem::RegisterAppletProxy(nn::Bit64 processId, IntegratedApplet* p) NN_NOEXCEPT
{
    return m_AppletProxyManager.RegisterAppletProxy(processId, std::move(p));
}

void AppletSystem::UnregisterAppletProxy(nn::Bit64 processId) NN_NOEXCEPT
{
    m_AppletProxyManager.UnregisterAppletProxy(processId);
}

void AppletSystem::NotifyGoBackQuestMenu() NN_NOEXCEPT
{
    auto pSystemApplet = this->GetTheSystemApplet();
    if (pSystemApplet)
    {
        pSystemApplet->PushMessageToProcess(applet::Message_RequestToGoBackQuestMenu);
    }
}

Result AppletSystem::GetAppletProxy(std::shared_ptr<AppletProxy>* pOut, nn::Bit64 processId, sf::NativeHandle processHandle, const ProxyOption& proxyOption, const AppletAttribute* pAttribute) NN_NOEXCEPT
{
    auto processCreationCount = m_ProcessCreationCount.load(); // 先読み
    NN_RESULT_TRY( m_AppletProxyManager.GetAppletProxy(pOut, processId, std::move(processHandle), proxyOption, pAttribute) )
        NN_RESULT_CATCH(ResultNoAppletProxy)
        {
            // 検索失敗時
            if (processCreationCount > 0)
            {
                // プロセス作成中だった場合には success: nullptr を返す
                *pOut = nullptr;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

Result AppletSystem::CreateLibraryApplet(sf::SharedPointer<am::service::ILibraryAppletAccessor>* pOut, Bit32 appletId, applet::LibraryAppletMode libraryAppletMode, std::shared_ptr<IntegratedApplet> pParent) NN_NOEXCEPT
{
    // TODO: appletId チェック
    auto p = MakeAppletImpl<IntegratedLibraryApplet>(this, static_cast<applet::AppletId>(appletId), libraryAppletMode);
    p->SetParent(std::move(pParent));
    auto pAcc = p->CreateAccessor();
    *pOut = std::move(pAcc);
    NN_RESULT_SUCCESS;
}

void AppletSystem::CleanAppletListForDebug() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_AppletListForDebugMutex.IsLockedByCurrentThread());
    m_AppletListForDebug.remove_if([] (decltype(m_AppletListForDebug.front())& e)
    {
        return e.expired();
    });
}

Result AppletSystem::CreateApplicationImpl(sf::SharedPointer<am::service::IApplicationAccessor>* pOut, const applet::AppletIdentityInfo& appletIdentityInfo, std::shared_ptr<IntegratedApplet> pParent, const applet::ApplicationLaunchRequestInfo& launchRequestInfo) NN_NOEXCEPT
{
    auto p = MakeAppletImpl<IntegratedApplication>(this, appletIdentityInfo);
    p->SetParent(std::move(pParent));
    p->SetApplicationLaunchRequestInfo(launchRequestInfo);
    this->m_pLastCreatedApplication = p;
    {
        std::lock_guard<decltype(m_AppletListForDebugMutex)> lk(m_AppletListForDebugMutex);
        CleanAppletListForDebug();
        this->m_AppletListForDebug.push_back(p);
    }
    auto pAcc = p->CreateAccessor();
    *pOut = std::move(pAcc);
    NN_RESULT_SUCCESS;
}

void AppletSystem::TerminateAllAppletsForDebug() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_AppletListForDebugMutex)> lk(m_AppletListForDebugMutex);
    while (!m_AppletListForDebug.empty())
    {
        auto p = m_AppletListForDebug.back().lock();
        m_AppletListForDebug.pop_back();
        if (p)
        {
            static_cast<core::Applet*>(p.get())->Terminate();
        }
    }
}

Result AppletSystem::CreateApplication(sf::SharedPointer<am::service::IApplicationAccessor>* pOut, ncm::ApplicationId id, std::shared_ptr<IntegratedApplet> pParent, const applet::ApplicationLaunchRequestInfo& launchRequestInfo) NN_NOEXCEPT
{
    NN_UNUSED(pParent);
    return CreateApplicationImpl(pOut, applet::AppletIdentityInfo::Make(id), nullptr, launchRequestInfo);
}

Result AppletSystem::CreateSystemApplication(sf::SharedPointer<am::service::IApplicationAccessor>* pOut, ncm::SystemApplicationId id, std::shared_ptr<IntegratedApplet> pParent) NN_NOEXCEPT
{
    NN_UNUSED(pParent);
    return CreateApplicationImpl(pOut, applet::AppletIdentityInfo::Make(id), nullptr, {});
}

Result AppletSystem::CreateApplicationForReserved(std::shared_ptr<IntegratedApplication>* pOut, ncm::ApplicationId id, const applet::ApplicationLaunchRequestInfo& launchRequestInfo) NN_NOEXCEPT
{
    auto p = MakeAppletImpl<IntegratedApplication>(this, id);
    p->SetApplicationLaunchRequestInfo(launchRequestInfo);
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result AppletSystem::CreateSystemApplet(std::shared_ptr<core::Applet>* pOut, Bit32 appletId) NN_NOEXCEPT
{
    TerminateAllAppletsForDebug();
    auto p = MakeAppletImpl<IntegratedSystemApplet>(this, static_cast<applet::AppletId>(appletId));
    RegisterSystemApplet(p);
    if (am::service::AbortsOnSystemAppletLost())
    {
        p->SetFatalResultOnExit(ResultSystemAppletExitedUnexpected());
    }
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result AppletSystem::CreateOverlayApplet(std::shared_ptr<core::Applet>* pOut, Bit32 appletId) NN_NOEXCEPT
{
    auto p = MakeAppletImpl<IntegratedOverlayApplet>(this, static_cast<applet::AppletId>(appletId));
    this->m_pTheOverlayApplet = p;
    if (am::service::AbortsOnOverlayAppletLost())
    {
        p->SetFatalResultOnExit(ResultOverlayAppletExitedUnexpected());
    }
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result AppletSystem::CreateApplicationForRunningProcess(std::shared_ptr<core::Applet>* pOut, os::ProcessId processId, const applet::ApplicationLaunchRequestInfo& launchRequestInfo) NN_NOEXCEPT
{
    // NS から起動要求中のプロセスの起動情報を取得する。
    // 情報が未登録の場合は ResultNoAppletProxy で一旦返す。
    ns::ApplicationLaunchInfo launchInfo;
    NN_RESULT_TRY(GetNsProcessManager()->AcquireApplicationLaunchInfoForDevelop(&launchInfo, {processId}))
        NN_RESULT_CATCH(ns::ResultApplicationLaunchInfoNotFound)
        {
            NN_RESULT_THROW(am::ResultNoAppletProxy());
        }
    NN_RESULT_END_TRY

    auto p = MakeAppletImpl<IntegratedApplication>(this, processId, launchInfo);
    p->SetApplicationLaunchRequestInfo(launchRequestInfo);
    p->AttachProcessAndStart(processId);
    {
        std::lock_guard<decltype(m_AppletListForDebugMutex)> lk(m_AppletListForDebugMutex);
        CleanAppletListForDebug();
        this->m_AppletListForDebug.push_back(p);
    }
    m_FloatingApplicationList.PushApplication(p);
    this->m_pLastCreatedApplication = p;
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result AppletSystem::CreateSystemApplicationForRunningProcess(std::shared_ptr<core::Applet>* pOut, os::ProcessId processId) NN_NOEXCEPT
{
    applet::AppletIdentityInfo appletIdentityInfo;
    {
        ncm::SystemApplicationId systemApplicationId = {};
        // TODO: systemApplicationId の推定
        appletIdentityInfo = applet::AppletIdentityInfo::Make(systemApplicationId);
    }
    auto p = MakeAppletImpl<IntegratedApplication>(this, appletIdentityInfo);
    {
        std::lock_guard<decltype(m_AppletListForDebugMutex)> lk(m_AppletListForDebugMutex);
        CleanAppletListForDebug();
        this->m_AppletListForDebug.push_back(p);
    }
    m_FloatingApplicationList.PushApplication(p);
    this->m_pLastCreatedApplication = p;
    p->AttachProcessAndStart(processId);
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result AppletSystem::CreateLibraryAppletForRunningProcess(std::shared_ptr<core::Applet>* pOut, Bit32 appletId, os::ProcessId processId) NN_NOEXCEPT
{
    NN_AM_SERVICE_CHECK_DEVELOPMENT_FUNCTION;
    auto libraryAppletMode = applet::LibraryAppletMode_AllForeground; // NEW_AM: TORIAEZU LibraryAppletMode_AllForeground で決め打ち
    auto p = MakeAppletImpl<IntegratedLibraryApplet>(this, static_cast<applet::AppletId>(appletId), libraryAppletMode);
    p->AttachProcessAndStart(processId);
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result AppletSystem::CreateSystemAppletForRunningProcess(std::shared_ptr<core::Applet>* pOut, Bit32 appletId, os::ProcessId processId) NN_NOEXCEPT
{
    NN_AM_SERVICE_CHECK_DEVELOPMENT_FUNCTION;
    TerminateAllAppletsForDebug();
    auto p = MakeAppletImpl<IntegratedSystemApplet>(this, static_cast<applet::AppletId>(appletId));
    {
        // 既に SA があった場合には終了をトリガ
        // TORIAEZU: 開発用機能なので、非同期に殺してしまって、待機はせずとも大抵は大丈夫なはず
        auto pTheSystemApplet = m_pTheSystemApplet.lock();
        if (pTheSystemApplet)
        {
            pTheSystemApplet->SetFatalResultOnExit(ResultSuccess());
            NN_AM_SERVICE_APPLET_LOG(seq, pTheSystemApplet.get(), "terminated because another SA is launched");
            static_cast<core::Applet*>(pTheSystemApplet.get())->Terminate();
        }
    }
    RegisterSystemApplet(p);
    p->AttachProcessAndStart(processId);
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

Result AppletSystem::CreateOverlayAppletForRunningProcess(std::shared_ptr<core::Applet>* pOut, Bit32 appletId, os::ProcessId processId) NN_NOEXCEPT
{
    NN_AM_SERVICE_CHECK_DEVELOPMENT_FUNCTION;
    auto p = MakeAppletImpl<IntegratedOverlayApplet>(this, static_cast<applet::AppletId>(appletId));
    {
        // 既に OA があった場合には終了をトリガ
        // TORIAEZU: 開発用機能なので、非同期に殺してしまって、待機はせずとも大抵は大丈夫なはず
        auto pTheOverlayApplet = m_pTheOverlayApplet.lock();
        if (pTheOverlayApplet)
        {
            pTheOverlayApplet->SetFatalResultOnExit(ResultSuccess());
            NN_AM_SERVICE_APPLET_LOG(seq, pTheOverlayApplet.get(), "terminated because another OA is launched");
            static_cast<core::Applet*>(pTheOverlayApplet.get())->Terminate();
        }
    }
    this->m_pTheOverlayApplet = p;
    p->AttachProcessAndStart(processId);
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

class AppletSystem::SelfLibraryAppletCreatorImpl
{
private:
    AppletSystem* m_pAppletSystem;
    os::ProcessId m_ProcessId;
public:
    explicit SelfLibraryAppletCreatorImpl(AppletSystem* pAppletSystem, os::ProcessId processId) NN_NOEXCEPT
        : m_pAppletSystem(pAppletSystem)
        , m_ProcessId(processId)
    {
    }
    Result CreateLibraryApplet(sf::Out<sf::SharedPointer<ILibraryAppletAccessor>> pOut, Bit32 appletId, uint32_t libraryAppletMode) NN_NOEXCEPT
    {
        auto p = m_pAppletSystem->MakeAppletImpl<IntegratedLibraryApplet>(m_pAppletSystem, static_cast<applet::AppletId>(appletId), static_cast<applet::LibraryAppletMode>(libraryAppletMode));
        p->SetAutoMoveToTop();
        p->AttachProcessAndStart(m_ProcessId);
        *pOut = p->CreateAccessor();
        NN_RESULT_SUCCESS;
    }
    Result TerminateAllLibraryApplets() NN_NOEXCEPT
    {
        NN_RESULT_THROW(nn::am::ResultInvalidCall());
    }
    Result AreAnyLibraryAppletsLeft(sf::Out<bool>) NN_NOEXCEPT
    {
        NN_RESULT_THROW(nn::am::ResultInvalidCall());
    }
    Result CreateStorage(sf::Out<nn::sf::SharedPointer<IStorage>> pOut, int64_t size) NN_NOEXCEPT
    {
        return MakeSimpleMemoryStorage(pOut, static_cast<size_t>(size));
    }
    Result CreateTransferMemoryStorage(sf::Out<sf::SharedPointer<IStorage>> pOut, sf::NativeHandle&& transferMemory, int64_t size, bool isWritable) NN_NOEXCEPT
    {
        return MakeLargeStorage(pOut, std::move(transferMemory), static_cast<size_t>(size), isWritable);
    }
    Result CreateHandleStorage(sf::Out<sf::SharedPointer<IStorage>> pOut, sf::NativeHandle&& transferMemory, int64_t size) NN_NOEXCEPT
    {
        return MakeHandleStorage(pOut, std::move(transferMemory), static_cast<size_t>(size));
    }
};

Result AppletSystem::CreateSelfLibraryAppletCreatorForDevelop(sf::Out<sf::SharedPointer<ILibraryAppletCreator>> pOut, os::ProcessId processId) NN_NOEXCEPT
{
    if (IsProdMode())
    {
        // 製品機の場合は、AppletProxy を作成せずに専用の Result を返す。
        // これにより、shim でポーリングを継続するようになる。
        // shim で既定のタイムアウト時間を検知したらそこでアボートする。
        NN_RESULT_THROW(ResultRefuseToCreateSelfLibraryAppletProxy());
    }
    NN_AM_SERVICE_CHECK_DEVELOPMENT_FUNCTION;
    *pOut = service::SfObjectFactory::CreateSharedEmplaced<ILibraryAppletCreator, SelfLibraryAppletCreatorImpl>(this, processId);
    NN_RESULT_SUCCESS;
}

Result AppletSystem::NotifyMessageToHomeMenuForDebug(am::AppletMessage message) NN_NOEXCEPT
{
    NN_UNUSED(message);
    NN_AM_SERVICE_NOT_IMPLEMENTED("TODO: implement: AppletSystem::NotifyMessageToHomeMenuForDebug()");
}

Result AppletSystem::OpenMainApplication(sf::Out<sf::SharedPointer<am::service::IApplicationAccessor>> pOut) NN_NOEXCEPT
{
    auto p = m_pLastCreatedApplication.lock();
    if (!p)
    {
        *pOut = nullptr;
        NN_RESULT_SUCCESS;
    }
    *pOut = p->CreateAccessor();
    NN_RESULT_SUCCESS;
}

void AppletSystem::PushLaunchRequestedApplication(std::shared_ptr<IntegratedApplication> pApplication) NN_NOEXCEPT
{
    auto p = m_pTheSystemApplet.lock();
    if (p)
    {
        p->PushLaunchRequestedApplication(std::move(pApplication));
    }
}

Result AppletSystem::SetControllerFirmwareUpdateSection(bool isInSection) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_InControllerFirmwareUpdateSectionMutex)> lk(m_InControllerFirmwareUpdateSectionMutex);
    if (m_InControllerFirmwareUpdateSection == isInSection)
    {
        return ResultUnbalancedControllerFirmwareUpdateSectionControl();
    }
    this->m_InControllerFirmwareUpdateSection = isInSection;
    NN_RESULT_SUCCESS;
}

void AppletSystem::SetVrModeEnabledImpl(bool isEnabled, bool fromOe) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_VrModeMutex)> lk(m_VrModeMutex);

    auto prev = m_VrModeEnabled;
    m_VrModeEnabled = isEnabled;
    if (prev != isEnabled)
    {
        if (isEnabled)
        {
            NN_AM_SERVICE_STUCK_CHECKED(lbl_EnableVrMode, 30, lbl::EnableVrMode());
        }
        else
        {
            NN_AM_SERVICE_STUCK_CHECKED(lbl_DisableVrMode, 30, lbl::DisableVrMode());
        }

        // OA の Window に VrMode フラグをセットする
        GetTheOverlayApplet()->SetVrModeFlag(isEnabled, fromOe);
    }
}

Result AppletSystem::EmulateButtonEvent(EmulatedButtonEvent e) NN_NOEXCEPT
{
    switch (e)
    {
        case EmulatedButtonEvent_PressHomeButtonShort:
        {
            m_WindowManager.NotifyHomeButtonShortPressed();
            NN_RESULT_SUCCESS;
        }
        case EmulatedButtonEvent_PressHomeButtonLong:
        {
            m_WindowManager.NotifyHomeButtonLongPressed();
            NN_RESULT_SUCCESS;
        }
        case EmulatedButtonEvent_PressPowerButtonShort:
        {
            m_WindowManager.NotifyPowerButtonShortPressed();
            NN_RESULT_SUCCESS;
        }
        case EmulatedButtonEvent_PressPowerButtonMiddle:
        {
            m_WindowManager.NotifyPowerButtonMiddlePressed();
            NN_RESULT_SUCCESS;
        }
        case EmulatedButtonEvent_PressPowerButtonLong:
        {
            m_WindowManager.NotifyPowerButtonLongPressed();
            NN_RESULT_SUCCESS;
        }
        default:
            NN_RESULT_THROW(ResultInvalidCall());
    }
}

Result AppletSystem::InvalidateTransitionLayer() NN_NOEXCEPT
{
    am::service::SetSystemLayerVisible(false);
    NN_RESULT_SUCCESS;
}

// シャットダウン開始（SA がいれば SA にシャットダウンを要求）
Result AppletSystem::RequestToShutdown() NN_NOEXCEPT
{
    auto pSystemApplet = this->GetTheSystemApplet();
    if (pSystemApplet)
    {
        auto p = static_cast<IntegratedApplet*>(pSystemApplet.get());
        if (p->IsSelfProxyAvailable())
        {
            p->PushMessageToProcess(applet::Message_RequestToShutdown);
            NN_RESULT_SUCCESS;
        }
    }
    return this->GetGlobalStateManager()->StartShutdownSequence();
}

// 再起動開始（SA がいれば SA に再起動を要求）
Result AppletSystem::RequestToReboot() NN_NOEXCEPT
{
    auto pSystemApplet = this->GetTheSystemApplet();
    if (pSystemApplet)
    {
        auto p = static_cast<IntegratedApplet*>(pSystemApplet.get());
        if (p->IsSelfProxyAvailable())
        {
            p->PushMessageToProcess(applet::Message_RequestToReboot);
            NN_RESULT_SUCCESS;
        }
    }
    return this->GetGlobalStateManager()->StartRebootSequence();
}

// アプリメモリを拝借して（獲得して）から返る。
Result AppletSystem::AllocateMemoryFromApplicationResource(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT
{
    std::lock_guard<os::SdkMutex> lk(m_MemoryHeapMutex);
    NN_SDK_ASSERT(m_CurrentMemoryHeapSize == 0);

    auto success = false;
    NsBoostSystemMemoryResourceLimit(size);
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            NsBoostSystemMemoryResourceLimit(0);
        }
    };

    uintptr_t address;
    NN_RESULT_DO(os::AllocateUnsafeMemory(&address, size));

    success = true;
    this->m_CurrentMemoryHeapSize = size;
    *pOutAddress = address;
    NN_RESULT_SUCCESS;
}

// 拝借していたアプリメモリを全て返却する
void AppletSystem::FreeMemoryToApplicationResource(uintptr_t address) NN_NOEXCEPT
{
    std::lock_guard<os::SdkMutex> lk(m_MemoryHeapMutex);
    NN_SDK_ASSERT(m_CurrentMemoryHeapSize > 0);

    NN_ABORT_UNLESS_RESULT_SUCCESS(os::FreeUnsafeMemory(address, m_CurrentMemoryHeapSize));
    NsBoostSystemMemoryResourceLimit(0);
    this->m_CurrentMemoryHeapSize = 0;
}

void AppletSystem::NotifyApplicationStarted(const ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    if (auto pOverlayApplet = GetTheOverlayApplet())
    {
        pOverlayApplet->NotifyApplicationStarted(applicationId);
    }
}

void AppletSystem::NotifyApplicationRunning(const ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    if (auto pOverlayApplet = GetTheOverlayApplet())
    {
        pOverlayApplet->NotifyApplicationRunning(applicationId);
    }
}

void AppletSystem::NotifyApplicationExited(const ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    if (auto pOverlayApplet = GetTheOverlayApplet())
    {
        pOverlayApplet->NotifyApplicationExited(applicationId);
    }
}

void AppletSystem::NotifyApplicationSuspendedByRightsError() NN_NOEXCEPT
{
    if (auto pSystemApplet = GetTheSystemApplet())
    {
        pSystemApplet->PushMessageToProcess(applet::Message_ApplicationSuspendedByRightsError);
    }
}

void AppletSystem::NotifyControllerFirmwareUpdateSectionChanged() NN_NOEXCEPT
{
    if (auto pOverlayApplet = GetTheOverlayApplet())
    {
        pOverlayApplet->NotifyControllerFirmwareUpdateSectionChanged();
    }
    if (auto pSystemApplet = GetTheSystemApplet())
    {
        pSystemApplet->NotifyControllerFirmwareUpdateSectionChanged();
    }
}

void AppletSystem::NotifyHomeButtonLongPressedToOverlayApplet() NN_NOEXCEPT
{
    if (auto pOverlayApplet = GetTheOverlayApplet())
    {
        pOverlayApplet->OnHomeButtonLongPressed();
    }
}

void AppletSystem::TerminateCurrentApplicationAndSetResult(Result result) NN_NOEXCEPT
{
    if (auto p = GetLastCreatedApplication())
    {
        p->TerminateAndSetResult(result);
    }
}

bool AppletSystem::AreAnyApplicationActive() NN_NOEXCEPT
{
    auto p = GetLastCreatedApplication();
    return p && (!p->IsDormant());
}

// SIGLO-79632
void AppletSystem::SetOperationMode(omm::OperationMode mode) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_OperationModeMutex)> lock(m_OperationModeMutex);
    this->m_pOperationMode = mode;
}

// SIGLO-79632:
//  未登録時は自力で omm から取得したものを返す
omm::OperationMode AppletSystem::GetOperationMode() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_OperationModeMutex)> lock(m_OperationModeMutex);
    if (!m_pOperationMode)
    {
        this->m_pOperationMode = NN_AM_SERVICE_STUCK_CHECKED(omm_Get, 60, nn::omm::GetOperationMode());
    }
    return *m_pOperationMode;
}

}}}
