﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_AppletService.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/nn_Abort.h>
#include <nn/am/service/am_AppletProxy.h>
#include <utility>
#include <nn/am/am_ResultPrivate.h>
#include <mutex>
#include <nn/am/service/am_IntegratedSystemApplet.h>
#include <nn/am/service/am_ServiceConfig.h>

#include "am_EventHandler.h"
#include "am_ApplicationTimer.h"
#include "am_OmmEvent.h"
#include "../../../Processes/am/am_OperationModeManagerMain.h"

namespace nn { namespace am { namespace service {

AppletSystem& AppletService::GetAppletSystem() NN_NOEXCEPT
{
    return m_AppletSystem;
}

template <typename Interface, typename AppletCreator>
inline Result AppletService::OpenProxyImpl(sf::Out<nn::sf::SharedPointer<Interface>> pOut, Bit64 processId, sf::NativeHandle processHandle, const ProxyOption& proxyOption, AppletCreator&& appletCreator, const AppletAttribute* pAttribute) NN_NOEXCEPT
{
    std::shared_ptr<AppletProxy> p;
    NN_RESULT_TRY(GetAppletSystem().GetAppletProxy(&p, processId, std::move(processHandle), proxyOption, pAttribute))
        NN_RESULT_CATCH(ResultNoAppletProxy)
        {
            NN_AM_SERVICE_LOG(dev, "cannot find AppletProxy for %llu creating...\n", processId);
            // 取得できなかった際には、appletCreator を呼んでアプレット作成要求を出し、一旦お帰り願う。
            // クライアントは再度、この取得を試みることを想定している。
            std::shared_ptr<core::Applet> pApplet;
            NN_RESULT_DO(appletCreator(&pApplet));
            if (pApplet)
            {
                this->GetAppletSystem().RegisterAppletProxy(processId, nullptr);
                NN_AM_SERVICE_LOG(dev, "created AppletProxy for %llu\n", processId);
            }
            else
            {
                NN_AM_SERVICE_LOG(dev, "failed to create AppletProxy for %llu\n", processId);
            }
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    NN_RESULT_THROW_UNLESS(p, ResultNoAppletProxy()); // 登録が間に合っていないのでリトライ指示
    *pOut = sf::CreateSharedObject<Interface>(std::move(p));
    NN_AM_SERVICE_LOG(seq, "return proxy for %llu\n", processId);
    NN_RESULT_SUCCESS;
}

namespace {

inline ProxyOption MakeProxyOptionForOe() NN_NOEXCEPT
{
    ProxyOption ret = {};
    ret.forOe = true;
    return ret;
}

inline ProxyOption MakeProxyOptionForAe() NN_NOEXCEPT
{
    ProxyOption ret = {};
    ret.forOe = false;
    return ret;
}

}

nn::Result AppletService::OpenApplicationProxy(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IApplicationProxy>> pOut, nn::Bit64 processId, sf::NativeHandle processHandle) NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(call, "OpenApplicationProxy for %llu\n", processId);
    return OpenProxyImpl(pOut, processId, std::move(processHandle), MakeProxyOptionForOe(), [this, processId] (std::shared_ptr<core::Applet>* p)
    {
        // TORIAEZU:
        //  フローティング起動時は AppletId_None および InvalidId を入れておく
        applet::ApplicationLaunchRequestInfo launchRequestInfo = {};
        return this->GetAppletSystem().CreateApplicationForRunningProcess(p, {processId}, launchRequestInfo);
    });
}

nn::Result AppletService::OpenSystemApplicationProxy(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IApplicationProxy>> pOut, nn::Bit64 processId, sf::NativeHandle processHandle) NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(call, "OpenSystemApplicationProxy for %llu\n", processId);
    return OpenProxyImpl(pOut, processId, std::move(processHandle), MakeProxyOptionForAe(), [this, processId] (std::shared_ptr<core::Applet>* p)
    {
        return this->GetAppletSystem().CreateSystemApplicationForRunningProcess(p, {processId});
    });
}

nn::Result AppletService::OpenSystemAppletProxy(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ISystemAppletProxy>> pOut, nn::Bit64 processId, sf::NativeHandle processHandle) NN_NOEXCEPT
{
    // スタートアップロゴを消す（既に消えている場合は何もしない）
    HideStartupLogoOfOperationModeManager();

    NN_AM_SERVICE_LOG(call, "OpenSystemAppletProxy for %llu\n", processId);
    return OpenProxyImpl(pOut, processId, std::move(processHandle), MakeProxyOptionForAe(), [this, processId] (std::shared_ptr<core::Applet>* p)
    {
        return this->GetAppletSystem().CreateSystemAppletForRunningProcess(p, applet::AppletId_SystemAppletMenu, {processId});
    });
}

nn::Result AppletService::OpenLibraryAppletProxy(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ILibraryAppletProxy>> pOut, nn::Bit64 processId, sf::NativeHandle processHandle, const AppletAttribute& attribute) NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(call, "OpenLibraryAppletProxy for %llu\n", processId);
    return OpenProxyImpl(pOut, processId, std::move(processHandle), MakeProxyOptionForAe(), [] (std::shared_ptr<core::Applet>* p)
    {
        // LA は floating 起動は廃止
        *p = nullptr;
        NN_RESULT_SUCCESS;
    }, &attribute);
}

nn::Result AppletService::OpenLibraryAppletProxyOld(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ILibraryAppletProxy>> pOut, nn::Bit64 processId, sf::NativeHandle processHandle) NN_NOEXCEPT
{
    AppletAttribute attribute = {};
    return OpenLibraryAppletProxy(pOut, processId, std::move(processHandle), attribute);
}

nn::Result AppletService::OpenOverlayAppletProxy(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IOverlayAppletProxy>> pOut, nn::Bit64 processId, sf::NativeHandle processHandle) NN_NOEXCEPT
{
    // スタートアップロゴを消す（既に消えている場合は何もしない）
    HideStartupLogoOfOperationModeManager();

    NN_AM_SERVICE_LOG(call, "OpenOverlayAppletProxy for %llu\n", processId);
    return OpenProxyImpl(pOut, processId, std::move(processHandle), MakeProxyOptionForAe(), [this, processId] (std::shared_ptr<core::Applet>* p)
    {
        return this->GetAppletSystem().CreateOverlayAppletForRunningProcess(p, applet::AppletId_OverlayApplet, {processId});
    });
}

void AppletService::EventHandlerLoopAuto() NN_NOEXCEPT
{
    // TORIAEZU: TODO:
    // HOME ボタンや POWER ボタンなどのイベントハンドリングと
    // それに伴なうメッセージ通知などの処理を行う。
    NN_AM_SERVICE_DIAGNOSTICS_SET_CONTEXT_NAME(EventObserver);

    GetApplicationTimer().SetAppletSystem( &this->GetAppletSystem() );
    GetOmmEvent().SetAppletSystem( &this->GetAppletSystem() );

    EventHandlerLoopAutoImpl( &GetAppletSystem() );
}

void AppletService::RunWindowManager() NN_NOEXCEPT
{
    NN_AM_SERVICE_DIAGNOSTICS_SET_CONTEXT_NAME(WindowSystem);
    GetAppletSystem().GetGlobalStateManager()->NotifyStart();
    GetAppletSystem().GetWindowManager()->Run();
}

nn::Result AppletService::CreateSelfLibraryAppletCreatorForDevelop(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ILibraryAppletCreator>> pOut, nn::Bit64 processId) NN_NOEXCEPT
{
    return this->GetAppletSystem().CreateSelfLibraryAppletCreatorForDevelop(pOut, {processId});
}

nn::Result AppletService::GetSystemAppletControllerForDebug(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ISystemAppletControllerForDebug>> pOut) NN_NOEXCEPT
{
    if (!IsDevelopmentFunctionEnabled())
    {
        // 開発時のみ使用可能
        *pOut = nullptr;
        NN_RESULT_SUCCESS;
    }
    auto p = this->GetAppletSystem().GetTheSystemApplet();
    if (!p)
    {
        // SA が起動していない
        *pOut = nullptr;
        NN_RESULT_SUCCESS;
    }
    *pOut = service::SfObjectFactory::CreateShared<nn::am::service::ISystemAppletControllerForDebug>(std::move(p));
    NN_RESULT_SUCCESS;
}


}}}
