﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <mutex>
#include <atomic>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/util/util_FormatString.h>

#include <nn/os.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MemoryFence.h>

#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/dd.h>
#include <nn/ahid/ahid.h>
#include <nn/ahid/ahid_Result.public.h>
#include "ahid_CreateClient.h"

namespace nn {
namespace ahid {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result Ahid::Initialize(nn::ahid::hdr::DeviceParameters* pDeviceParameters) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pDeviceParameters);
    NN_SDK_ASSERT(!m_ServerSession);
    NN_SDK_ASSERT(!m_CtrlSession);
    NN_SDK_ASSERT(!m_ReadSession);

    // Establish session to AHID server
    Result result = CreateClientByHipc(&m_ServerSession, const_cast<char*>(reinterpret_cast<char*>(pDeviceParameters->servicePath)));

    if (result.IsSuccess())
    {
        NN_SDK_ASSERT(m_ServerSession);

        // Acquire the device handle and get session for making transactions.
        result = m_ServerSession->AcquireDevice(pDeviceParameters->deviceHandle);

        if (result.IsSuccess())
        {
            GetSessions(pDeviceParameters);
        }
        else
        {
            m_ServerSession = NULL;
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::InitializeWith(nn::sf::SharedPointer<nn::ahid::IServerSession> pServerSession, nn::ahid::hdr::DeviceParameters* pDeviceParameters) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pDeviceParameters);
    NN_SDK_ASSERT(!m_ServerSession);
    NN_SDK_ASSERT(!m_CtrlSession);
    NN_SDK_ASSERT(!m_ReadSession);

    m_ServerSession = pServerSession;

    NN_SDK_ASSERT(m_ServerSession);

    // Acquire the device handle and get session for making transactions.
    Result result = m_ServerSession->AcquireDevice(pDeviceParameters->deviceHandle);

    if (result.IsSuccess())
    {
        GetSessions(pDeviceParameters);
    }
    else
    {
        m_ServerSession = NULL;
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_ServerSession);
    NN_SDK_ASSERT(m_CtrlSession);
    NN_SDK_ASSERT(m_ReadSession);

    m_CtrlSession   = nullptr;
    m_ReadSession   = nullptr;

    Result result = m_ServerSession->ReleaseDevice(m_DeviceParameters.deviceHandle);

    m_ServerSession = nullptr;
    //nn::sf::ReleaseSharedObject(reinterpret_cast<nn::sf::ISharedObject>(m_ServerSession));

    memset(&m_DeviceParameters, 0, sizeof(m_DeviceParameters));

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetDeviceParameters(nn::ahid::hdr::DeviceParameters *pDeviceParameters) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pDeviceParameters);

    memcpy(pDeviceParameters, &m_DeviceParameters, sizeof(nn::ahid::hdr::DeviceParameters));

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::CreateStateChangeEvent(nn::os::SystemEventType* pOutEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
{
    nn::sf::NativeHandle sfEventHandle;

    Result result = m_CtrlSession->GetStateChangeEvent(&sfEventHandle);

    if(result.IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pOutEvent, sfEventHandle.GetOsHandle(), sfEventHandle.IsManaged(), eventClearMode);
        sfEventHandle.Detach();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::DestroyStateChangeEvent(nn::os::SystemEventType* pInEvent)
{
    nn::os::DestroySystemEvent(pInEvent);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetString(void* pOutBuffer, size_t bufferLength, uint8_t stringIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);
    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), bufferLength);
    return m_CtrlSession->GetString(stringIndex, buffer);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetReport(void* pOutBuffer, size_t bufferLength, uint8_t reportType, uint8_t reportId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);
    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), bufferLength);
    return m_CtrlSession->GetReport(reportType, reportId, buffer);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::SetReport(void* pBuffer, size_t bufferLength,uint8_t reportType, uint8_t reportId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);
    nn::sf::InBuffer buffer(reinterpret_cast<char*>(pBuffer), bufferLength);
    return m_CtrlSession->SetReport(reportType, reportId, buffer);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetIdle(uint8_t* pOutIdle, uint8_t reportId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);
    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutIdle), 1);
    return m_CtrlSession->GetIdle(buffer, reportId);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::SetIdle(uint8_t idle, uint8_t reportId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);

    return m_CtrlSession->SetIdle(idle, reportId);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetProtocol(uint8_t* pOutProtocol) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);
    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutProtocol), 1);
    return m_CtrlSession->GetProtocol(buffer);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::SetProtocol(uint8_t protocol) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);

    return m_CtrlSession->SetProtocol(protocol);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetDescriptor(void* pOutBuffer, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex, uint16_t wLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);
    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), wLength);
    return m_CtrlSession->GetDescriptor(buffer, bmRequestType, wValue, wIndex);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::SetDescriptor(void* pInBuffer, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex, uint16_t wLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);
    nn::sf::InBuffer buffer(reinterpret_cast<char*>(pInBuffer), wLength);
    return m_CtrlSession->SetDescriptor(buffer, bmRequestType, wValue, wIndex);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::Read(uint32_t* pOutBytesRead, void* pOutBuffer, size_t bufferLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_ReadSession);
    nn::TimeSpan timeout = 0;

    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), bufferLength);
    return m_ReadSession->Read(buffer, pOutBytesRead, timeout);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::ReadWithTimeout(uint32_t* pOutBytesRead, void* pOutBuffer, size_t bufferLength, nn::TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_ReadSession);

    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), bufferLength);
    return m_ReadSession->Read(buffer, pOutBytesRead, timeout);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::Write(uint32_t* pOutBytesWritten, void* pBuffer, size_t bufferLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);
    nn::TimeSpan timeout = 0;

    nn::sf::InBuffer buffer(reinterpret_cast<char*>(pBuffer), bufferLength);
    return m_CtrlSession->Write(buffer, pOutBytesWritten, timeout);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::WriteWithTimeout(uint32_t* pOutBytesWritten, void* pBuffer, size_t bufferLength, nn::TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CtrlSession);

    nn::sf::InBuffer buffer(reinterpret_cast<char*>(pBuffer), bufferLength);
    return m_CtrlSession->Write(buffer, pOutBytesWritten, timeout);
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetCodeBookHeader(CodeBookHeader** ppCodeBookHeader) NN_NOEXCEPT
{
    *ppCodeBookHeader = reinterpret_cast<CodeBookHeader*>(m_CodeBook);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetInputItem(Item** ppOutItem, uint8_t reportId, uint8_t usagePage, uint8_t usageMin, uint8_t usageMax, uint8_t index) NN_NOEXCEPT
{
    CodeBookHeader *pCodeBookHeader = reinterpret_cast<CodeBookHeader*>(m_CodeBook);
    uint8_t        *p               = reinterpret_cast<uint8_t*>(&m_CodeBook[sizeof(CodeBookHeader)]);

    *ppOutItem = GetItem(
                            p,
                            pCodeBookHeader->inputItems,
                            reportId,
                            usagePage,
                            usageMin,
                            usageMax,
                            index
                            );

    if (*ppOutItem)
    {
        return ResultSuccess();
    }

    return ResultItemNotFound();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetOutputItem(Item** ppOutItem, uint8_t reportId, uint8_t usagePage, uint8_t usageMin, uint8_t usageMax, uint8_t index) NN_NOEXCEPT
{
    CodeBookHeader *pCodeBookHeader = reinterpret_cast<CodeBookHeader*>(m_CodeBook);
    uint8_t        *p               = reinterpret_cast<uint8_t*>(&m_CodeBook[sizeof(CodeBookHeader)]);

    p = SkipItems(p, pCodeBookHeader->inputItems);

    *ppOutItem = GetItem(
                            p,
                            pCodeBookHeader->outputItems,
                            reportId,
                            usagePage,
                            usageMin,
                            usageMax,
                            index
                            );

    if (*ppOutItem)
    {
        return ResultSuccess();
    }

    return ResultItemNotFound();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetFeatureItem(Item** ppOutItem, uint8_t reportId, uint8_t usagePage, uint8_t usageMin, uint8_t usageMax, uint8_t index) NN_NOEXCEPT
{
    CodeBookHeader *pCodeBookHeader = reinterpret_cast<CodeBookHeader*>(m_CodeBook);
    uint8_t        *p               = reinterpret_cast<uint8_t*>(&m_CodeBook[sizeof(CodeBookHeader)]);

    p = SkipItems(p, pCodeBookHeader->inputItems);
    p = SkipItems(p, pCodeBookHeader->outputItems);

    *ppOutItem = GetItem(
                            p,
                            pCodeBookHeader->featureItems,
                            reportId,
                            usagePage,
                            usageMin,
                            usageMax,
                            index
                            );

    if (*ppOutItem)
    {
        return ResultSuccess();
    }

    return ResultItemNotFound();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::Decode(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT
{
    Result result = ResultInvalidParameter();

    if (pOutData)
    {
        *pOutData = 0; // initialize to 0 incase other params are invalid :0

        if (pInputData && pItem)
        {
            switch (pItem->type)
            {
            case DataType::DataType_Bitmask8:

                result = DecodeBitmask8(pOutData, pInputData, pItem);

                break;

            case DataType::DataType_Logical8Mask:

                result = DecodeLogical8Mask(pOutData, pInputData, pItem);

                break;

            case DataType::DataType_Logical8:

                result = DecodeLogical8(pOutData, pInputData, pItem);

                break;

            case DataType::DataType_Logical16Mask:

                result = DecodeLogical16Mask(pOutData, pInputData, pItem);

                break;

            case DataType::DataType_Logical16:

                result = DecodeLogical16(pOutData, pInputData, pItem);

                break;

            case DataType::DataType_Physical8Mask:

                result = DecodePhysical8Mask(pOutData, pInputData, pItem);

                break;

            case DataType::DataType_Physical8:

                result = DecodePhysical8(pOutData, pInputData, pItem);

                break;

            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::Encode(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT
{
    Result result = ResultInvalidParameter();

    if (pData && pOutputData && pItem)
    {
        switch (pItem->type)
        {
        case DataType::DataType_Bitmask8:

            result = EncodeBitmask8(pData, pOutputData, pItem);

            break;

        case DataType::DataType_Logical8Mask:

            result = EncodeLogical8Mask(pData, pOutputData, pItem);

            break;

        case DataType::DataType_Logical8:

            result = EncodeLogical8(pData, pOutputData, pItem);

            break;

        case DataType::DataType_Logical16Mask:

            result = EncodeLogical16Mask(pData, pOutputData, pItem);

            break;

        case DataType::DataType_Logical16:

            result = EncodeLogical16(pData, pOutputData, pItem);

            break;

        case DataType::DataType_Physical8Mask:

            result = EncodePhysical8Mask(pData, pOutputData, pItem);

            break;

        case DataType::DataType_Physical8:

            result = EncodePhysical8(pData, pOutputData, pItem);

            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetUnit(int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t *pExponent, Item* pItem) NN_NOEXCEPT
{
    if (pUnit && pItem)
    {
        switch (pItem->type)
        {
        case DataType::DataType_Bitmask8:

            GetBitmask8Context(pMinimum, pMaximum, pUnit, pExponent, pItem);

            break;

        case DataType::DataType_Logical8Mask:

            GetLogical8MaskContext(pMinimum, pMaximum, pUnit, pExponent, pItem);

            break;

        case DataType::DataType_Logical8:

            GetLogical8Context(pMinimum, pMaximum, pUnit, pExponent, pItem);

            break;

        case DataType::DataType_Logical16Mask:

            GetLogical16MaskContext(pMinimum, pMaximum, pUnit, pExponent, pItem);

            break;

        case DataType::DataType_Logical16:

            GetLogical16Context(pMinimum, pMaximum, pUnit, pExponent, pItem);

            break;

        case DataType::DataType_Physical8Mask:

            GetPhysical8MaskContext(pMinimum, pMaximum, pUnit, pExponent, pItem);

            break;

        case DataType::DataType_Physical8:

            GetPhysical8Context(pMinimum, pMaximum, pUnit, pExponent, pItem);

            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        return ResultSuccess();
    }

    return ResultInvalidParameter();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetReportCount(uint8_t *pOutReportCount) NN_NOEXCEPT
{
    CodeBookHeader *pCodeBookHeader = reinterpret_cast<CodeBookHeader*>(m_CodeBook);

    *pOutReportCount = pCodeBookHeader->reportCount;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::GetReportIdUsage(uint8_t *pOutReportId, uint8_t *pOutUsagePage, uint8_t *pOutUsage, int index) NN_NOEXCEPT
{
    Result result;
    CodeBookHeader *pCodeBookHeader = reinterpret_cast<CodeBookHeader*>(m_CodeBook);

    if (index < pCodeBookHeader->reportCount)
    {
        *pOutReportId   = pCodeBookHeader->reportId[index];
        *pOutUsagePage  = pCodeBookHeader->usagePage[index];
        *pOutUsage      = pCodeBookHeader->usage[index];

        result = ResultSuccess();
    }
    else
    {
        *pOutReportId   = 0;
        *pOutUsagePage  = 0;
        *pOutUsage      = 0;

        result = ResultInvalidParameter();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
void Ahid::GetSessions(nn::ahid::hdr::DeviceParameters* pDeviceParameters) NN_NOEXCEPT
{
    m_ServerSession->GetCtrlSession(pDeviceParameters->deviceHandle, &m_CtrlSession);
    m_ServerSession->GetReadSession(pDeviceParameters->deviceHandle, &m_ReadSession);

    NN_SDK_ASSERT(m_CtrlSession);
    NN_SDK_ASSERT(m_ReadSession);

    memcpy(&m_DeviceParameters, pDeviceParameters, sizeof(m_DeviceParameters));

    // get the code book
    nn::sf::OutBuffer buffer(reinterpret_cast<char*>(m_CodeBook), sizeof(m_CodeBook));
    m_CtrlSession->GetCodeBook(buffer);
}


//////////////////////////////////////////////////////////////////////////////
uint8_t* Ahid::SkipItems(uint8_t* p, uint32_t items) NN_NOEXCEPT
{
    while (items--)
    {
        Item *pItem = reinterpret_cast<Item*>(p);

        p += pItem->length;
    }

    return p;
}


//////////////////////////////////////////////////////////////////////////////
Item* Ahid::GetItem(uint8_t* p, uint32_t entries, uint8_t reportId, uint8_t usagePage, uint8_t usageMin, uint8_t usageMax, uint8_t index) NN_NOEXCEPT
{
    while (entries--)
    {
        Item *pItem = reinterpret_cast<Item*>(p);

        if ((pItem->reportId == reportId) && (pItem->usagePage == usagePage))
        {
            // Looking for specific item
            if (usageMin == usageMax)
            {
                if (
                    (pItem->usageMin == usageMin) &&
                    (pItem->usageMax == usageMax) &&
                    (pItem->index == index)
                    )
                {
                    return pItem;
                }
            }
            else
            {
                if (
                    (pItem->usageMin != pItem->usageMax) &&
                    (pItem->usageMin <= usageMin) &&
                    (pItem->usageMax >= usageMax) &&
                    (pItem->index == index)
                    )
                {
                    return pItem;
                }
            }
        }

        p += pItem->length;
    }

    return nullptr;
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::DecodeBitmask8(int32_t* pDataOut, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT
{
    ItemBitmask8* pItemBitmask8 = reinterpret_cast<ItemBitmask8*>(pItem);

    *pDataOut = static_cast<int32_t>((pInputData[pItemBitmask8->dataOffset] & pItemBitmask8->dataBits) >> pItemBitmask8->dataShift);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::EncodeBitmask8(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT
{
    ItemBitmask8* pItemBitmask8 = reinterpret_cast<ItemBitmask8*>(pItem);

    pOutputData[pItemBitmask8->dataOffset] |= static_cast<uint8_t>((*pData << pItemBitmask8->dataShift) & pItemBitmask8->dataBits);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::DecodeLogical8Mask(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT
{
    ItemLogical8Mask *pItemLogicalMask8 = reinterpret_cast<ItemLogical8Mask*>(pItem);

    int8_t data = static_cast<int8_t>((pInputData[pItemLogicalMask8->dataOffset] & pItemLogicalMask8->dataBits) >> pItemLogicalMask8->dataShift);

    *pOutData = static_cast<int32_t>(data);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::EncodeLogical8Mask(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT
{
    ItemLogical8Mask *pItemLogicalMask8 = reinterpret_cast<ItemLogical8Mask*>(pItem);

    int8_t data = static_cast<int8_t>(*pData);

    // clear data bits
    pOutputData[pItemLogicalMask8->dataOffset] &= ~pItemLogicalMask8->dataBits;

    pOutputData[pItemLogicalMask8->dataOffset] |= static_cast<uint8_t>((data << pItemLogicalMask8->dataShift) & pItemLogicalMask8->dataBits);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::DecodeLogical8(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT
{
    ItemLogical8 *pItemLogical8 = reinterpret_cast<ItemLogical8*>(pItem);

    if (pItemLogical8->logicalMaximum < 0)  // not signed data :0
    {
        *pOutData = pInputData[pItemLogical8->dataOffset];
    }
    else
    {
        int8_t data = static_cast<int8_t>(pInputData[pItemLogical8->dataOffset]);

        *pOutData = static_cast<int32_t>(data);
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::EncodeLogical8(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT
{
    ItemLogical8 *pItemLogical8 = reinterpret_cast<ItemLogical8*>(pItem);

    int8_t data = static_cast<int8_t>(*pData);

    pOutputData[pItemLogical8->dataOffset] = static_cast<uint8_t>(data);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::DecodeLogical16Mask(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT
{
    ItemLogical16Mask *pItemLogicalMask16 = reinterpret_cast<ItemLogical16Mask*>(pItem);
    int16_t data = Read16BitData(&pInputData[pItemLogicalMask16->dataOffset]);

    data &= pItemLogicalMask16->dataBits;

    if (pItemLogicalMask16->dataShift)
    {
        data >>= pItemLogicalMask16->dataShift;
    }

    // see if we have to sign extend the number :0
    if (pItemLogicalMask16->logicalMinimum < 0)
    {
        if (data & pItemLogicalMask16->signBit)
        {
            data |= ~(pItemLogicalMask16->dataBits >> pItemLogicalMask16->dataShift);
        }
    }

    *pOutData = static_cast<int32_t>(data);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::EncodeLogical16Mask(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT
{
    ItemLogical16Mask *pItemLogicalMask16 = reinterpret_cast<ItemLogical16Mask*>(pItem);

    int16_t data        = static_cast<int16_t>(*pData);
    int16_t outputData  = Read16BitData(&pOutputData[pItemLogicalMask16->dataOffset]);

    // clear data bits
    outputData &= ~pItemLogicalMask16->dataBits;

    outputData |= (data << pItemLogicalMask16->dataShift) & pItemLogicalMask16->dataBits;

    Write16BitData(outputData, &pOutputData[pItemLogicalMask16->dataOffset]);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::DecodeLogical16(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT
{
    ItemLogical16 *pItemLogical16 = reinterpret_cast<ItemLogical16*>(pItem);

    int16_t data = Read16BitData(&pInputData[pItemLogical16->dataOffset]);

    *pOutData = static_cast<int32_t>(data);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::EncodeLogical16(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT
{
    ItemLogical16 *pItemLogical16 = reinterpret_cast<ItemLogical16*>(pItem);

    int16_t data = static_cast<int16_t>(*pData);

    Write16BitData(data, &pOutputData[pItemLogical16->dataOffset]);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::DecodePhysical8Mask(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT
{
    ItemPhysical8Mask *pItemPhysical8Mask = reinterpret_cast<ItemPhysical8Mask*>(pItem);
    int8_t inputData = pInputData[pItemPhysical8Mask->dataOffset];
    int32_t outputData;

    inputData &= pItemPhysical8Mask->dataBits;
    inputData >>= pItemPhysical8Mask->dataShift;

    outputData = inputData * (pItemPhysical8Mask->physicalMaximum / pItemPhysical8Mask->logicalMaximum);

    *pOutData = outputData;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::EncodePhysical8Mask(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT
{
    ItemPhysical8Mask *pItemPhysical8Mask = reinterpret_cast<ItemPhysical8Mask*>(pItem);

    int32_t inputData   = *pData;
    int8_t tempData;
    int8_t outputData   = pOutputData[pItemPhysical8Mask->dataOffset];

    tempData = static_cast<int8_t>(inputData / (pItemPhysical8Mask->physicalMaximum / pItemPhysical8Mask->logicalMaximum));

    // clear data bits
    outputData &= ~pItemPhysical8Mask->dataBits;
    outputData |= (tempData << pItemPhysical8Mask->dataShift) & pItemPhysical8Mask->dataBits;

    pOutputData[pItemPhysical8Mask->dataOffset] = outputData;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::DecodePhysical8(int32_t* pOutData, uint8_t* pInputData, Item* pItem) NN_NOEXCEPT
{
    ItemPhysical8 *pItemPhysical8 = reinterpret_cast<ItemPhysical8*>(pItem);
    int8_t inputData = pInputData[pItemPhysical8->dataOffset];
    int32_t outputData = inputData * pItemPhysical8->physicalMaximum / pItemPhysical8->logicalMaximum;

    *pOutData = outputData;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Ahid::EncodePhysical8(int32_t* pData, uint8_t* pOutputData, Item* pItem) NN_NOEXCEPT
{
    ItemPhysical8 *pItemPhysical8 = reinterpret_cast<ItemPhysical8*>(pItem);

    int32_t inputData = *pData;
    int8_t outputData;

    outputData = static_cast<int8_t>(inputData / (pItemPhysical8->physicalMaximum / pItemPhysical8->logicalMaximum));

    pOutputData[pItemPhysical8->dataOffset] = outputData;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
void Ahid::GetBitmask8Context(int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT
{
    ItemBitmask8* pItemBitmask8 = reinterpret_cast<ItemBitmask8*>(pItem);

    *pMinimum   = 0;
    *pMaximum   = pItemBitmask8->dataBits >> pItemBitmask8->dataShift;
    *pUnit      = 0;
    *pExponent  = 0;
}


//////////////////////////////////////////////////////////////////////////////
void Ahid::GetLogical8MaskContext(int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT
{
    ItemLogical8Mask* pItemLogical8Mask = reinterpret_cast<ItemLogical8Mask*>(pItem);

    *pMinimum   = pItemLogical8Mask->logicalMinimum;
    *pMaximum   = pItemLogical8Mask->logicalMaximum;
    *pUnit      = 0;
    *pExponent  = 0;
}


//////////////////////////////////////////////////////////////////////////////
void Ahid::GetLogical8Context(int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT
{
    ItemLogical8* pItemLogical8 = reinterpret_cast<ItemLogical8*>(pItem);

    *pMinimum   = pItemLogical8->logicalMinimum;
    *pMaximum   = pItemLogical8->logicalMaximum;
    *pUnit      = 0;
    *pExponent  = 0;
}


//////////////////////////////////////////////////////////////////////////////
void Ahid::GetLogical16MaskContext(int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT
{
    ItemLogical16Mask* pItemLogical16Mask = reinterpret_cast<ItemLogical16Mask*>(pItem);

    *pMinimum   = pItemLogical16Mask->logicalMinimum;
    *pMaximum   = pItemLogical16Mask->logicalMaximum;
    *pUnit      = 0;
    *pExponent  = 0;
}


//////////////////////////////////////////////////////////////////////////////
void Ahid::GetLogical16Context(int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT
{
    ItemLogical16* pItemLogical16 = reinterpret_cast<ItemLogical16*>(pItem);

    *pMinimum   = pItemLogical16->logicalMinimum;
    *pMaximum   = pItemLogical16->logicalMaximum;
    *pUnit      = 0;
    *pExponent  = 0;
}


//////////////////////////////////////////////////////////////////////////////
void Ahid::GetPhysical8MaskContext(int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT
{
    ItemPhysical8Mask* pItemPhysical8Mask = reinterpret_cast<ItemPhysical8Mask*>(pItem);

    *pMinimum   = pItemPhysical8Mask->physicalMinimum;
    *pMaximum   = pItemPhysical8Mask->physicalMaximum;
    *pUnit      = pItemPhysical8Mask->unit;
    *pExponent  = pItemPhysical8Mask->unitExponont;
}


//////////////////////////////////////////////////////////////////////////////
void Ahid::GetPhysical8Context(int32_t* pMinimum, int32_t* pMaximum, uint32_t* pUnit, uint32_t* pExponent, Item* pItem) NN_NOEXCEPT
{
    ItemPhysical8* pItemPhysical8 = reinterpret_cast<ItemPhysical8*>(pItem);

    *pMinimum   = pItemPhysical8->physicalMinimum;
    *pMaximum   = pItemPhysical8->physicalMaximum;
    *pUnit      = pItemPhysical8->unit;
    *pExponent  = pItemPhysical8->unitExponont;
}


//////////////////////////////////////////////////////////////////////////////
int16_t Ahid::Read16BitData(uint8_t* pInputData) NN_NOEXCEPT
{
    return static_cast<int16_t>((static_cast<int8_t>(pInputData[1]) << 8) | (pInputData[0]));
}


//////////////////////////////////////////////////////////////////////////////
void Ahid::Write16BitData(int16_t inputData, uint8_t* pOutputData) NN_NOEXCEPT
{
    *pOutputData++  = static_cast<uint8_t>(inputData >> 8);
    *pOutputData    = static_cast<uint8_t>(inputData & 0xff);
}


} // end of namespace ahid
} // end of namespace nn

