﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ae/ae_OverlayAppletApi.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/am/am_Shim.h>
#include <nn/ns/ns_Result.h>

#include <nn/applet/applet_Types.h>
#include <nn/applet/applet_Result.h>

#include <nn/ae/ae_Result.h>
#include <nn/ae/ae_Types.h>

#include <nn/os/os_Mutex.h>
#include <nv/nv_ServiceName.h>

#include <mutex>


namespace nn { namespace ae {

namespace {

    OverlayAppletParameters  g_OverlayAppletParameters;

    uint64_t g_ScreenShotCaptureForbiddenSectionCount{0};
    os::Mutex g_Mutex{false};

}   // namespace anonymous

//-----------------------------------------------------------------------------
//  オーバレイアプレットの初期化
//
void InvokeOverlayAppletMain(AppletId appletId, OverlayAppletMainFunction function) NN_NOEXCEPT
{
    nv::SetGraphicsServiceName("nvdrv:a");

    // TORIAEZU: TODO:
    // 既存 SA がいる場合に、その SA を終了してから自身の初期化を完了させる。
    // この処理は Proxy 側で行なうべきで、ここにあったコードは削除済みだが、
    // Proxy 側の対応がなされるまで、あえてコメントを残しておく。
    am::InitializeOverlayAppletInterfaces();

    // アプレットパラメータの設定
    std::memset(&g_OverlayAppletParameters, 0, sizeof(g_OverlayAppletParameters));

    // アプレットのメイン関数の呼出し。
    // この後、FG 起動なら BG->FG 遷移要求が自動的に通知されてくる
    function( &g_OverlayAppletParameters );

    // 終了処理
    am::FinalizeOverlayAppletInterfaces();
}

//-----------------------------------------------------------------------------
//  オーバレイアプレット用のユーザインタラクション開始
//
void BeginOverlayUserInteraction() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->BeginToWatchShortHomeButtonMessage());
}


//-----------------------------------------------------------------------------
//  オーバレイアプレット用のユーザインタラクション終了
//
void EndOverlayUserInteraction() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->EndToWatchShortHomeButtonMessage());
}

//-----------------------------------------------------------------------------
//  オーバレイアプレット用のユーザインタラクション監視開始
//  DevOverlayDisp のデバッグ表示機能用
//
void BeginOverlayUserInteractionObservationForDevelop() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->BeginToObserveHidInputForDevelop());
}

//-----------------------------------------------------------------------------
//  オーバレイアプレット用の APD 設定時間の有効無効設定
//
void EnableAutoSleepTimeAndDimmingTime() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->SetAutoSleepTimeAndDimmingTimeEnabled(true));
}

void DisableAutoSleepTimeAndDimmingTime() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->SetAutoSleepTimeAndDimmingTimeEnabled(false));
}

//-----------------------------------------------------------------------------
//  スクリーンショット撮影機能のシステムレベルでの禁止／許可
//
void BeginScreenShotCaptureForbiddenSection() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(g_Mutex);
    if (g_ScreenShotCaptureForbiddenSectionCount == 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->SetScreenShotPermissionGlobally(false));
    }
    ++g_ScreenShotCaptureForbiddenSectionCount;
}

void EndScreenShotCaptureForbiddenSection() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(g_Mutex);
    NN_SDK_ASSERT(g_ScreenShotCaptureForbiddenSectionCount > 0);
    --g_ScreenShotCaptureForbiddenSectionCount;
    if (g_ScreenShotCaptureForbiddenSectionCount == 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->SetScreenShotPermissionGlobally(true));
    }
}


ncm::ApplicationId GetApplicationIdForLogo() NN_NOEXCEPT
{
    ncm::ApplicationId ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->GetApplicationIdForLogo(&ret));
    return ret;
}

void TerminateApplicationAndSetReason(Result result) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->TerminateApplicationAndSetReason(result.GetInnerValueForDebug()));
}

void SetGpuTimeSliceBoost(TimeSpan timeSpan) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetOverlayFunctions()->SetGpuTimeSliceBoost(timeSpan.GetNanoSeconds()));
}

}}
