﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ae/ae_LibraryAppletApi.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <utility>
#include <nn/am/am_Shim.h>
#include <nn/am/service/am_CommonTypes.h>
#include <nn/am/am_Result.h>
#include <nn/ae/ae_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/applet/applet_LibraryAppletSelf.h>
#include <nn/applet/applet_Storage.h>
#include <nn/os/os_Thread.h>

#include <nv/nv_ServiceName.h>


namespace nn { namespace ae {

// 実装的に LA に近いため、一旦ここに置いておく
void InitializeAsSystemApplication() NN_NOEXCEPT
{
    nv::SetGraphicsServiceName("nvdrv:a");

    am::InitializeSystemApplicationInterfaces();
}

namespace {

applet::LibraryAppletSelfHandle g_LibraryAppletSelfHandle = applet::InvalidLibraryAppletSelfHandle;
LibraryAppletStartHook g_LibraryAppletStartHook;

// utility
template <typename I>
sf::SharedPointer<I> MakeSharedWithAddReference(I* p) NN_NOEXCEPT
{
    return sf::SharedPointer<I>(p, true);
}


bool LibraryAppletStartHookForApplet(sf::SharedPointer<am::service::ILibraryAppletAccessor> p, applet::AppletId appletId, applet::LibraryAppletMode libraryAppletMode, void* userArgument) NN_NOEXCEPT
{
    if (!g_LibraryAppletStartHook)
    {
        return false;
    }
    switch (g_LibraryAppletStartHook(appletId, libraryAppletMode, userArgument))
    {
        case LibraryAppletStartHookResult_Normal:
        {
            return false;
        }
        case LibraryAppletStartHookResult_WindProgram:
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetProcessWindingController()->ReserveToStartAndWaitAndUnwindThis(std::move(p)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetProcessWindingController()->WindAndDoReserved());
            for (;;)
            {
                os::SleepThread(TimeSpan::FromDays(1));
            }
        }
        default: NN_UNEXPECTED_DEFAULT;
    }
}

}

namespace {

    applet::LibraryAppletHandle (*g_CreateSelfLibraryApplet)();
    applet::LibraryAppletHandle g_SelfLibraryAppletHandle;

    am::AppletAttribute g_AppletAttribute = {};

}

void SetCreateSelfLibraryAppletForDevelop(applet::LibraryAppletHandle (*createSelfLibraryApplet)()) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(createSelfLibraryApplet);
    g_CreateSelfLibraryApplet = createSelfLibraryApplet;
    am::SetLibraryAppletResolver([]()
    {
        am::InvokeWithSelfLibraryCreatorEnabled([]()
        {
            NN_SDK_ASSERT_NOT_NULL(g_CreateSelfLibraryApplet);
            applet::SetCreatingSelfLibraryApplet(true);
            auto selfLibraryAppletHandle = g_CreateSelfLibraryApplet();
            applet::SetCreatingSelfLibraryApplet(false);
            NN_SDK_REQUIRES(selfLibraryAppletHandle != applet::InvalidLibraryAppletHandle);
            g_SelfLibraryAppletHandle = selfLibraryAppletHandle;
        });
    });
}

void InvokeLibraryAppletMain(void (*libraryAppletMain)(const LibraryAppletSelfInfo& info)) NN_NOEXCEPT
{
    nv::SetGraphicsServiceName("nvdrv:a");

    applet::SetStartLibraryAppletHook(LibraryAppletStartHookForApplet);

    am::InitializeLibraryAppletInterfaces(&g_AppletAttribute);

    g_LibraryAppletSelfHandle = applet::CreateLibraryAppletSelf(MakeSharedWithAddReference(am::GetLibraryAppletSelfAccessor()), MakeSharedWithAddReference(am::GetProcessWindingController()));

    LibraryAppletSelfInfo info = {};
    am::service::LibraryAppletInfo libraryAppletInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->GetLibraryAppletInfo(&libraryAppletInfo));
    info.appletId = static_cast<decltype(info.appletId)>(libraryAppletInfo.appletId);
    info.libraryAppletMode = static_cast<decltype(info.libraryAppletMode)>(libraryAppletInfo.mode);

    am::service::AppletProcessLaunchReason launchReason;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetProcessWindingController()->GetLaunchReason(&launchReason));
    if (launchReason.unwinding)
    {
        // unwinding 起動
        info.isUnwound = true;
        sf::SharedPointer<am::service::ILibraryAppletAccessor> previousLa;
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetProcessWindingController()->OpenCallingLibraryApplet(&previousLa));
        info.previousLibraryAppletHandle = applet::CreateLibraryAppletHandleOnUnwinding(std::move(previousLa));
    }
    else
    {
        // winding 起動
        info.isUnwound = false;
        info.previousLibraryAppletHandle = applet::InvalidLibraryAppletHandle;
    }

    // メイン処理
    libraryAppletMain(info);

    ExitLibraryApplet();
}

NN_NORETURN void ExitLibraryApplet() NN_NOEXCEPT
{
    // 終了マーク
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->ExitProcessAndReturn());
    for (;;)
    {
        os::SleepThread(TimeSpan::FromDays(1));
    }
}

//typedef ae::LibraryAppletStartHookResult (*ae::LibraryAppletStartHook)(AppletId appletId, applet::LibraryAppletMode libraryAppletMode);
void SetLibraryAppletStartHook(LibraryAppletStartHook hook) NN_NOEXCEPT
{
    g_LibraryAppletStartHook = hook;
}

applet::AppletIdentityInfo GetMainAppletIdentityInfo() NN_NOEXCEPT
{
    am::service::AppletIdentityInfo info;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->GetMainAppletIdentityInfo(&info));
    applet::AppletIdentityInfo ret = {};
    ret.appletId = static_cast<decltype(ret.appletId)>(info.appletId);
    ret.applicationId = {info.applicationId};
    return ret;
}

applet::AppletIdentityInfo GetCallerAppletIdentityInfo() NN_NOEXCEPT
{
    am::service::AppletIdentityInfo info;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->GetCallerAppletIdentityInfo(&info));
    applet::AppletIdentityInfo ret = {};
    ret.appletId = static_cast<decltype(ret.appletId)>(info.appletId);
    ret.applicationId = {info.applicationId};
    return ret;
}

applet::AppletIdentityInfo GetNextReturnDestinationAppletIdentityInfo() NN_NOEXCEPT
{
    am::service::AppletIdentityInfo info;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->GetNextReturnDestinationAppletIdentityInfo(&info));
    applet::AppletIdentityInfo ret = {};
    ret.appletId = static_cast<decltype(ret.appletId)>(info.appletId);
    ret.applicationId = {info.applicationId};
    return ret;
}

Result GetMainAppletApplicationControlProperty(ns::ApplicationControlProperty* pOut) NN_NOEXCEPT
{
    return am::GetLibraryAppletSelfAccessor()->GetMainAppletApplicationControlProperty(pOut);
}

Result GetMainAppletApplicationDesiredLanguage(settings::LanguageCode* pOut) NN_NOEXCEPT
{
    auto result = am::GetLibraryAppletSelfAccessor()->GetMainAppletApplicationDesiredLanguage(pOut);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH_CONVERT(am::ResultMainAppletNotApplication, ae::ResultMainAppletNotApplication());
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

bool GetMainAppletStorageId(ncm::StorageId* pOut) NN_NOEXCEPT
{
    return am::GetLibraryAppletSelfAccessor()->GetMainAppletStorageId(pOut).IsSuccess();
}

bool CanUseApplicationCore() NN_NOEXCEPT
{
    bool canUse;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->CanUseApplicationCore(&canUse));
    return canUse;
}

void RequestToLaunchApplication(ncm::ApplicationId applicationId, const void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    applet::StorageHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(applet::CreateStorage(&handle, bufferSize));
    if (bufferSize > 0)
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);
        NN_ABORT_UNLESS_RESULT_SUCCESS(applet::WriteToStorage(handle, 0, buffer, bufferSize));
    }

    // applet::RequestToLaunchApplication() と実質的に等価だが、
    // こちらは am::GetLibraryAppletSelfAccessor() の IPC を呼ぶ。
    sf::SharedPointer<am::service::IStorage> storage(static_cast<am::service::IStorage*>(handle._p), false);
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->CreateApplicationAndPushAndRequestToLaunch(applicationId, std::move(storage)));
}

bool TryPopFromInChannel(StorageHandle* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    return applet::TryPopFromInChannel(pOut, g_LibraryAppletSelfHandle);
}

bool TryPopFromInteractiveInChannel(StorageHandle* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    return applet::TryPopFromInteractiveInChannel(pOut, g_LibraryAppletSelfHandle);
}

os::SystemEventType* GetPopFromInteractiveInChannelEvent() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    return applet::GetPopFromInteractiveInChannelEvent(g_LibraryAppletSelfHandle);
}

void PushToOutChannel(StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    return applet::PushToOutChannel(g_LibraryAppletSelfHandle, storageHandle);
}

void PushToInteractiveOutChannel(StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    return applet::PushToInteractiveOutChannel(g_LibraryAppletSelfHandle, storageHandle);
}

void PushToContextStack(StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    return applet::PushToContextStack(g_LibraryAppletSelfHandle, storageHandle);
}

bool TryPopFromContextStack(StorageHandle* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    return applet::TryPopFromContextStack(pOut, g_LibraryAppletSelfHandle);
}

void UnpopToInChannel(StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    return applet::UnpopToInChannel(g_LibraryAppletSelfHandle, storageHandle);
}

Result MapTransferStorage(void** pOutAddress, size_t* pOutSize, StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetSelfController()->EnterFatalSection() );
    auto result = applet::MapHandleStorage(pOutAddress, pOutSize, storageHandle);
    if (!result.IsSuccess())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetSelfController()->LeaveFatalSection() );
    }
    return result;
}

void UnmapTransferStorage(StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_LibraryAppletSelfHandle != applet::InvalidLibraryAppletSelfHandle);
    applet::UnmapHandleStorage(storageHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetSelfController()->LeaveFatalSection() );
}

void BeginControllerFirmwareUpdateSection() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetControllerFirmwareUpdateSection(true));
}

void EndControllerFirmwareUpdateSection() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetControllerFirmwareUpdateSection(false));
}

bool IsInControllerFirmwareUpdateSection() NN_NOEXCEPT
{
    bool isInSection;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetCommonStateGetter()->IsInControllerFirmwareUpdateSection(&isInSection));
    return isInSection;
}

vi::IndirectProducerHandleType GetIndirectLayerProducerHandle() NN_NOEXCEPT
{
    return applet::GetIndirectLayerProducerHandle(g_LibraryAppletSelfHandle);
}

void ReportVisibleError(err::ErrorCode errorCode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->ReportVisibleError(errorCode));
}

void ReportVisibleError(err::ErrorCode errorCode, const err::ErrorContext& errorContext) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->ReportVisibleErrorWithErrorContext(errorCode, errorContext));
}

//-----------------------------------------------------------------------------
// ae::InvokeLibraryAppletMain() より前に呼ぶべき関数

void SetVrModeDualScreenSupported(bool isSupported) NN_NOEXCEPT
{
    g_AppletAttribute.isVrModeDualScreenSupported = isSupported;
}

}}
