﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>
#include <atomic>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/diag/diag_AbortObserver.h>

#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/am/am_Shim.h>
#include <nn/am/am_Result.h>

#include <nn/applet/applet_InternalFunctions.h>
#include <nn/applet/applet_Types.h>
#include <nn/applet/applet_Result.h>

#include <nn/ae/ae_CommonApi.h>
#include <nn/ae/ae_Result.h>
#include <nn/ae/ae_Types.h>
#include <nn/ae/ae_OperationModeApi.h>
#include <nn/ae/ae_BootModeApi.h>

namespace nn { namespace ae {

namespace {

    Result HandleAcquireCaptureBufferResult(Result result) NN_NOEXCEPT
    {
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH( applet::ResultCaptureBufferBusy )
            {
                NN_RESULT_THROW( ae::ResultCaptureBufferBusy() );
            }
            NN_RESULT_CATCH( applet::ResultScreenShotDisabled )
            {
                NN_RESULT_THROW( ae::ResultScreenShotDisabled() );
            }
            NN_RESULT_CATCH( applet::ResultCaptureBufferOperationNotSupported )
            {
                NN_RESULT_THROW( ae::ResultNotSupported() );
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS( result );
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }

    class Capture
    {
    private:

        virtual Result Acquire(bool* pOutIsScreenShotEnabled, sf::NativeHandle* pHandle) NN_NOEXCEPT = 0;
        virtual void Release() NN_NOEXCEPT = 0;

        mutable nn::os::Mutex m_Mutex{false};
        nn::os::TransferMemoryType m_TransferMemory;
        void* m_MappedAddress{nullptr};
        bool m_IsScreenShotEnabled;
        bool m_Aborted{false};

        Result MapImpl() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
            NN_RESULT_THROW_UNLESS(!m_MappedAddress, ae::ResultCaptureBufferBusy());
            auto success = false;

            bool isScreenShotEnabled;
            sf::NativeHandle handle;
            NN_RESULT_DO(HandleAcquireCaptureBufferResult(this->Acquire(&isScreenShotEnabled, &handle)));
            NN_UTIL_SCOPE_EXIT
            {
                if (!success)
                {
                    this->Release();
                }
            };

            nn::os::AttachTransferMemory(&m_TransferMemory, CaptureBufferSize, handle.GetOsHandle(), handle.IsManaged());
            handle.Detach();
            NN_UTIL_SCOPE_EXIT
            {
                if (!success)
                {
                    nn::os::DestroyTransferMemory(&m_TransferMemory);
                }
            };

            void* address;
            NN_RESULT_DO(nn::os::MapTransferMemory(&address, &m_TransferMemory, nn::os::MemoryPermission_ReadWrite));

            this->m_IsScreenShotEnabled = isScreenShotEnabled;
            this->m_MappedAddress = address;
            success = true;
            NN_RESULT_SUCCESS;
        }

        void UnmapImpl() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Mutex.IsLockedByCurrentThread());
            nn::os::UnmapTransferMemory(&m_TransferMemory);
            nn::os::DestroyTransferMemory(&m_TransferMemory);
            this->Release();
            this->m_MappedAddress = nullptr;
        }

    public:

        Result Map() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            if (m_Aborted)
            {
                for (;;)
                {
                    os::SleepThread(TimeSpan::FromDays(1));
                }
            }
            return MapImpl();
        }

        void Unmap() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            UnmapImpl();
        }

        void UnmapOnAbort() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            if (m_MappedAddress)
            {
                UnmapImpl();
            }
            this->m_Aborted = true;
        }

        void* GetAddress() const NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            NN_SDK_REQUIRES(m_MappedAddress, "not mapped.");
            return m_MappedAddress;
        }

        bool IsScreenShotEnabled() const NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            NN_SDK_REQUIRES(m_MappedAddress, "not mapped.");
            return m_IsScreenShotEnabled;
        }

    };

    class LastApplicationCapture
        : public Capture
    {
    private:
        virtual Result Acquire(bool* pOutIsScreenShotEnabled, sf::NativeHandle* pHandle) NN_NOEXCEPT NN_OVERRIDE
        {
            return am::GetDisplayController()->AcquireLastApplicationCaptureBufferEx(pOutIsScreenShotEnabled, pHandle);
        }
        virtual void Release() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetDisplayController()->ReleaseLastApplicationCaptureBuffer());
        }
    };

    class LastForegroundCapture
        : public Capture
    {
    private:
        virtual Result Acquire(bool* pOutIsScreenShotEnabled, sf::NativeHandle* pHandle) NN_NOEXCEPT NN_OVERRIDE
        {
            return am::GetDisplayController()->AcquireLastForegroundCaptureBufferEx(pOutIsScreenShotEnabled, pHandle);
        }
        virtual void Release() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetDisplayController()->ReleaseLastForegroundCaptureBuffer());
        }
    };

    class CallerAppletCapture
        : public Capture
    {
    private:
        virtual Result Acquire(bool* pOutIsScreenShotEnabled, sf::NativeHandle* pHandle) NN_NOEXCEPT NN_OVERRIDE
        {
            return am::GetDisplayController()->AcquireCallerAppletCaptureBufferEx(pOutIsScreenShotEnabled, pHandle);
        }
        virtual void Release() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetDisplayController()->ReleaseCallerAppletCaptureBuffer());
        }
    };

    LastApplicationCapture g_LastApplicationCapture;
    LastForegroundCapture g_LastForegroundCapture;
    CallerAppletCapture g_CallerAppletCapture;

    std::atomic_bool g_IsAbortObserved;

    class AbortObserverHolderToUnmap
        : public diag::AbortObserverHolder
    {
    public:

        AbortObserverHolderToUnmap() NN_NOEXCEPT
        {
            diag::InitializeAbortObserverHolder(this, [] (const diag::AbortInfo& abortInfo)
            {
                NN_UNUSED(abortInfo);
                if (!g_IsAbortObserved.exchange(true))
                {
                    NN_SDK_LOG("WARNING: unmap all capture buffers on abort.\n");
                    g_LastApplicationCapture.UnmapOnAbort();
                    g_LastForegroundCapture.UnmapOnAbort();
                    g_CallerAppletCapture.UnmapOnAbort();
                }
            });
        }

    };

}   // namespace

//-----------------------------------------------------------------------------
//  通知メッセージ用のシステムイベント
//
void InitializeNotificationMessageEvent(os::SystemEventType* event) NN_NOEXCEPT
{
    // SystemEvent の取得
    sf::NativeHandle sfHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetCommonStateGetter()->GetEventHandle(&sfHandle) );

    os::AttachReadableHandleToSystemEvent(event,
                                          sfHandle.GetOsHandle(),
                                          sfHandle.IsManaged(),
                                          os::EventClearMode_ManualClear);
    sfHandle.Detach();
}


//-----------------------------------------------------------------------------
//  通知メッセージの取得
//
Message GetNotificationMessage() NN_NOEXCEPT
{
    am::AppletMessage message;
    NN_RESULT_TRY( am::GetCommonStateGetter()->ReceiveMessage(&message) )
        NN_RESULT_CATCH(am::ResultNoMessage)
        {
            return Message_None;
        }
    NN_RESULT_END_TRY

    return static_cast<Message>(message.message[0]);
}


//-----------------------------------------------------------------------------
//  Foreground 権限の取得
//
void AcquireForegroundRights() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetWindowController()->AcquireForegroundRights() );
}


//-----------------------------------------------------------------------------
//  Foreground 権限の開放
//
void ReleaseForegroundRights() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetWindowController()->ReleaseForegroundRights() );
}


//-----------------------------------------------------------------------------
//  通知メッセージの待機と取得（ユーティリティ）
//
Message WaitForNotificationMessage(nn::os::SystemEventType* event) NN_NOEXCEPT
{
    for (;;)
    {
        os::WaitSystemEvent(event);
        auto message = GetNotificationMessage();
        if (message != Message_None)
        {
            return message;
        }
    }
}


//-----------------------------------------------------------------------------
// アプリケーションの最後の画面バッファを取得する
//
Result GetLastApplicationCaptureBuffer(bool* pOutIsScreenShotEnabled, void* pOutBuffer, size_t dataSize) NN_NOEXCEPT
{
    sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), dataSize);
    return HandleAcquireCaptureBufferResult(am::GetDisplayController()->GetLastApplicationCaptureImageEx(pOutIsScreenShotEnabled, buffer));
}

//-----------------------------------------------------------------------------
// Foreground プロセスの最後の画面バッファを取得する
//
Result GetLastForegroundCaptureBuffer(bool* pOutIsScreenShotEnabled, void* pOutBuffer, size_t dataSize) NN_NOEXCEPT
{
    sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), dataSize);
    return HandleAcquireCaptureBufferResult(am::GetDisplayController()->GetLastForegroundCaptureImageEx(pOutIsScreenShotEnabled, buffer));
}

//-----------------------------------------------------------------------------
// LA 呼出元プロセスでのキャプチャ画像を取得する
//
Result GetCallerAppletCaptureBuffer(bool* pOutIsScreenShotEnabled, void* pOutBuffer, size_t dataSize) NN_NOEXCEPT
{
    sf::OutBuffer buffer(reinterpret_cast<char*>(pOutBuffer), dataSize);
    return HandleAcquireCaptureBufferResult(am::GetDisplayController()->GetCallerAppletCaptureImageEx(pOutIsScreenShotEnabled, buffer));
}

//-----------------------------------------------------------------------------
// キャプチャバッファ間でコピーを行なう
//
Result CopyBetweenCaptureBuffers(CaptureBufferIndex dstIndex, CaptureBufferIndex srcIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(dstIndex >= CaptureBufferIndex_LastApplication && dstIndex <= CaptureBufferIndex_CallerApplet);
    NN_SDK_REQUIRES(srcIndex >= CaptureBufferIndex_LastApplication && srcIndex <= CaptureBufferIndex_CallerApplet);
    return HandleAcquireCaptureBufferResult( am::GetDisplayController()->CopyBetweenCaptureBuffers(dstIndex, srcIndex) );
}

//-----------------------------------------------------------------------------
// 現在の画面バッファを Foreground プロセスの最後の画面バッファとして保持させる
//
void HoldCurrentCaptureBufferAsLastForeground() NN_NOEXCEPT
{
    am::GetDisplayController()->UpdateLastForegroundCaptureImage();
}

//-----------------------------------------------------------------------------
// アプリケーションの最後の画面バッファをマップする
//
Result MapLastApplicationCaptureBuffer(bool* pOutIsScreenShotEnabled, void** pOutAddress) NN_NOEXCEPT
{
    NN_RESULT_DO(g_LastApplicationCapture.Map());
    *pOutIsScreenShotEnabled = g_LastApplicationCapture.IsScreenShotEnabled();
    *pOutAddress = g_LastApplicationCapture.GetAddress();
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
// アプリケーションの最後の画面バッファをアンマップする
//
void UnmapLastApplicationCaptureBuffer() NN_NOEXCEPT
{
    g_LastApplicationCapture.Unmap();
}

//-----------------------------------------------------------------------------
// Foreground プロセスの最後の画面バッファをマップする
//
Result MapLastForegroundCaptureBuffer(bool* pOutIsScreenShotEnabled, void** pOutAddress) NN_NOEXCEPT
{
    NN_RESULT_DO(g_LastForegroundCapture.Map());
    *pOutIsScreenShotEnabled = g_LastForegroundCapture.IsScreenShotEnabled();
    *pOutAddress = g_LastForegroundCapture.GetAddress();
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
// Foreground プロセスの最後の画面バッファをアンマップする
//
void UnmapLastForegroundCaptureBuffer() NN_NOEXCEPT
{
    g_LastForegroundCapture.Unmap();
}

//-----------------------------------------------------------------------------
// LA 呼出元プロセスのキャプチャ画像をマップする
//
Result MapCallerAppletCaptureBuffer(bool* pOutIsScreenShotEnabled, void** pOutAddress) NN_NOEXCEPT
{
    NN_RESULT_DO(g_CallerAppletCapture.Map());
    *pOutIsScreenShotEnabled = g_CallerAppletCapture.IsScreenShotEnabled();
    *pOutAddress = g_CallerAppletCapture.GetAddress();
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
// LA 呼出元プロセスのキャプチャ画像をアンマップする
//
void UnmapCallerAppletCaptureBuffer() NN_NOEXCEPT
{
    g_CallerAppletCapture.Unmap();
}

diag::AbortObserverHolder* GetAbortObserverHolderToUnmapCaptureBuffer() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(AbortObserverHolderToUnmap, g_AbortObserverHolderToUnmap);
    return &g_AbortObserverHolderToUnmap;
}

//-----------------------------------------------------------------------------
// キャプチャバッファをクリアする
//
Result ClearLastApplicationCaptureBuffer(bool isScreenShotEnabled, Bit32 clearColor) NN_NOEXCEPT
{
    return am::GetDisplayController()->ClearCaptureBuffer(applet::CaptureBufferIndex_LastApplication, isScreenShotEnabled, clearColor);
}

Result ClearLastForegroundCaptureBuffer(bool isScreenShotEnabled, Bit32 clearColor) NN_NOEXCEPT
{
    return am::GetDisplayController()->ClearCaptureBuffer(applet::CaptureBufferIndex_LastForeground, isScreenShotEnabled, clearColor);
}

Result ClearCallerAppletCaptureBuffer(bool isScreenShotEnabled, Bit32 clearColor) NN_NOEXCEPT
{
    return am::GetDisplayController()->ClearCaptureBuffer(applet::CaptureBufferIndex_CallerApplet, isScreenShotEnabled, clearColor);
}

Result ClearAppletTransitionTextureBuffer(Bit32 clearColor) NN_NOEXCEPT
{
    return am::GetDisplayController()->ClearAppletTransitionBuffer(clearColor);
}

//-----------------------------------------------------------------------------
// スクリーンショット撮影の禁止許可状態の変更
//
void SetScreenShotPermission(ScreenShotPermission permission) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(permission == ScreenShotPermission_Inherit ||
                    permission == ScreenShotPermission_Permit  ||
                    permission == ScreenShotPermission_Forbid);

    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetSelfController()->SetScreenShotPermission(permission) );
}

//-----------------------------------------------------------------------------
// スクリーンショット撮影時の保存ファイルに適用するアプレット情報の設定
//
void SetScreenShotAppletIdentityInfo(applet::AppletIdentityInfo appletInfo) NN_NOEXCEPT
{
    am::service::AppletIdentityInfo info = {};
    info.appletId      = appletInfo.appletId;
    info.applicationId = appletInfo.applicationId.value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetScreenShotAppletIdentityInfo(info));
}

//-----------------------------------------------------------------------------
// 現在の動作モードの取得
//
OperationMode GetOperationMode() NN_NOEXCEPT
{
    Bit8 mode;
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetCommonStateGetter()->GetOperationMode( &mode ) );

    return static_cast<OperationMode>(mode);
}

//-----------------------------------------------------------------------------
// 現在 VR モードが有効か否かを取得
//
bool IsVrMode() NN_NOEXCEPT
{
    bool isValid;
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetCommonStateGetter()->IsVrModeEnabled( &isValid ) );
    return isValid;
}

//-----------------------------------------------------------------------------

CradleStatus GetCradleStatus() NN_NOEXCEPT
{
    Bit8 mode;
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetCommonStateGetter()->GetCradleStatus( &mode ) );

    return static_cast<CradleStatus>(mode);
}

bool GetCradleFwVersion(CradleFwVersion* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    return am::GetCommonStateGetter()->GetCradleFwVersion(&pOutValue->pdcH, &pOutValue->pdcA, &pOutValue->mcu, &pOutValue->dp2hdmi).IsSuccess();
}

BootMode GetBootMode() NN_NOEXCEPT
{
    Bit8 mode;
    NN_ABORT_UNLESS_RESULT_SUCCESS( am::GetCommonStateGetter()->GetBootMode( &mode ) );

    return static_cast<BootMode>(mode);
}

void EnableHandlingForRequestToDisplay() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetHandlesRequestToDisplay(true));
}

void DisableHandlingForRequestToDisplay() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetHandlesRequestToDisplay(false));
}

void ApproveToDisplay() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->ApproveToDisplay());
}

namespace {

    os::Mutex g_EnableHandlingForExitRequestCountMutex{false};
    int g_EnableHandlingForExitRequestCount = 0;

}

void EnableHandlingForExitRequest() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_EnableHandlingForExitRequestCountMutex)> lk(g_EnableHandlingForExitRequestCountMutex);
    if (g_EnableHandlingForExitRequestCount == 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->LockExit());
    }
    ++g_EnableHandlingForExitRequestCount;
}

void DisableHandlingForExitRequest() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_EnableHandlingForExitRequestCountMutex)> lk(g_EnableHandlingForExitRequestCountMutex);
    --g_EnableHandlingForExitRequestCount;
    if (g_EnableHandlingForExitRequestCount == 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->UnlockExit());
    }
}

namespace {

os::Mutex g_LockAccessorMutex{false};

struct LockAccessor
{
    am::service::ILockAccessor* _pAccessor;
    os::SystemEventType _event;

    template <typename F>
    void Initialize(F&& f, applet::ReaderWriterLockIndex index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(index >= 0 && index < applet::ReaderWriterLockIndex_Max);
        std::lock_guard<decltype(g_LockAccessorMutex)> lk(g_LockAccessorMutex);
        if (!_pAccessor)
        {
            this->_pAccessor = f(index).Detach();
            sf::NativeHandle handle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(_pAccessor->GetEvent(&handle));
            os::AttachReadableHandleToSystemEvent(&_event, handle.GetOsHandle(), handle.IsManaged(), os::EventClearMode_ManualClear);
            handle.Detach();
        }
    }

    bool TryLockImpl() NN_NOEXCEPT
    {
        bool success;
        sf::NativeHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(_pAccessor->TryLock(&success, &handle, false));
        return success;
    }

    void Lock() NN_NOEXCEPT
    {
        for (;;)
        {
            os::WaitSystemEvent(&_event);
            if (TryLockImpl())
            {
                return;
            }
        }
    }

    bool TryLock() NN_NOEXCEPT
    {
        return os::TryWaitSystemEvent(&_event) && TryLockImpl();
    }

    void Unlock() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(_pAccessor->Unlock());
    }

};

sf::SharedPointer<am::service::ILockAccessor> GetWriterLockAccessor(applet::ReaderWriterLockIndex index) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::ILockAccessor> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetHomeMenuFunctions()->GetWriterLockAccessorEx(&p, index));
    return p;
}

sf::SharedPointer<am::service::ILockAccessor> GetReaderLockAccessor(applet::ReaderWriterLockIndex index) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::ILockAccessor> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetCommonStateGetter()->GetReaderLockAccessorEx(&p, index));
    return p;
}

LockAccessor g_WriterLockAccessor[applet::ReaderWriterLockIndex_Max];
LockAccessor g_ReaderLockAccessor[applet::ReaderWriterLockIndex_Max];

}

//-----------------------------------------------------------------------------
// Home ボタン処理区間および禁止区間の制御
//
void EnterHomeButtonProcessSection() NN_NOEXCEPT
{
    auto& writerLockAccessor = g_WriterLockAccessor[applet::ReaderWriterLockIndex_HomeButton];
    writerLockAccessor.Initialize(GetWriterLockAccessor, applet::ReaderWriterLockIndex_HomeButton);
    writerLockAccessor.Lock();
}

bool TryEnterHomeButtonProcessSection() NN_NOEXCEPT
{
    auto& writerLockAccessor = g_WriterLockAccessor[applet::ReaderWriterLockIndex_HomeButton];
    writerLockAccessor.Initialize(GetWriterLockAccessor, applet::ReaderWriterLockIndex_HomeButton);
    return writerLockAccessor.TryLock();
}

void LeaveHomeButtonProcessSection() NN_NOEXCEPT
{
    auto& writerLockAccessor = g_WriterLockAccessor[applet::ReaderWriterLockIndex_HomeButton];
    return writerLockAccessor.Unlock();
}

void EnterHomeButtonProhibitionSection() NN_NOEXCEPT
{
    auto& readerLockAccessor = g_ReaderLockAccessor[applet::ReaderWriterLockIndex_HomeButton];
    readerLockAccessor.Initialize(GetReaderLockAccessor, applet::ReaderWriterLockIndex_HomeButton);
    readerLockAccessor.Lock();
}

bool TryEnterHomeButtonProhibitionSection() NN_NOEXCEPT
{
    auto& readerLockAccessor = g_ReaderLockAccessor[applet::ReaderWriterLockIndex_HomeButton];
    readerLockAccessor.Initialize(GetReaderLockAccessor, applet::ReaderWriterLockIndex_HomeButton);
    return readerLockAccessor.TryLock();
}

void LeaveHomeButtonProhibitionSection() NN_NOEXCEPT
{
    auto& readerLockAccessor = g_ReaderLockAccessor[applet::ReaderWriterLockIndex_HomeButton];
    readerLockAccessor.Unlock();
}

//-----------------------------------------------------------------------------
// エントランス処理区間および禁止区間の制御
//
void EnterEntranceProcessSection() NN_NOEXCEPT
{
    auto& writerLockAccessor = g_WriterLockAccessor[applet::ReaderWriterLockIndex_Entrance];
    writerLockAccessor.Initialize(GetWriterLockAccessor, applet::ReaderWriterLockIndex_Entrance);
    writerLockAccessor.Lock();
}

bool TryEnterEntranceProcessSection() NN_NOEXCEPT
{
    auto& writerLockAccessor = g_WriterLockAccessor[applet::ReaderWriterLockIndex_Entrance];
    writerLockAccessor.Initialize(GetWriterLockAccessor, applet::ReaderWriterLockIndex_Entrance);
    return writerLockAccessor.TryLock();
}

void LeaveEntranceProcessSection() NN_NOEXCEPT
{
    auto& writerLockAccessor = g_WriterLockAccessor[applet::ReaderWriterLockIndex_Entrance];
    return writerLockAccessor.Unlock();
}

void EnterEntranceProhibitionSection() NN_NOEXCEPT
{
    auto& readerLockAccessor = g_ReaderLockAccessor[applet::ReaderWriterLockIndex_Entrance];
    readerLockAccessor.Initialize(GetReaderLockAccessor, applet::ReaderWriterLockIndex_Entrance);
    readerLockAccessor.Lock();
}

bool TryEnterEntranceProhibitionSection() NN_NOEXCEPT
{
    auto& readerLockAccessor = g_ReaderLockAccessor[applet::ReaderWriterLockIndex_Entrance];
    readerLockAccessor.Initialize(GetReaderLockAccessor, applet::ReaderWriterLockIndex_Entrance);
    return readerLockAccessor.TryLock();
}

void LeaveEntranceProhibitionSection() NN_NOEXCEPT
{
    auto& readerLockAccessor = g_ReaderLockAccessor[applet::ReaderWriterLockIndex_Entrance];
    readerLockAccessor.Unlock();
}

//-----------------------------------------------------------------------------
// 割込みシーン処理区間および禁止区間の制御
//
void EnterInterruptSceneProcessSection() NN_NOEXCEPT
{
    auto& writerLockAccessor = g_WriterLockAccessor[applet::ReaderWriterLockIndex_InterruptScene];
    writerLockAccessor.Initialize(GetWriterLockAccessor, applet::ReaderWriterLockIndex_InterruptScene);
    writerLockAccessor.Lock();
}

bool TryEnterInterruptSceneProcessSection() NN_NOEXCEPT
{
    auto& writerLockAccessor = g_WriterLockAccessor[applet::ReaderWriterLockIndex_InterruptScene];
    writerLockAccessor.Initialize(GetWriterLockAccessor, applet::ReaderWriterLockIndex_InterruptScene);
    return writerLockAccessor.TryLock();
}

void LeaveInterruptSceneProcessSection() NN_NOEXCEPT
{
    auto& writerLockAccessor = g_WriterLockAccessor[applet::ReaderWriterLockIndex_InterruptScene];
    return writerLockAccessor.Unlock();
}

void EnterInterruptSceneProhibitionSection() NN_NOEXCEPT
{
    auto& readerLockAccessor = g_ReaderLockAccessor[applet::ReaderWriterLockIndex_InterruptScene];
    readerLockAccessor.Initialize(GetReaderLockAccessor, applet::ReaderWriterLockIndex_InterruptScene);
    readerLockAccessor.Lock();
}

bool TryEnterInterruptSceneProhibitionSection() NN_NOEXCEPT
{
    auto& readerLockAccessor = g_ReaderLockAccessor[applet::ReaderWriterLockIndex_InterruptScene];
    readerLockAccessor.Initialize(GetReaderLockAccessor, applet::ReaderWriterLockIndex_InterruptScene);
    return readerLockAccessor.TryLock();
}

void LeaveInterruptSceneProhibitionSection() NN_NOEXCEPT
{
    auto& readerLockAccessor = g_ReaderLockAccessor[applet::ReaderWriterLockIndex_InterruptScene];
    readerLockAccessor.Unlock();
}

void SetWirelessPriorityModeForLibraryApplet(WirelessPriorityMode wirelessPrioriryMode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(am::IsInitializedAsLibraryApplet());

    switch (wirelessPrioriryMode)
    {
        case WirelessPriorityMode_Default:
        {
            applet::SetWirelessPriorityMode(applet::WirelessPriorityMode_None);
            return;
        }
        case WirelessPriorityMode_OptimizedForWlan:
        {
            applet::SetWirelessPriorityMode(applet::WirelessPriorityMode_OptimizedForWlan);
            return;
        }
        default: NN_UNEXPECTED_DEFAULT;
    }
}

Result SetMemoryHeapSizeWithRetry(size_t size) NN_NOEXCEPT
{
    auto retryRestTime = TimeSpan::FromMilliSeconds(1000);
    auto retrySleepTime = TimeSpan::FromMilliSeconds(10);
    auto retryCount = 0;
retry:
    if (retryCount > 0)
    {
        os::SleepThread(retrySleepTime);
        retryRestTime -= retrySleepTime;
        retrySleepTime = retrySleepTime + retrySleepTime;
    }
    NN_RESULT_TRY(os::SetMemoryHeapSize(size))
        NN_RESULT_CATCH(os::ResultOutOfMemory)
        {
            if (!(retryRestTime >= 0))
            {
                NN_SDK_LOG("failed to os::SetMemoryHeapSize(). give up.\n");
                NN_RESULT_RETHROW;
            }
            NN_SDK_LOG("failed to os::SetMemoryHeapSize(). sleep %dms before retry[%d]\n", static_cast<int>(retrySleepTime.GetMilliSeconds()), retryCount);
            ++retryCount;
            goto retry;
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

Result GetApplicationIdByContentActionName(ncm::ApplicationId* pOut, const char* name) NN_NOEXCEPT
{
    Bit64 id;
    NN_RESULT_DO(am::GetCommonStateGetter()->GetApplicationIdByContentActionName(&id, {name, std::strlen(name) + 1}));
    *pOut = {id};
    NN_RESULT_SUCCESS;
}

}}  // namespace nn::ae

