﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/am/am_Shim.h>
#include <nn/am/am_Result.h>

#include <nn/applet/applet_Types.h>
#include <nn/applet/applet_Result.h>

#include <nn/ae/ae_ApplicationControlApi.h>
#include <nn/ae/ae_ApplicationLaunchPropertyApi.h>
#include <nn/ae/ae_Result.h>
#include <nn/ae/ae_Types.h>
#include <nn/arp/arp_Types.h>

#include <nn/lr/lr_Result.h>
#include <nn/audio/audio_Applet.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_SdkSystemEventApi.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_Span.h>
#include <nn/account/account_Types.h>
#include <nn/ns/ns_ApplicationRightsTypes.h>

namespace nn { namespace applet {

bool RequestExitLibraryAppletOrTerminate(am::service::IAppletAccessor* laAccessor, TimeSpan timeout) NN_NOEXCEPT;

}}

namespace nn { namespace ae {

namespace detail {

struct ApplicationHandleObject
{
    am::service::IApplicationAccessor* pAccessor;
    os::SystemEventType exitEvent;
};

namespace {

    os::Mutex g_Mutex{false};
    ApplicationHandleObject g_Objects[8];

    ApplicationHandleObject* CreateObject(sf::SharedPointer<am::service::IApplicationAccessor> pAccessor) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
        for (auto&& e : g_Objects)
        {
            if (!e.pAccessor)
            {
                sf::NativeHandle eventHandle;
                NN_ABORT_UNLESS_RESULT_SUCCESS(pAccessor->GetAppletStateChangedEvent(&eventHandle));
                os::AttachReadableHandleToSystemEvent(&e.exitEvent, eventHandle.GetOsHandle(), eventHandle.IsManaged(), os::EventClearMode_ManualClear);
                eventHandle.Detach();
                e.pAccessor = pAccessor.Detach();
                return &e;
            }
        }
        NN_ABORT("too many applications");
    }

    void DeleteObject(ApplicationHandleObject* p) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
        auto&& e = *p;
        os::DestroySystemEvent(&e.exitEvent);
        sf::ReleaseSharedObject(e.pAccessor);
        e.pAccessor = nullptr;
    }

}

}

ApplicationHandle CreateApplication(ncm::ApplicationId id) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::IApplicationAccessor> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationCreator()->CreateApplication(&p, id));
    return {detail::CreateObject(std::move(p))};
}

ApplicationHandle CreateSystemApplication(ncm::SystemApplicationId id) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::IApplicationAccessor> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationCreator()->CreateSystemApplication(&p, id));
    return {detail::CreateObject(std::move(p))};
}

bool TryPopLaunchRequestedApplication(ApplicationHandle* pOut) NN_NOEXCEPT
{
    sf::SharedPointer<am::service::IApplicationAccessor> p;
    NN_RESULT_TRY(am::GetApplicationCreator()->PopLaunchRequestedApplication(&p))
        NN_RESULT_CATCH(am::ResultNoStorage)
        {
            return false;
        }
    NN_RESULT_END_TRY
    *pOut = {detail::CreateObject(std::move(p))};
    return true;
}

void PushApplicationLaunchParameter(ApplicationHandle applicationHandle, applet::LaunchParameterKind kind, applet::StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationHandle.IsValid());
    auto p = applicationHandle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    sf::SharedPointer<am::service::IStorage> storage(static_cast<am::service::IStorage*>(storageHandle._p), false);
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->PushLaunchParameter(kind, std::move(storage)));
}

void StartApplication(ApplicationHandle applicationHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationHandle.IsValid());
    auto p = applicationHandle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    NN_ABORT_UNLESS_RESULT_SUCCESS(p->Start());
}

ncm::ApplicationId GetApplicationId(ApplicationHandle applicationHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationHandle.IsValid());
    auto p = applicationHandle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    ncm::ApplicationId ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetApplicationId(&ret));

    return ret;
}

Result GetApplicationControlProperty(nn::ns::ApplicationControlProperty* outValue, ApplicationHandle applicationHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationHandle.IsValid());
    auto p = applicationHandle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    for (int i=0; i<50; ++i)
    {
        auto result = p->GetApplicationControlProperty(nn::sf::OutBuffer(reinterpret_cast<char*>(outValue), sizeof(nn::ns::ApplicationControlProperty)));
        if (result.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }
        if (result <= am::ResultApplicationHasNoProperty())
        {
            NN_RESULT_THROW(result);
        }
        if (!(result <= am::ResultApplicationPropertyNotReady()))
        {
            NN_RESULT_THROW(result);
        }
        os::SleepThread( TimeSpan::FromMilliSeconds(100) );
    }
    NN_RESULT_THROW( am::ResultApplicationPropertyNotReady() );
}

Result GetApplicationLaunchProperty(nn::arp::ApplicationLaunchProperty* outValue, ApplicationHandle applicationHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationHandle.IsValid());
    auto p = applicationHandle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    for (int i=0; i<50; ++i)
    {
        auto result = p->GetApplicationLaunchProperty(nn::sf::OutBuffer(reinterpret_cast<char*>(outValue), sizeof(nn::arp::ApplicationLaunchProperty)));
        if (result.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }
        if (result <= am::ResultApplicationHasNoProperty())
        {
            NN_RESULT_THROW(result);
        }
        if (!(result <= am::ResultApplicationPropertyNotReady()))
        {
            NN_RESULT_THROW(result);
        }
        os::SleepThread( TimeSpan::FromMilliSeconds(100) );
    }
    NN_RESULT_THROW( am::ResultApplicationPropertyNotReady() );
}

//-----------------------------------------------------------------------------
//  アプリケーションを起動する
//
ApplicationHandle LaunchApplication(nn::ncm::ApplicationId id, bool foreground) NN_NOEXCEPT
{
    // アプリを起動して ApplicationHandle を設定する
    sf::SharedPointer<am::service::IApplicationAccessor> p;
    auto result = am::GetApplicationCreator()->CreateApplication(&p, id);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( lr::ResultProgramNotFound )
        {
            NN_SDK_LOG("指定された ApplicationId のアプリケーションが見つかりません。\n");
            NN_SDK_LOG("ApplicationId: 0x%016llx\n", id.value);
            NN_RESULT_THROW( ae::ResultApplicationNotFound() );
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    // TORIAEZU: TODO:
    // SA->アプリ へ渡す起動パラメータの設定が必要な場合はここで行なう。

    // アプリを開始する
    NN_ABORT_UNLESS_RESULT_SUCCESS( p->Start() );
    if (foreground)
    {
        // TORIAEZU: 互換性のため起動したアプリを fore にする
        NN_ABORT_UNLESS_RESULT_SUCCESS( p->RequestForApplicationToGetForeground() );
    }

    return {detail::CreateObject(std::move(p))};
}


//-----------------------------------------------------------------------------
//  外部起動されたアプリケーションハンドルを取得します（デバッグ起動待ち用）
//
ApplicationHandle PopFloatingApplicationHandle() NN_NOEXCEPT
{
    sf::SharedPointer<am::service::IApplicationAccessor> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationCreator()->PopFloatingApplicationForDevelopment(&p));
    if (!p)
    {
        return ApplicationHandle::GetInvalidHandle();
    }
    return {detail::CreateObject(std::move(p))};
}


//-----------------------------------------------------------------------------
//  対象アプリケーションに終了するように要求を出す
//
Result RequestApplicationExit(ApplicationHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    auto result = p->RequestExit();
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( applet::ResultApplicationExited )
        {
            NN_RESULT_THROW( ae::ResultApplicationExited() );
        }
        NN_RESULT_CATCH( applet::ResultApplicationExitedAbnormally )
        {
            NN_RESULT_THROW( ae::ResultApplicationExitedAbnormally() );
        }
        NN_RESULT_CATCH( applet::ResultApplicationTerminated )
        {
            NN_RESULT_THROW( ae::ResultApplicationTerminated() );
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
//  対象アプリケーションを強制終了させる
//
Result TerminateApplication(ApplicationHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    auto result = p->Terminate();
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( applet::ResultApplicationExited )
        {
            NN_RESULT_THROW( ae::ResultApplicationExited() );
        }
        NN_RESULT_CATCH( applet::ResultApplicationExitedAbnormally )
        {
            NN_RESULT_THROW( ae::ResultApplicationExitedAbnormally() );
        }
        NN_RESULT_CATCH( applet::ResultApplicationTerminated )
        {
            NN_RESULT_THROW( ae::ResultApplicationTerminated() );
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
//  対象アプリケーションが再度 Foreground となるように要求を出す
//
Result RequestApplicationGetForeground(ApplicationHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    auto result = p->RequestForApplicationToGetForeground();
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( applet::ResultApplicationExited )
        {
            NN_RESULT_THROW( ae::ResultApplicationExited() );
        }
        NN_RESULT_CATCH( applet::ResultApplicationExitedAbnormally )
        {
            NN_RESULT_THROW( ae::ResultApplicationExitedAbnormally() );
        }
        NN_RESULT_CATCH( applet::ResultApplicationTerminated )
        {
            NN_RESULT_THROW( ae::ResultApplicationTerminated() );
        }
        NN_RESULT_CATCH( applet::ResultAlreadyForeground )
        {
            NN_RESULT_THROW( ae::ResultAlreadyForeground() );
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
//  アプリケーションの結果を取得する
//

os::SystemEventType* GetApplicationExitEvent(ApplicationHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    return &handle._p->exitEvent;
}

Result GetApplicationResultRawForDebug(ApplicationHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);
    os::WaitSystemEvent(GetApplicationExitEvent(handle));
    return p->GetResult();
}

ApplicationResult GetApplicationResult(ApplicationHandle handle) NN_NOEXCEPT
{
    NN_RESULT_TRY(GetApplicationResultRawForDebug(handle))
        NN_RESULT_CATCH(am::ResultAppletExitAbnormally)
        {
            return ApplicationResult_AbnormallyExited;
        }
        NN_RESULT_CATCH(am::ResultAppletTerminatedManually)
        {
            return ApplicationResult_TerminatedManually;
        }
        NN_RESULT_CATCH(am::ResultAppletTerminatedByMediaLost)
        {
            return ApplicationResult_TerminatedByMediaLost;
        }
        NN_RESULT_CATCH(am::ResultApplicationExitedVoluntarily)
        {
            return ApplicationResult_VoluntarilyExited;
        }
        NN_RESULT_CATCH_ALL
        {
            return ApplicationResult_Unexpected;
        }
    NN_RESULT_END_TRY
    return ApplicationResult_NormallyExited;
}

//-----------------------------------------------------------------------------
//  アプリケーションハンドルをクローズする
//
void CloseApplicationHandle(ApplicationHandle handle) NN_NOEXCEPT
{
    auto p = handle._p;
    if (p)
    {
        detail::DeleteObject(p);
    }
}


//-----------------------------------------------------------------------------
// アプリケーション起点の全ての LA を強制終了します。
//
bool RequestExitLibraryAppletOfApplicationOrTerminate(ApplicationHandle handle, TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    sf::SharedPointer<am::service::IAppletAccessor> laAccessor;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetCurrentLibraryApplet(&laAccessor));
    return applet::RequestExitLibraryAppletOrTerminate(laAccessor.Get(), timeout);
}

void RequestExitLibraryAppletOfApplication(ApplicationHandle handle) NN_NOEXCEPT
{
    auto success = RequestExitLibraryAppletOfApplicationOrTerminate(handle, TimeSpan::FromSeconds(-1));
    NN_UNUSED(success);
    NN_SDK_ASSERT(success);
}

namespace detail {

ns::RightsEnvironmentHandle GetNsRightsEnvironmentHandle(ApplicationHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);
    Bit64 nsRightsEnvironmentHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetNsRightsEnvironmentHandle(&nsRightsEnvironmentHandle));
    return static_cast<ns::RightsEnvironmentHandle>(nsRightsEnvironmentHandle);
}

}

Result SetApplicationUsers(ApplicationHandle handle, const account::Uid uids[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);
    return p->SetUsers(false, util::MakeSpan(uids, count));
}

Result SetApplicationUsers(ApplicationHandle handle, AllUserTag) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);
    account::Uid dummy;
    return p->SetUsers(true, {&dummy, &dummy});
}

bool CheckApplicationRightsAvailability(ApplicationHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);
    bool ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->CheckRightsEnvironmentAvailable(&ret));
    return ret;
}

//-----------------------------------------------------------------------------
// アプリケーション起動要求元の情報を返します。
//
applet::ApplicationLaunchRequestInfo GetApplicationLaunchRequestInfo(ApplicationHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(handle.IsValid());
    auto p = handle._p->pAccessor;
    NN_SDK_ASSERT(p != NULL);

    applet::ApplicationLaunchRequestInfo info = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(p->GetApplicationLaunchRequestInfo(&info));
    return info;
}


//-----------------------------------------------------------------------------

}}  // namespace nn::ae

