﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/user/account_UserStateManager.h>

#include <mutex>
#include <nn/result/result_HandlingUtility.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/detail/account_Log.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace account { namespace user {

ApplicationUserStateManager::~ApplicationUserStateManager() NN_NOEXCEPT
{
#if !defined(NN_SDK_BUILD_RELEASE)
    Uid buffer[UserCountMax];
    int ct = 0;
    for (const auto& u : m_OpenUsers)
    {
        if (u.IsValid())
        {
            buffer[ct++] = u;
        }
    }
    for (int i = 0; i < ct; ++ i)
    {
        NN_DETAIL_ACCOUNT_ERROR("ApplicationUserStateManager: Still opened: %016llx_%016llx\n", buffer[i]._data[0], buffer[i]._data[1]);
    }
    NN_SDK_ASSERT(ct == 0);
#endif
}
void ApplicationUserStateManager::EnableQualificationLimitation(const Uid users[], int count) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsQualificationLimitationEnabled);
    NN_SDK_ASSERT(count <= UserCountMax);
    NN_SDK_ASSERT(std::find(users, users + count, InvalidUid) == users + count);

    auto p = std::find_if(
        m_OpenUsers.begin(), m_OpenUsers.end(),
        [](const UserRef& u) -> bool { return u.IsValid(); });
    NN_ABORT_UNLESS(p == m_OpenUsers.end());

    m_IsQualificationLimitationEnabled = true;
    std::copy(users, users + count, m_QualifiedUsers);
    m_QualifiedUserCount = count;
#if !defined(NN_SDK_BUILD_RELEASE)
    NN_DETAIL_ACCOUNT_TRACE("ApplicationUserStateManager: QualificationLimitationEnabled\n");
    for (auto i = 0; i < count; ++ i)
    {
        NN_DETAIL_ACCOUNT_TRACE("    - %016lx_%016lx\n", users[i]._data[0], users[i]._data[1]);
    }
#endif
}
bool ApplicationUserStateManager::IsQualificationLimitationEnabled() const NN_NOEXCEPT
{
    return m_IsQualificationLimitationEnabled;
}
int ApplicationUserStateManager::ListQualifiedUsers(Uid buffer[], int capacity) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsQualificationLimitationEnabled);
    NN_SDK_ASSERT(capacity >= UserCountMax);
    NN_UNUSED(capacity);

    std::copy(m_QualifiedUsers, m_QualifiedUsers + m_QualifiedUserCount, buffer);
    return m_QualifiedUserCount;
}
bool ApplicationUserStateManager::IsUserQualified(const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsQualificationLimitationEnabled);
    return std::find(m_QualifiedUsers, m_QualifiedUsers + m_QualifiedUserCount, user) != m_QualifiedUsers + m_QualifiedUserCount;
}
void ApplicationUserStateManager::SetUserStateOpened(UserRef&& user) NN_NOEXCEPT
{
    auto uid = static_cast<Uid>(user);
    NN_UNUSED(uid);

#if !defined(NN_SDK_BUILD_RELEASE)
    auto p0 = std::find_if(
        m_OpenUsers.begin(), m_OpenUsers.end(),
        [&uid](const UserRef& u) -> bool { return u.IsValid() && u == uid; });
    NN_SDK_ASSERT(p0 == m_OpenUsers.end());
#endif

    NN_ABORT_UNLESS(!IsQualificationLimitationEnabled() || IsUserQualified(user));

    auto p = std::find_if(
        m_OpenUsers.begin(), m_OpenUsers.end(),
        [](const UserRef& u) -> bool { return !u.IsValid(); });
    NN_ABORT_UNLESS(p != m_OpenUsers.end());
    *p = std::move(user);
    NN_DETAIL_ACCOUNT_TRACE("ApplicationUserStateManager: SetUserStateOpened: %016llx_%016llx\n", uid._data[0], uid._data[1]);
}
void ApplicationUserStateManager::SetUserStateClosed(const Uid& uid) NN_NOEXCEPT
{
    auto p = std::find_if(
        m_OpenUsers.begin(), m_OpenUsers.end(),
        [uid](const UserRef& u) -> bool { return u.IsValid() && u == uid; });
    NN_ABORT_UNLESS(p != m_OpenUsers.end());
    p->Release();
    NN_DETAIL_ACCOUNT_TRACE("ApplicationUserStateManager: SetUserStateClosed: %016llx_%016llx\n", uid._data[0], uid._data[1]);
}
bool ApplicationUserStateManager::IsUserOpened(const Uid& uid) const NN_NOEXCEPT
{
    auto p = std::find_if(
        m_OpenUsers.begin(), m_OpenUsers.end(),
        [uid](const UserRef& u) -> bool { return u.IsValid() && u == uid; });
    return p != m_OpenUsers.end();
}
int ApplicationUserStateManager::ListOpenUsers(Uid buffer[], int capacity) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(capacity == UserCountMax);
    NN_UNUSED(capacity);
    int ct = 0;
    for (const auto& u : m_OpenUsers)
    {
        if (u.IsValid())
        {
            buffer[ct++] = u;
        }
    }
    return ct;
}

void UserStateManager::EnableQualificationLimitation(ApplicationId appId, const Uid users[], int count) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);
    auto* p = this->FindUnsafe(appId);
    NN_ABORT_UNLESS_NOT_NULL(p);
    p->EnableQualificationLimitation(users, count);
}
int UserStateManager::SelectQualifiedUsers(
    Uid buffer[], int capacity,
    ApplicationId appId, const Uid users[], int count) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(capacity >= count);
    NN_UNUSED(capacity);

    NN_UTIL_LOCK_GUARD(*this);
    auto* p = this->FindUnsafe(appId);
    NN_ABORT_UNLESS_NOT_NULL(p);

    if (!p->IsQualificationLimitationEnabled())
    {
        auto end = std::copy(users, users + count, buffer);
        return static_cast<int>(end - buffer);
    }

    Uid qualified[UserCountMax];
    auto qualifiedCount = p->ListQualifiedUsers(qualified, std::extent<decltype(qualified)>::value);
    auto end = std::copy_if(
        users, users + count, buffer,
        [&qualified, qualifiedCount](const Uid& u) -> bool {
            return std::find(qualified, qualified + qualifiedCount, u) != qualified + qualifiedCount;
        }
    );
    return static_cast<int>(end - buffer);
}
Result UserStateManager::SetUserStateOpened(ApplicationId appId, UserRef&& user) NN_NOEXCEPT
{
    auto uid = static_cast<Uid>(user);
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    NN_UTIL_LOCK_GUARD(*this);
    auto* p = this->FindUnsafe(appId);
    NN_ABORT_UNLESS_NOT_NULL(p);

    if (p->IsQualificationLimitationEnabled())
    {
        NN_RESULT_THROW_UNLESS(p->IsUserQualified(uid), ResultUserUnqualified());
    }
    NN_RESULT_THROW_UNLESS(!p->IsUserOpened(uid), ResultUserAlreadyOpened());
    p->SetUserStateOpened(std::move(user));
    m_LastOpened = uid;
    Notifiable::SignalEvents(UserStateManagerTag_StateChange);
    NN_RESULT_SUCCESS;
}
Result UserStateManager::SetUserStateClosed(ApplicationId appId, const Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    NN_UTIL_LOCK_GUARD(*this);
    auto* p = this->FindUnsafe(appId);
    NN_ABORT_UNLESS_NOT_NULL(p);

    NN_RESULT_THROW_UNLESS(p->IsUserOpened(uid), ResultUserAlreadyClosed());
    p->SetUserStateClosed(uid);
    Notifiable::SignalEvents(UserStateManagerTag_StateChange);
    NN_RESULT_SUCCESS;
}
bool UserStateManager::IsUserOpened(const Uid& uid) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserId());

    NN_UTIL_LOCK_GUARD(*this);
    auto itr = this->GetIteratorUnsafe();
    while (auto v = itr.Next())
    {
        if (v.pResource->IsUserOpened(uid))
        {
            NN_DETAIL_ACCOUNT_TRACE("UserStateManager: IsUserOpen: %016llx_%016llx in %016llx\n", uid._data[0], uid._data[1], v.applicationId);
            return true;
        }
    }
    return false;
}
int UserStateManager::ListOpenUsers(Uid buffer[], int capacity) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(capacity >= UserCountMax);
    NN_UNUSED(capacity);

    NN_UTIL_LOCK_GUARD(*this);
    int count = 0;
    auto itr = this->GetIteratorUnsafe();
    while (auto v = itr.Next())
    {
        Uid tmp[UserCountMax];
        auto tmpCt = v.pResource->ListOpenUsers(tmp, std::extent<decltype(tmp)>::value);
        for (Uid* u = tmp; u != tmp + tmpCt; ++u)
        {
            if (std::find(buffer, buffer + count, *u) == buffer + count)
            {
                buffer[count++] = *u;
            }
        }
    }
    return count;
}
int UserStateManager::ListOpenUsers(Uid buffer[], int capacity, ApplicationId appId) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);
    auto* p = this->FindUnsafe(appId);
    NN_ABORT_UNLESS_NOT_NULL(p);
    return p->ListOpenUsers(buffer, capacity);
}
bool UserStateManager::TryGetLastOpenedUser(Uid* pOut) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);
    if (!m_LastOpened)
    {
        return false;
    }
    *pOut = m_LastOpened;
    return true;
}
void UserStateManager::InvalidateLastOpenedUser() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);
    m_LastOpened = InvalidUid;
}
Result UserStateManager::GetUserStateChangeEvent(os::SystemEventType** pOutEvent) NN_NOEXCEPT
{
    return Notifiable::AcquireSystemEvent(pOutEvent, UserStateManagerTag_StateChange);
}
}}} // namespace nn::account::user
