﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nn;
using nn.sf;
using std;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;

namespace nn
{
    [ExternalStruct(8, 8)]
    [Pre(@"_ != ::nn::ApplicationId::GetInvalidId()")]
    [CppRefPath("<nn/nn_ApplicationId.h>")]
    public struct ApplicationId
    {
    }
}

namespace nn.npns
{
    [ExternalStruct(40, 1)]
    [CppRefPath("<nn/npns/npns_Types.h>")]
    public struct NotificationToken
    {
    }
}

namespace nn.time
{
    [ExternalStruct(8,8)]
    [CppRefPath("<nn/time/time_PosixTime.h>")]
    public struct PosixTime
    {
    }
}

namespace nn.account
{
    [ExternalStruct(16, 8)]
    [Pre(@"static_cast<bool>(_)")]
    [CppRefPath("<nn/account/account_Types.h>")]
    public struct Uid
    {
    }

    [ExternalStruct(16, 1)]
    [CppRefPath("<nn/account/account_TypesForSystemServices.h>")]
    public struct ProfileDigest
    {
    }

    [ExternalStruct(8, 8)]
    [CppRefPath("<nn/account/account_Types.h>")]
    public struct NetworkServiceAccountId
    {
    }

    [LargeData]
    [ExternalStruct(16, 8)]
    [Pre(@"_.id != ::nn::ApplicationId::GetInvalidId()")]
    [CppRefPath("<nn/account/account_TypesForSystemServices.h>")]
    public struct SystemProgramIdentification
    {
    }

    [LargeData]
    [ExternalStruct(4096, 1)]
    [CppRefPath("<nn/account/account_TypesForSystemServices.h>")]
    public struct RequestUrl
    {
    }

    [LargeData]
    [ExternalStruct(256, 1)]
    [CppRefPath("<nn/account/account_TypesForSystemServices.h>")]
    public struct CallbackUri
    {
    }

    [ExternalStruct(8, 8)]
    [Pre(@"static_cast<bool>(_)")]
    [CppRefPath("<nn/account/account_NintendoAccountTypes.h>")]
    public struct NintendoAccountId
    {
    }

    [LargeData]
    [ExternalStruct(512, 1)]
    [CppRefPath("<nn/account/account_Types.h>")]
    public struct NintendoAccountAuthorizationRequestParameters
    {
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter, AllowMultiple = true)]
    public class NotNullAttribute : PreAttribute
    {
        public NotNullAttribute()
            : base("_.GetPointerUnsafe() != nullptr && _.GetSize() > 0")
        {
        }
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter, AllowMultiple = true)]
    public class NotEmptyAttribute : PreAttribute
    {
        public NotEmptyAttribute()
            : base("_.GetData() != nullptr && _.GetLength() > 0")
        {
        }
    }
}

namespace nn.account.detail
{
    [ExternalStruct(16, 4)]
    [Pre(@"static_cast<bool>(_)")]
    [CppRefPath("<nn/account/detail/account_InternalTypes.h>")]
    public struct Uuid
    {
    }
}

namespace nn.account.profile
{
    [ExternalStruct(56, 8)]
    [Pre(@"static_cast<bool>(_)")]
    [CppRefPath("<nn/account/profile/account_ProfileTypes.h>")]
    public struct ProfileBase
    {
    }

    [LargeData]
    [ExternalStruct(128, 1)]
    [CppRefPath("<nn/account/profile/account_ProfileTypes.h>")]
    public struct UserData
    {
    }
}

namespace nn.account.nas
{
    [LargeData]
    [ExternalStruct(128, 1)]
    [CppRefPath("<nn/account/nas/account_NasTypes.h>")]
    public struct State
    {
    }

    [LargeData]
    [ExternalStruct(264, 8)]
    [Pre(@"static_cast<bool>(_)")]
    [CppRefPath("<nn/account/nas/account_NasTypes.h>")]
    public struct NasClientInfo
    {
    }

    [LargeData]
    [ExternalStruct(592, 1)]
    [CppRefPath("<nn/account/nas/account_NasTypes.h>")]
    public struct NasUserBase
    {
    }

    [LargeData]
    [ExternalStruct(104, 1)]
    [CppRefPath("<nn/account/nas/account_NasTypes.h>")]
    public struct NasUserBaseForApplication
    {
    }
}

namespace nn.account
{
    internal static class PathInfo
    {
        public const string Prefix = "<";
        public const string Postfix = ">";
        public const string Root = @"nn/account";
        public const string Path = Prefix + Root + @"/account_IAccountService.sfdl.h" + Postfix;
    }

    // 共通インターフェイス
    [CppRefPath(PathInfo.Path)]
    public interface IAccountServiceBase : IServiceObject
    {
        [MethodId(0)]
        Result GetUserCount(Out<int32_t> pOutCount);

        [MethodId(1)]
        Result GetUserExistence(Out<bool> pOutExistence, Uid uid);

        [MethodId(2)]
        Result ListAllUsers(
            [Pre(@"_.GetLength() == ::nn::account::UserCountMax")] OutArray<Uid> outUsers);

        [MethodId(3)]
        Result ListOpenUsers(
            [Pre(@"_.GetLength() == ::nn::account::UserCountMax")] OutArray<Uid> outUsers);

        [MethodId(4)]
        Result GetLastOpenedUser(Out<Uid> pOutUid);

        [MethodId(5)]
        Result GetProfile(Out<profile.IProfile> pOutProfile, Uid uid);

        [MethodId(6)]
        Result GetProfileDigest(Out<ProfileDigest> pOut, Uid uid);

        // ------------------------------------------------------------------------------------

        [MethodId(50)]
        Result IsUserRegistrationRequestPermitted(Out<bool> pOut, [ProcessId] Bit64 pid);

        [MethodId(51)]
        Result TrySelectUserWithoutInteraction(Out<Uid> pOut, bool isNetworkServiceAccountRequired);

        // ------------------------------------------------------------------------------------

        [MethodId(99)]
        Result DebugActivateOpenContextRetention(Out<detail.ISessionObject> pOut);
    }

    // 一般アプリ向けのインターフェイス
    [CppRefPath(PathInfo.Path)]
    public interface IAccountServiceForApplication : IAccountServiceBase
    {
        // 互換維持用 (SIGLO-81696)
        [MethodId(100)]
        Result InitializeApplicationInfoV0([ProcessId] Bit64 pid);

        [MethodId(101)]
        Result GetBaasAccountManagerForApplication(Out<baas.IManagerForApplication> pOut, Uid uid);

        [MethodId(102)]
        Result AuthenticateApplicationAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(103)]
        Result CheckNetworkServiceAvailabilityAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(110)]
        Result StoreSaveDataThumbnail(Uid uid, [NotNull] InBuffer imageBuffer);

        [MethodId(111)]
        Result ClearSaveDataThumbnail(Uid uid);

        [MethodId(120)]
        Result CreateGuestLoginRequest(Out<baas.IGuestLoginRequest> pOutRequest, NativeHandle workMemory, UInt32 workMemorySize);

        [MethodId(130)]
        Result LoadOpenContext(Out<baas.IManagerForApplication> outPtr0, Uid uid);

        [MethodId(131)]
        Result ListOpenContextStoredUsers(
            [Pre(@"_.GetLength() == ::nn::account::UserCountMax")] OutArray<Uid> outUsers);

        [MethodId(140)]
        Result InitializeApplicationInfo([ProcessId] Bit64 pid);

        [MethodId(141)]
        Result ListQualifiedUsers(
            [Pre(@"_.GetLength() == ::nn::account::UserCountMax")] OutArray<Uid> outUsers);
    }

    // psel や shell 以外の本体機能, システム向けのインターフェイス
    [CppRefPath(PathInfo.Path)]
    public interface IAccountServiceForSystemService : IAccountServiceBase
    {
        [MethodId(100)]
        Result GetUserRegistrationNotifier(Out<detail.INotifier> pOutNotifier);

        [MethodId(101)]
        Result GetUserStateChangeNotifier(Out<detail.INotifier> pOutNotifier);

        [MethodId(104)]
        Result GetProfileUpdateNotifier(Out<detail.INotifier> pOutNotifier);

        [MethodId(102)]
        Result GetBaasAccountManagerForSystemService(Out<baas.IManagerForSystemService> pOut, Uid uid);

        [MethodId(103)]
        Result GetBaasUserAvailabilityChangeNotifier(Out<detail.INotifier> pOutNotifier);

        [MethodId(105)]
        Result CheckNetworkServiceAvailabilityAsync(Out<detail.IAsyncContext> pOutContext, SystemProgramIdentification identification, [ProcessId] Bit64 pid);

        [MethodId(110)]
        Result StoreSaveDataThumbnail(Uid uid, ApplicationId appId, [NotNull] InBuffer imageBuffer);

        [MethodId(111)]
        Result ClearSaveDataThumbnail(Uid uid, ApplicationId appId);

        [MethodId(112)]
        Result LoadSaveDataThumbnail(Out<UInt32> pOutActualSize, [NotNull] OutBuffer pOut, Uid uid, ApplicationId appId);

        [MethodId(113)]
        Result GetSaveDataThumbnailExistence(Out<bool> pOut, Uid uid, ApplicationId appId);

        [MethodId(130)]
        Result ActivateOpenContextRetention(Out<detail.ISessionObject> pOut, ApplicationId appId);

        [MethodId(140)]
        Result ListQualifiedUsers(
            [Pre(@"_.GetLength() == ::nn::account::UserCountMax")] OutArray<Uid> outUsers, ApplicationId appId);

        [MethodId(190)]
        Result GetUserLastOpenedApplication(Out<ApplicationId> pOutApplicationId, Out<UInt32> pOutApplicationVersion, Uid uid);

        [MethodId(997)]
        Result DebugInvalidateUserResourceCache(Uid uid);
        [MethodId(998)]
        Result DebugSetUserStateClose(Uid uid);
        [MethodId(999)]
        Result DebugSetUserStateOpen(Uid uid);
    }

    // psel や shell 向けのインターフェイス
    [CppRefPath(PathInfo.Path)]
    public interface IAccountServiceForAdministrator : IAccountServiceForSystemService
    {
        [MethodId(200)]
        Result BeginUserRegistration(Out<Uid> pOutUid);

        [MethodId(201)]
        Result CompleteUserRegistration(Uid uid);

        [MethodId(202)]
        Result CancelUserRegistration(Uid uid);

        [MethodId(203)]
        Result DeleteUser(Uid uid);

        [MethodId(204)]
        Result SetUserPosition(Uid uid, int position);

        [MethodId(205)]
        Result GetProfileEditor(Out<profile.IProfileEditor> pOutProfile, Uid uid);

        [MethodId(206)]
        Result CompleteUserRegistrationForcibly(Uid uid);

        [MethodId(210)]
        Result CreateFloatingRegistrationRequest(Out<baas.IFloatingRegistrationRequest> pOutRequest, NativeHandle workMemory, UInt32 workMemorySize);

        [MethodId(230)]
        Result AuthenticateServiceAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(250)]
        Result GetBaasAccountAdministrator(Out<baas.IAdministrator> pOut, Uid uid);

        [MethodId(290)]
        Result ProxyProcedureForGuestLoginWithNintendoAccount(Out<nas.IOAuthProcedureForExternalNsa> pOut, detail.Uuid sessionId);

        [MethodId(291)]
        Result ProxyProcedureForFloatingRegistrationWithNintendoAccount(Out<nas.IOAuthProcedureForExternalNsa> pOut, detail.Uuid sessionId);

        [MethodId(299)]
        Result SuspendBackgroundDaemon(Out<detail.ISessionObject> pOut);
    }

    // ネットワークサービスアカウントへのアクセサ
    [CppRefPath(PathInfo.Path)]
    public interface IBaasAccessTokenAccessor : IServiceObject
    {
        [MethodId(0)]
        Result EnsureCacheAsync(Out<detail.IAsyncContext> pOutContext, Uid uid);

        [MethodId(1)]
        Result LoadCache(Out<UInt32> pOutActualSize, [NotNull] OutBuffer pOut, Uid uid);

        [MethodId(2)]
        Result GetDeviceAccountId(Out<UInt64> pOut, Uid uid);

        [MethodId(50)]
        Result RegisterNotificationTokenAsync(Out<detail.IAsyncContext> pOutContext, Uid uid, npns.NotificationToken nt);

        [MethodId(51)]
        Result UnregisterNotificationTokenAsync(Out<detail.IAsyncContext> pOutContext, Uid uid);
    }
}

namespace nn.account.detail
{
    internal static class PathInfo
    {
        public const string Prefix = "<";
        public const string Postfix = ">";
        public const string Root = @"nn/account/detail";
        public const string NotifierPath = Prefix + Root + @"/account_INotifier.sfdl.h" + Postfix;
        public const string AsyncContextPath = Prefix + Root + @"/account_IAsyncContext.sfdl.h" + Postfix;
        public const string SessionObjectPath = Prefix + Root + @"/account_ISessionObject.sfdl.h" + Postfix;
        public const string AsyncNetworkServiceLicenseKindContext = Prefix + Root + @"/account_IAsyncNetworkServiceLicenseKindContext.sfdl.h" + Postfix;
    }

    // 汎用通知インターフェイス
    [CppRefPath(PathInfo.NotifierPath)]
    public interface INotifier : IServiceObject
    {
        [MethodId(0)]
        Result GetSystemEvent(Out<NativeHandle> pOutHandle);
    }

    [CppRefPath(PathInfo.AsyncContextPath)]
    public interface IAsyncContext : IServiceObject
    {
        [MethodId(0)]
        Result GetSystemEvent(Out<NativeHandle> pOutHandle);

        [MethodId(1)]
        Result Cancel();

        [MethodId(2)]
        Result HasDone(Out<bool> pOut);

        [MethodId(3)]
        Result GetResult();
    }

    [CppRefPath(PathInfo.SessionObjectPath)]
    public interface ISessionObject : IServiceObject
    {
        [MethodId(999)]
        Result Dummy();
    }

    [CppRefPath(PathInfo.AsyncNetworkServiceLicenseKindContext)]
    public interface IAsyncNetworkServiceLicenseKindContext : IAsyncContext
    {
        [MethodId(100)]
        Result GetNetworkServiceLicenseKind(Out<int32_t> pOutLicense);
    }
}

namespace nn.account.http
{
    internal static class PathInfo
    {
        public const string Prefix = "<";
        public const string Postfix = ">";
        public const string Root = @"nn/account/http";
        public const string Path = Prefix + Root + @"/account_Interface.sfdl.h" + Postfix;
    }

    [CppRefPath(PathInfo.Path)]
    public interface IOAuthProcedure : IServiceObject
    {
        [MethodId(0)]
        Result PrepareAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(1)]
        Result GetRequest(Out<RequestUrl> pOutRequestUrl, Out<CallbackUri> pOutCallbackUri);

        [MethodId(2)]
        Result ApplyResponse([NotEmpty] InArray<char_t> response);

        [MethodId(3)]
        Result ApplyResponseAsync(Out<detail.IAsyncContext> pOutContext, [NotEmpty] InArray<char_t> response);

        [MethodId(10)]
        Result Suspend(Out<detail.Uuid> pOutSessionId);
    }
}

namespace nn.account.profile
{
    internal static class PathInfo
    {
        public const string Prefix = "<";
        public const string Postfix = ">";
        public const string Root = @"nn/account/profile";
        public const string Path = Prefix + Root + @"/account_Interface.sfdl.h" + Postfix;
    }

    // プロファイルの ReadOnly なアクセッサ
    [CppRefPath(PathInfo.Path)]
    public interface IProfile : IServiceObject
    {
        // 本体ストレージからの再読み込み
        [MethodId(0)]
        Result Get(Out<ProfileBase> pOutProfileBase, Out<UserData> pOutUserData);

        [MethodId(1)]
        Result GetBase(Out<ProfileBase> pOutProfileBase);

        // プロフィール画像の読み込み
        [MethodId(10)]
        Result GetImageSize(Out<UInt32> pOutSize);

        [MethodId(11)]
        Result LoadImage(Out<UInt32> pOutActualSize, [NotNull] OutBuffer pOutImage);
    }

    // プロファイルの Read/Write 可能なアクセッサ
    [CppRefPath(PathInfo.Path)]
    public interface IProfileEditor : IProfile
    {
        [MethodId(100)]
        Result Store(ProfileBase profileBase, UserData userData);

        [MethodId(101)]
        Result StoreWithImage(ProfileBase profileBase, UserData userData, [NotNull] InBuffer pImage);
    }
}

namespace nn.account.baas
{
    internal static class PathInfo
    {
        public const string Prefix = "<";
        public const string Postfix = ">";
        public const string Root = @"nn/account/baas";
        public const string Path = Prefix + Root + @"/account_Interface.sfdl.h" + Postfix;
    }

    [CppRefPath(PathInfo.Path)]
    public interface IGuestLoginRequest : IServiceObject
    {
        [MethodId(0)]
        Result GetSessionId(Out<detail.Uuid> pOut);

        [MethodId(21)]
        Result LoadIdTokenCache(Out<UInt32> pOutActualSize, [NotNull] OutBuffer pOut);

        [MethodId(12)]
        Result GetAccountId(Out<NetworkServiceAccountId> pOutId);

        [MethodId(13)]
        Result GetLinkedNintendoAccountId(Out<NintendoAccountId> pOut);

        [MethodId(14)]
        Result GetNickname(
            [Pre(@"_.GetLength() == ::nn::account::NicknameBytesMax + 1")] OutArray<char_t> nickname);

        [MethodId(15)]
        Result GetProfileImage(Out<UInt32> pOutActualSize, /* null は許容する*/ OutBuffer pOutImage);
    }

    [CppRefPath(PathInfo.Path)]
    public interface IFloatingRegistrationRequest : IGuestLoginRequest
    {
        [MethodId(100)]
        Result RegisterUser(Out<Uid> uid);

        [MethodId(101)]
        Result RegisterUserWithUid(Uid uid);

        [MethodId(102)]
        Result RegisterNetworkServiceAccountAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(103)]
        Result RegisterNetworkServiceAccountWithUidAsync(Out<detail.IAsyncContext> pOutContext, Uid uid);

        [MethodId(110)]
        Result SetSystemProgramIdentification(SystemProgramIdentification identification, [ProcessId] Bit64 pid);

        [MethodId(111)]
        Result EnsureIdTokenCacheAsync(Out<detail.IAsyncContext> pOutContext);
    }

    [CppRefPath(PathInfo.Path)]
    public interface IManagerBase : IServiceObject
    {
        [MethodId(0)]
        Result CheckAvailability();

        [MethodId(1)]
        Result GetAccountId(Out<NetworkServiceAccountId> pOutId);

        [MethodId(2)]
        Result EnsureIdTokenCacheAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(3)]
        Result LoadIdTokenCache(Out<UInt32> pOutActualSize, [NotNull] OutBuffer pOut);
    }

    [CppRefPath(PathInfo.Path)]
    public interface IManagerForApplication : IManagerBase
    {
        [MethodId(130)]
        Result GetNintendoAccountUserResourceCacheForApplication(
            Out<NintendoAccountId> pOutId,
            Out<nas.NasUserBaseForApplication> pOutBase,
            [NotNull] OutBuffer workBuffer);

        [MethodId(150)]
        Result CreateAuthorizationRequest(Out<nas.IAuthorizationRequest> pOut, NintendoAccountAuthorizationRequestParameters param, NativeHandle transferMemory, UInt32 size);

        [MethodId(160)]
        Result StoreOpenContext();

        [MethodId(170)]
        Result LoadNetworkServiceLicenseKindAsync(Out<detail.IAsyncNetworkServiceLicenseKindContext> pOut);
    }

    [CppRefPath(PathInfo.Path)]
    public interface IManagerForSystemService : IManagerBase
    {
        [MethodId(100)]
        Result SetSystemProgramIdentification(SystemProgramIdentification identification, [ProcessId] Bit64 pid);

        [MethodId(110)]
        Result GetServiceEntryRequirementCache(Out<UInt32> pOut, ApplicationId appId);
        [MethodId(111)]
        Result InvalidateServiceEntryRequirementCache(ApplicationId appId);
        [MethodId(112)]
        Result InvalidateTokenCache(ApplicationId appId);

        [MethodId(120)]
        Result GetNintendoAccountId(Out<NintendoAccountId> pOutId);

        [MethodId(130)]
        Result GetNintendoAccountUserResourceCache(
            Out<NintendoAccountId> pOutId,
            Out<nas.NasUserBase> pOutBase,
            [NotNull] OutBuffer workBuffer);

        [MethodId(131)]
        Result RefreshNintendoAccountUserResourceCacheAsync(Out<detail.IAsyncContext> pOut);

        [MethodId(132)]
        Result RefreshNintendoAccountUserResourceCacheAsyncIfSecondsElapsed(Out<bool> pOutMatched, Out<detail.IAsyncContext> pOutContext, UInt32 seconds);

        [MethodId(140)]
        Result GetNetworkServiceLicenseCache(Out<int32_t> pOutLicense, Out<time.PosixTime> pOutExpiration);

        [MethodId(141)]
        Result RefreshNetworkServiceLicenseCacheAsync(Out<detail.IAsyncContext> pOut);

        [MethodId(142)]
        Result RefreshNetworkServiceLicenseCacheAsyncIfSecondsElapsed(Out<bool> pOutMatched, Out<detail.IAsyncContext> pOutContext, UInt32 seconds);

        [MethodId(150)]
        Result CreateAuthorizationRequest(
            Out<nas.IAuthorizationRequest> pOut,
            nas.NasClientInfo clientInfo, NintendoAccountAuthorizationRequestParameters param,
            NativeHandle transferMemory, UInt32 size);
    }

    [CppRefPath(PathInfo.Path)]
    public interface IAdministrator : IManagerForSystemService
    {
        [MethodId(200)]
        Result IsRegistered(Out<bool> pOut);

        [MethodId(201)]
        Result RegisterAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(202)]
        Result UnregisterAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(203)]
        Result DeleteRegistrationInfoLocally();

        [MethodId(220)]
        Result SynchronizeProfileAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(222)]
        Result SynchronizeProfileAsyncIfSecondsElapsed(
            Out<bool> pOutMatched,
            Out<detail.IAsyncContext> pOutContext,
            UInt32 seconds);

        [MethodId(221)]
        Result UploadProfileAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(250)]
        Result IsLinkedWithNintendoAccount(Out<bool> pOut);

        [MethodId(251)]
        Result CreateProcedureToLinkWithNintendoAccount(Out<nas.IOAuthProcedureForNintendoAccountLinkage> pOut);

        [MethodId(252)]
        Result ResumeProcedureToLinkWithNintendoAccount(Out<nas.IOAuthProcedureForNintendoAccountLinkage> pOut, detail.Uuid sessionId);

        [MethodId(255)]
        Result CreateProcedureToUpdateLinkageStateOfNintendoAccount(Out<http.IOAuthProcedure> pOut);

        [MethodId(256)]
        Result ResumeProcedureToUpdateLinkageStateOfNintendoAccount(Out<http.IOAuthProcedure> pOut, detail.Uuid sessionId);

        [MethodId(260)]
        Result CreateProcedureToLinkNnidWithNintendoAccount(Out<http.IOAuthProcedure> pOut);

        [MethodId(261)]
        Result ResumeProcedureToLinkNnidWithNintendoAccount(Out<http.IOAuthProcedure> pOut, detail.Uuid sessionId);

        [MethodId(280)]
        Result ProxyProcedureToAcquireApplicationAuthorizationForNintendoAccount(Out<http.IOAuthProcedure> pOut, detail.Uuid sessionId);

        [MethodId(300)]
        Result TryRecoverNintendoAccountUserStateAsync(Out<detail.IAsyncContext> pOut);

        [MethodId(997)]
        Result DebugUnlinkNintendoAccountAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(998)]
        Result DebugSetAvailabilityErrorDetail(UInt32 detail);
    }
}

namespace nn.account.nas
{
    internal static class PathInfo
    {
        public const string Prefix = "<";
        public const string Postfix = ">";
        public const string Root = @"nn/account/nas";
        public const string Path = Prefix + Root + @"/account_Interface.sfdl.h" + Postfix;
    }

    [CppRefPath(PathInfo.Path)]
    public interface IOAuthProcedureForNintendoAccountLinkage : http.IOAuthProcedure
    {
        [MethodId(100)]
        Result GetRequestWithTheme(Out<RequestUrl> pOutRequestUrl, Out<CallbackUri> pOutCallbackUri, Int32 themeRaw);

        [MethodId(101)]
        Result IsNetworkServiceAccountReplaced(Out<bool> pOut);
    }

    [CppRefPath(PathInfo.Path)]
    public interface IAuthorizationRequest : IServiceObject
    {
        [MethodId(0)]
        Result GetSessionId(Out<detail.Uuid> pOut);

        [MethodId(10)]
        Result InvokeWithoutInteractionAsync(Out<detail.IAsyncContext> pOutContext);

        [MethodId(19)]
        Result IsAuthorized(Out<bool> pOut);

        [MethodId(20)]
        Result GetAuthorizationCode(Out<UInt32> pOutActualSize, [NotNull] OutBuffer pOut);

        [MethodId(21)]
        Result GetIdToken(Out<UInt32> pOutActualSize, [NotNull] OutBuffer pOut);

        [MethodId(22)]
        Result GetState(Out<State> pOut);
    }

    [CppRefPath(PathInfo.Path)]
    public interface IOAuthProcedureForExternalNsa : http.IOAuthProcedure
    {
        [MethodId(100)]
        Result GetAccountId(Out<NetworkServiceAccountId> pOutId);

        [MethodId(101)]
        Result GetLinkedNintendoAccountId(Out<NintendoAccountId> pOut);

        [MethodId(102)]
        Result GetNickname(
            [Pre(@"_.GetLength() == ::nn::account::NicknameBytesMax + 1")] OutArray<char_t> nickname);

        [MethodId(103)]
        Result GetProfileImage(Out<UInt32> pOutActualSize, /* null は許容する*/ OutBuffer pOutImage);
    }
}
