﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/ndas/account_NdasOperator.h>

#include <nn/account/account_RuntimeResource.h>
#include <nn/account/ndas/account_ResultForNdas.h>
#include <nn/dauth/dauth_Result.h>
#include <nn/http/http_Result.h>

#include "../http/account_HttpUtil.h"

namespace nn { namespace account { namespace ndas {

Result NdasOperator::HandleDeviceAuthenticationError(const Result& result) NN_NOEXCEPT
{
    if (dauth::ResultCancelled::Includes(result))
    {
        NN_RESULT_THROW(ResultCancelled());
    }
    else if (dauth::ResultNdasStatusNo0007::Includes(result))
    {
        NN_RESULT_THROW(ResultUnacceptableSystemVersion());
    }

#define CONVERT_RESULT_IF_MATCH(name, r) \
    if (::nn::dauth:: name ::Includes(r)) \
    { \
        NN_RESULT_THROW(name()); \
    }

    else CONVERT_RESULT_IF_MATCH(ResultNdasStatusNo0004, result)
    else CONVERT_RESULT_IF_MATCH(ResultNdasStatusNo0008, result)
    else CONVERT_RESULT_IF_MATCH(ResultNdasStatusNo0009, result)
    else CONVERT_RESULT_IF_MATCH(ResultNdasStatusNo0010, result)
    else CONVERT_RESULT_IF_MATCH(ResultNdasStatusNo0011, result)
    else CONVERT_RESULT_IF_MATCH(ResultNdasStatusNo0012, result)
    else CONVERT_RESULT_IF_MATCH(ResultNdasStatusNo0013, result)
    else CONVERT_RESULT_IF_MATCH(ResultNdasStatusNo0014, result)
    else CONVERT_RESULT_IF_MATCH(ResultNdasStatusNo0015, result)
#undef CONVERT_RESULT_IF_MATCH

    else if (dauth::ResultNetworkCommunicationError::Includes(result))
    {
        // dauth のそれ以外の通信エラーは適当な nn::account エラーに丸める
        NN_ACCOUNT_HTTP_CONVERT_HTTP_STATUS_RESULT_IF_MATCH(::nn::http, result);
        NN_RESULT_THROW(ResultNdasCommunicationError());
    }
    NN_RESULT_THROW(result);
}

Result NdasOperator::AuthenticateServiceImpl(uint64_t clientId, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    return HandleDeviceAuthenticationError(
        m_NdasDriver.EnsureDeviceAuthenticationCache(clientId, false /* force */, pCancellable));
}

Result NdasOperator::ReAuthenticateServiceImpl(uint64_t clientId, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    return HandleDeviceAuthenticationError(
        m_NdasDriver.EnsureDeviceAuthenticationCache(clientId, true /* force */, pCancellable));
}

Result NdasOperator::AuthenticateApplication(const detail::ApplicationInfo& appInfo, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_DO(AuthenticateServiceForApplications(resource, pCancellable));
    return m_NdasDriver.EnsureApplicationAuthenticationCache(
        appInfo, false /* force */,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable);
}
Result NdasOperator::ReAuthenticateApplication(const detail::ApplicationInfo& appInfo, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_DO(AuthenticateServiceForApplications(resource, pCancellable));
    return m_NdasDriver.EnsureApplicationAuthenticationCache(
        appInfo, true /* force */,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable);
}

}}} // ~namespace nn::account::ndas
