﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/ndas/account_NdasDriver.h>

#include "../detail/account_CacheUtil.h"
#include "account_AuthenticationAdaptor.h"
#include <nn/account/http/account_CurlInputStream.h>
#include <nn/account/http/account_ResultForHttp.h>
#include <nn/account/json/account_RapidJsonApi.h>

#include <algorithm>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/es/es_Api.h>
#include <nn/dauth/dauth_Api.h>
#endif
#include <nn/es/es_Result.h>
#include <nn/es/es_MaxTicketSize.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace account { namespace ndas {

namespace {
static const char AltairUrl[] = "https://aauth-%.ndas.srv.nintendo.net/v2-44cd4221f90742b5f37a4948b37dacf024d0bb14dde86db0af20ec300a36a0fe/application_auth_token";

static const size_t MaxTicketKeySize = 256;

Result LoadDeviceAuthenticationCacheImpl(size_t* pOutSizeActual, char* buffer, size_t bufferSize, uint64_t clientId) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    TimeSpan tmpExpiration;
    int length;
    NN_RESULT_DO(dauth::AcquireDeviceAuthenticationToken(&tmpExpiration, &length, buffer, bufferSize, clientId, false, nullptr));
    *pOutSizeActual = static_cast<size_t>(length);
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutSizeActual);
    NN_UNUSED(buffer);
    NN_UNUSED(bufferSize);
    NN_UNUSED(clientId);
    NN_RESULT_THROW(ResultNotImplemented());
#endif
}

Result LoadGameCardDeviceCert(
    int* pOutLengthActual, char* buffer, size_t bufferSize,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= fs::GameCardDeviceCertificateSize);

    NN_RESULT_THROW_UNLESS(fs::IsGameCardInserted(), ResultInvalidProtocolAccess());

    static const char g_GameCardMountName[] = "@card";
    fs::GameCardHandle handle;
    NN_RESULT_DO(fs::GetGameCardHandle(&handle));
    NN_RESULT_DO(fs::MountGameCardPartition(g_GameCardMountName, handle, fs::GameCardPartition::Secure));
    NN_UTIL_SCOPE_EXIT
    {
        fs::Unmount(g_GameCardMountName);
    };
    NN_RESULT_DO(fs::GetGameCardDeviceCertificate(workBuffer, workBufferSize, handle));
    auto status = util::Base64::ToBase64String(
        buffer, bufferSize, workBuffer, fs::GameCardDeviceCertificateSize, util::Base64::Mode_UrlSafe);
    NN_RESULT_THROW_UNLESS(status == util::Base64::Status_Success, ResultNotSupported());

    *pOutLengthActual = static_cast<int>(strnlen(buffer, bufferSize));
    NN_RESULT_SUCCESS;
}

Result LoadTicket(
    int* pOutCertLength, char* certBuffer, size_t certBufferSize,
    int* pOutCertKeyLength, char* certKeyBuffer, size_t certKeyBufferSize,
    const detail::ApplicationInfo& appInfo,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    struct Buffer
    {
        char encryptedTicket[es::MaxEncryptedTicketSize];
        char encryptionKey[MaxTicketKeySize];
    };
    NN_SDK_REQUIRES(workBufferSize >= sizeof(Buffer));
    Buffer& buffer = *reinterpret_cast<Buffer*>(workBuffer);
    NN_UNUSED(workBufferSize);

    // fs から RightsId を取得する
    es::RightsIdIncludingKeyId rightsId;
    {
        fs::RightsId tmp;
        NN_RESULT_DO(fs::GetRightsId(&tmp, appInfo.launchProperty.id, appInfo.launchProperty.storageId));
        NN_SDK_LOG(
            "[nn::account] RightsId: %02x%02x%02x%02x_%02x%02x%02x%02x_%02x%02x%02x%02x_%02x%02x%02x%02x\n",
            tmp.data[ 0], tmp.data[ 1], tmp.data[ 2], tmp.data[ 3], tmp.data[ 4], tmp.data[ 5], tmp.data[ 6], tmp.data[ 7],
            tmp.data[ 8], tmp.data[ 9], tmp.data[10], tmp.data[11], tmp.data[12], tmp.data[13], tmp.data[14], tmp.data[15]);
        rightsId = es::RightsIdIncludingKeyId::Construct(tmp);
    }

    // チケットの大きさと種別の判定
    es::TicketId ticketId;
    size_t outSize;
    NN_RESULT_DO(es::GetEncryptedTicketData(
        &ticketId, &outSize, buffer.encryptedTicket, sizeof(buffer.encryptedTicket),
        buffer.encryptionKey, sizeof(buffer.encryptionKey),
        rightsId));
    NN_SDK_ASSERT(outSize <= sizeof(buffer.encryptedTicket));

    // Base64 エンコード
    auto status = util::Base64::ToBase64String(
        certBuffer, certBufferSize, buffer.encryptedTicket, outSize, util::Base64::Mode_UrlSafe);
    NN_RESULT_THROW_UNLESS(status == util::Base64::Status_Success, ResultNotSupported());
    status = util::Base64::ToBase64String(
        certKeyBuffer, certKeyBufferSize, buffer.encryptionKey, sizeof(buffer.encryptionKey), util::Base64::Mode_UrlSafe);
    NN_RESULT_THROW_UNLESS(status == util::Base64::Status_Success, ResultNotSupported());

    *pOutCertLength = strnlen(certBuffer, certBufferSize);
    *pOutCertKeyLength = strnlen(certKeyBuffer, certKeyBufferSize);
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(certBuffer);
    NN_UNUSED(certBufferSize);
    NN_UNUSED(certKeyBuffer);
    NN_UNUSED(certKeyBufferSize);
    NN_UNUSED(workBuffer);
    NN_UNUSED(workBufferSize);
    NN_UNUSED(&appInfo);

    *pOutCertLength = *pOutCertKeyLength = 0;
    NN_RESULT_SUCCESS;
#endif
}

Result MakePostDataForGameCard(
    char* postDataBuffer, size_t postDataBufferSize,
    const detail::ApplicationInfo& appInfo,
    const char* cert, int certLength,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= detail::NdasDevAuthTokenSizeMax);

    size_t devAuthLength;
    NN_RESULT_DO(LoadDeviceAuthenticationCacheImpl(&devAuthLength, workBuffer, workBufferSize, ClientIdForApplications));

    static const char PostDataFormat[] =
        "application_id=%016llx"
        "&application_version=%08x"
        "&device_auth_token=%.*s"
        "&media_type=GAMECARD"
        "&cert=%.*s";
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        appInfo.launchProperty.id.value, appInfo.launchProperty.version,
        devAuthLength, workBuffer,
        certLength, cert);
    NN_ABORT_UNLESS(l < static_cast<int>(postDataBufferSize));
    NN_RESULT_SUCCESS;
}

Result MakePostDataForDigital(
    char* postDataBuffer, size_t postDataBufferSize,
    const detail::ApplicationInfo& appInfo,
    const char* cert, int certLength,
    const char* certKey, int certKeyLength,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= detail::NdasDevAuthTokenSizeMax);

    size_t devAuthLength;
    NN_RESULT_DO(LoadDeviceAuthenticationCacheImpl(&devAuthLength, workBuffer, workBufferSize, ClientIdForApplications));

    static const char PostDataFormat[] =
        "application_id=%016llx"
        "&application_version=%08x"
        "&device_auth_token=%.*s"
        "&media_type=DIGITAL"
        "&cert=%.*s"
        "&cert_key=%.*s";
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        appInfo.launchProperty.id.value, appInfo.launchProperty.version,
        devAuthLength, workBuffer,
        certLength, cert,
        certKeyLength, certKey);
    NN_ABORT_UNLESS(l < static_cast<int>(postDataBufferSize));
    NN_RESULT_SUCCESS;
}

Result MakePostDataForSystem(
    char* postDataBuffer, size_t postDataBufferSize,
    const detail::ApplicationInfo& appInfo,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= detail::NdasDevAuthTokenSizeMax);

    size_t devAuthLength;
    NN_RESULT_DO(LoadDeviceAuthenticationCacheImpl(&devAuthLength, workBuffer, workBufferSize, ClientIdForApplications));

    static const char PostDataFormat[] =
        "application_id=%016llx"
        "&application_version=%08x"
        "&device_auth_token=%.*s"
        "&media_type=SYSTEM";
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        appInfo.launchProperty.id.value, appInfo.launchProperty.version,
        devAuthLength, workBuffer);
    NN_ABORT_UNLESS(l < static_cast<int>(postDataBufferSize));
    NN_RESULT_SUCCESS;
}

Result MakePostDataForNoCert(
    char* postDataBuffer, size_t postDataBufferSize,
    const detail::ApplicationInfo& appInfo,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= detail::NdasDevAuthTokenSizeMax);

    size_t devAuthLength;
    NN_RESULT_DO(LoadDeviceAuthenticationCacheImpl(&devAuthLength, workBuffer, workBufferSize, ClientIdForApplications));

    static const char PostDataFormat[] =
        "application_id=%016llx"
        "&application_version=%08x"
        "&device_auth_token=%.*s"
        "&media_type=NO_CERT";
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        appInfo.launchProperty.id.value, appInfo.launchProperty.version,
        devAuthLength, workBuffer);
    NN_ABORT_UNLESS(l < static_cast<int>(postDataBufferSize));
    NN_RESULT_SUCCESS;
}

#define NN_ACCOUNT_MAX_3(a, b, c) NN_ACCOUNT_MAX(NN_ACCOUNT_MAX((a), (b)), (c))
static const size_t PostDataBufferSizeMax = 0
    + detail::NdasDevAuthTokenSizeMax
    + (NN_ACCOUNT_MAX(
        fs::GameCardDeviceCertificateSize,
        es::MaxEncryptedTicketSize + MaxTicketKeySize) * 4) / 3 + 8
    + 512;
static const size_t CertBufferSizeMax = 0
    + (NN_ACCOUNT_MAX(
        fs::GameCardDeviceCertificateSize,
        es::MaxEncryptedTicketSize) * 4) / 3 + 8;
static const size_t CertKeyBufferSizeMax = (MaxTicketKeySize * 4) / 3 + 8;
static const size_t WorkBufferSizeMax = 0
    + NN_ACCOUNT_MAX_3(
        detail::NdasDevAuthTokenSizeMax,
        fs::GameCardDeviceCertificateSize,
        es::MaxEncryptedTicketSize + MaxTicketKeySize);
#undef NN_ACCOUNT_MAX_3

struct PostDataWorkBuffer
{
    char certBuffer[CertBufferSizeMax];
    char certKeyBuffer[CertKeyBufferSizeMax];
    char io[WorkBufferSizeMax];
};

Result MakePostData(char* buffer, size_t bufferSize, const detail::ApplicationInfo& appInfo, PostDataWorkBuffer* workBuffer) NN_NOEXCEPT
{
    switch (appInfo.mediaType)
    {
    case detail::ApplicationMediaType::GameCard:
        {
            int certLength;
            NN_RESULT_DO(LoadGameCardDeviceCert(
                &certLength, workBuffer->certBuffer, sizeof(workBuffer->certBuffer),
                workBuffer->io, sizeof(workBuffer->io)));
            NN_RESULT_DO(MakePostDataForGameCard(
                buffer, bufferSize,
                appInfo, workBuffer->certBuffer, certLength,
                workBuffer->io, sizeof(workBuffer->io)));
        }
        break;
    case detail::ApplicationMediaType::Digital:
        {
            int certLength;
            int certKeyLength;
            auto r = LoadTicket(
                &certLength, workBuffer->certBuffer, sizeof(workBuffer->certBuffer),
                &certKeyLength, workBuffer->certKeyBuffer, sizeof(workBuffer->certKeyBuffer),
                appInfo,
                workBuffer->io, sizeof(workBuffer->io));
            // TODO: チケットロード失敗時に必ず API 呼び出しを失敗するようにする。 (SIGLO-42944 完了後)
            // TORIAEZU: 今は nspd 起動時に rights ID も eTicket も取得できないので、その場合はパラメータを省略する
            NN_RESULT_THROW_UNLESS(false
                || r.IsSuccess()
                || fs::ResultTargetNotFound::Includes(r)
                || es::ResultRightsIdNotFound::Includes(r), r);
            if (!r.IsSuccess())
            {
                NN_SDK_LOG("[nn::account] LoadTicket failed with %03d-%04d\n", r.GetModule(), r.GetDescription());
                NN_RESULT_DO(MakePostDataForNoCert(
                    buffer, bufferSize,
                    appInfo, workBuffer->io, sizeof(workBuffer->io)));
                break;
            }

            NN_RESULT_DO(MakePostDataForDigital(
                buffer, bufferSize,
                appInfo,
                workBuffer->certBuffer, certLength,
                workBuffer->certKeyBuffer, certKeyLength,
                workBuffer->io, sizeof(workBuffer->io)));
        }
        break;
    case detail::ApplicationMediaType::System:
        NN_RESULT_DO(MakePostDataForSystem(
            buffer, bufferSize,
            appInfo, workBuffer->io, sizeof(workBuffer->io)));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}
} // ~namespace nn::account::ndas::<anonymous>

const size_t NdasDriver::RequiredBufferSize = 8 * 1024u;

NdasDriver::NdasDriver(
    ApplicationAuthenticationCache& appAuthCache) NN_NOEXCEPT
    : m_AppAuthCache(appAuthCache)
{
}

Result NdasDriver::LoadDeviceAuthenticationCache(size_t* pOutSizeActual, char* buffer, size_t bufferSize, uint64_t clientId) const NN_NOEXCEPT
{
    return LoadDeviceAuthenticationCacheImpl(pOutSizeActual, buffer, bufferSize, clientId);
}

Result NdasDriver::EnsureDeviceAuthenticationCache(uint64_t clientId, bool force, detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return dauth::EnsureDeviceAuthenticationTokenCache(clientId, force, pCancellable);
#else
    NN_UNUSED(clientId);
    NN_UNUSED(force);
    NN_UNUSED(pCancellable);
    NN_RESULT_THROW(ResultNotImplemented());
#endif
}

Result NdasDriver::EnsureApplicationAuthenticationCache(
    const detail::ApplicationInfo& appInfo, bool force,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // バッファ定義
    struct Buffer
    {
        char postDataBuffer[PostDataBufferSizeMax];
        union
        {
            PostDataWorkBuffer postDataWorkBuffer;
            struct
            {
                char stringBuffer[detail::NdasAppAuthTokenSizeMax + 8];
                char inputBuffer[detail::IoBufferSizeMin]; // 末尾に置くこと
            } io;
        } u;
    };
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES(appInfo);
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // 既に有効な認証トークンがあるなら、即時返る。
    if (!force && m_AppAuthCache.IsAvailable(appInfo))
    {
        NN_RESULT_SUCCESS;
    }

    // リクエストの作成
    auto& buffer = *reinterpret_cast<Buffer*>(rawBuffer);
    NN_RESULT_DO(MakePostData(buffer.postDataBuffer, sizeof(buffer.postDataBuffer), appInfo, &buffer.u.postDataWorkBuffer));

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer.u.io.stringBuffer, sizeof(buffer.u.io.stringBuffer));
    stream.SetInputBuffer(buffer.u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));
    NN_RESULT_DO(stream.SetHttpPost(buffer.postDataBuffer, false));
    NN_RESULT_DO(stream.SetUrl(AltairUrl));

    // 通信の実行とデータの適用
    ApplicationAuthenticationAdaptor adaptor(m_AppAuthCache, appInfo);
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    return adaptor.Adapt(stream.GetHttpCode());
}

}}} // ~namespace nn::account::ndas
