﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/nas/account_NasUserDriver.h>

#include "account_NasAuthorizationHeader.h"
#include "account_NasResourceResolver.h"
#include "account_NasUserAdaptor.h"
#include "../detail/account_CacheUtil.h"
#include <nn/account/http/account_CurlInputStream.h>
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/http/account_ResultForHttp.h>

#include <cstring>

#include <nn/util/util_ScopeExit.h>

namespace nn {
namespace account {
namespace nas {

namespace {

Result SetupStreamForGetSelf(
    http::CurlInputStream& stream,
    struct curl_slist** headers,
    const detail::Uuid& accessTokenCacheId,
    const detail::AbstractLocalStorage& storage,
    char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForAuthorizationHeader);

    // Authorization ヘッダの準備
    NN_RESULT_DO(CreateAuthorizationHeader(workBuffer, workBufferSize, accessTokenCacheId, storage));
    *headers = curl_slist_append(*headers, workBuffer);
    NN_RESULT_THROW_UNLESS(*headers != nullptr, http::ResultCurlSlistAppendFailure());

    // リクエスト生成
    NN_RESULT_DO(stream.SetHeaders(*headers));

    // URL の作成
    char url[256];
    auto l = NasResourceResolver::GetUrlForUserResource(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l)  < sizeof(url));
    NN_UNUSED(l);
    NN_RESULT_DO(stream.SetUrl(url));

    NN_RESULT_SUCCESS;
}

} // ~namespace nn::account::nas::<anonymous>

const size_t NasUserDriver::RequiredBufferSize = 8 * 1024u;

NasUserDriver::NasUserDriver(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : m_Storage(storage)
{
}

Result NasUserDriver::AcquireUserResourceCache(
    NintendoAccountId* pOutNintendoAccountId, NasUserResourceCache& nasUserResourceCache,
    const detail::Uuid& accessTokenCache,
    CURL* curlHandle, void* rawBuffer, size_t bufferSize,
    const detail::Cancellable* pCancellable) const NN_NOEXCEPT
{
    // バッファ定義
    struct Buffer
    {
        union
        {
            char workBuffer[RequiredBufferSizeForAuthorizationHeader];
            struct
            {
                char stringBuffer[1024]; // User リソースのうち、欲しいフィールドの最大長
                char outputBuffer[1024];;
                char inputBuffer[detail::IoBufferSizeMin];
            } io;
        } u;
    };;
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSize);
    NN_STATIC_ASSERT(std::alignment_of<Buffer>::value <= std::alignment_of<std::max_align_t>::value);

    // 事前条件
    NN_SDK_REQUIRES_NOT_NULL(pOutNintendoAccountId);
    NN_SDK_REQUIRES(accessTokenCache);
    NN_SDK_REQUIRES(curlHandle != nullptr);
    NN_SDK_REQUIRES(rawBuffer != nullptr);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<std::max_align_t>::value == 0);
    NN_SDK_REQUIRES(bufferSize >= sizeof(Buffer));

    // リクエストの作成
    auto* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    http::CurlInputStream stream(curlHandle, pCancellable);
    NN_RESULT_DO(stream.Initialize());
    stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
    stream.SetInputBuffer(buffer->u.io.inputBuffer, bufferSize - offsetof(Buffer, u.io.inputBuffer));

    NN_RESULT_DO(SetupStreamForGetSelf(
        stream, &headers, accessTokenCache, m_Storage,
        buffer->u.workBuffer, sizeof(buffer->u.workBuffer)));

    // 通信の実行と結果の適用
    NasUserAdaptor adaptor(m_Storage, buffer->u.io.outputBuffer, sizeof(buffer->u.io.outputBuffer));
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(json::ImportJsonByRapidJson(adaptor, stream, pCancellable));
    stream.Close();
    NN_RESULT_DO(adaptor.Adapt(stream.GetHttpCode()));

    // 結果の取得
    NN_RESULT_DO(adaptor.PullUserResourceCache(pOutNintendoAccountId, nasUserResourceCache));
    NN_RESULT_SUCCESS;
}

}
}
} // ~namespace nn::account::nas
