﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/nas/account_NasSessionCache.h>

#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/detail/account_InternalTypes.h>
#include "../detail/account_CacheUtil.h"

#include <mutex>

#include <nn/os/os_Tick.h>
#include <nn/util/util_ScopeExit.h>

namespace nn {
namespace account {
namespace nas {

/* --------------------------------------------------------------------------------------------
    NasPkceSessionOperator
*/
bool PkceSessionOperator::IsExpired(int64_t expiration) NN_NOEXCEPT
{
    // セッションは失効しないようにしておく (とりあえず)
    NN_UNUSED(expiration);
    return false;
}
void PkceSessionOperator::DeleteCacheData(const std::pair<std::pair<PkceSessionTag, Uid>, detail::Uuid>& data, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    detail::CacheUtil::DeleteCacheFile(data.second, storage);
}

/* --------------------------------------------------------------------------------------------
    NasPkceSessionCache
*/

Result PkceSessionCache::StoreImpl(
    detail::Uuid* pOut,
    PkceSessionTag tag, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT
{
    std::lock_guard<Base> lock(*static_cast<Base*>(this));
    Base::InvalidateUnsafe();

    detail::Uuid cacheId = detail::InvalidUuid;
    NN_UTIL_SCOPE_EXIT
    {
        if (cacheId)
        {
            detail::CacheUtil::DeleteCacheFile(cacheId, Base::GetStorageRef());
        }
    };

    NN_RESULT_DO(detail::CacheUtil::StoreCacheFile(&cacheId, &prop, sizeof(prop), Base::GetStorageRef()));
    std::pair<PkceSessionTag, Uid> t(tag, user);
    Base::StoreUnsafe(os::GetSystemTick().ToTimeSpan().GetSeconds(), std::pair<std::pair<PkceSessionTag, Uid>, detail::Uuid>(t, cacheId));

    *pOut = cacheId;
    cacheId = detail::InvalidUuid;
    NN_RESULT_SUCCESS;
}

Result PkceSessionCache::ExtractImpl(
    NasPkceProperty* pOutProp,
    PkceSessionTag tag, const Uid& user, const detail::Uuid& cacheId) NN_NOEXCEPT
{
    std::lock_guard<Base> lock(*static_cast<Base*>(this));

    std::pair<std::pair<PkceSessionTag, Uid>, detail::Uuid> data;
    NN_RESULT_DO(Base::GetUnsafe(&data));
    NN_RESULT_THROW_UNLESS(data.first.first == tag && data.first.second == user, ResultNintendoAccountAuthorizationSessionNotExist());
    NN_RESULT_THROW_UNLESS(data.second == cacheId, ResultNintendoAccountAuthorizationSessionNotExist());

    size_t actualSize;
    NN_RESULT_DO((detail::CacheUtil::LoadCacheFile<sizeof(NasPkceProperty), ResultNotSupported>(
        &actualSize, pOutProp, sizeof(*pOutProp), cacheId, Base::GetStorageRef())));
    NN_SDK_ASSERT(actualSize == sizeof(NasPkceProperty));
    Base::InvalidateUnsafe();
    NN_RESULT_SUCCESS;
}

} // ~namespace nn::account::nas
}
}
