﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_NasOp2Adaptor.h"

#include <nn/nn_SdkLog.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/nas/account_ResultForNas.h>
#include <nn/result/result_HandlingUtility.h>
#include "../http/account_HttpUtil.h"

namespace nn { namespace account { namespace nas {

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
        NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

NasOp2MembershipAdaptor::NasOp2MembershipAdaptor() NN_NOEXCEPT
    : m_MembershipActiveEntry("$.membership.active")
    , m_MembershipExpiresAtEntry("$.membership.expires_at")
    , m_IsAdapted(false)
{
}

void NasOp2MembershipAdaptor::GetMembership(NetworkServiceLicense* pOutLicense, time::PosixTime* pOutExpiration) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsAdapted);

    if (!m_Active)
    {
        *pOutLicense = NetworkServiceLicense_None;
        *pOutExpiration = time::PosixTime();
        return;
    }

    *pOutLicense = NetworkServiceLicense_Common;
    time::PosixTime posixTime;
    posixTime.value = m_ExpirationAbs / 1000; // msec -> sec
    *pOutExpiration = posixTime;
}

Result NasOp2MembershipAdaptor::Adapt(int32_t httpCode) NN_NOEXCEPT
{
    if (httpCode != 200)
    {
        switch (httpCode)
        {
        case 400:
            NN_RESULT_THROW(ResultNasOp2Status400());
        case 403:
            NN_RESULT_THROW(ResultNasOp2Status403());
        case 503:
            NN_RESULT_THROW(ResultNasOp2Status503());
        default:
            NN_RESULT_THROW(http::HandleHttpStatusCode(httpCode));
        }
    }

    if (!(m_MembershipActiveEntry && (m_MembershipExpiresAtEntry || !m_Active)))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: GET OP2 Membership Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_MembershipActiveEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_MembershipExpiresAtEntry);
        NN_RESULT_THROW(ResultNasDataBroken());
    }

    m_IsAdapted = true;
    NN_RESULT_SUCCESS;
}

void NasOp2MembershipAdaptor::Update(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT
{
    if (m_MembershipActiveEntry.CanAccept(jsonPath))
    {
        m_Active = value;
        m_MembershipActiveEntry.MarkAccepted();
    }
}
void NasOp2MembershipAdaptor::Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_MembershipExpiresAtEntry.CanAccept(jsonPath))
    {
        m_ExpirationAbs = value;
        m_MembershipExpiresAtEntry.MarkAccepted();
    }
}

}}} // ~namespace nn::account::nas
