﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_OAuthUtil.h"

#include <random>

#include <nn/nn_Common.h>
#include "../detail/account_ByteUtil.h"
#include "../detail/account_UuidUtil.h"

namespace nn { namespace account { namespace http {

bool IsValidScope(const char* scope, size_t scopeSize) NN_NOEXCEPT
{
    // https://tools.ietf.org/html/rfc6749#appendix-A
    for (auto i = 0u; i < scopeSize; ++ i)
    {
        auto c = scope[i];
        if (c == '\0')
        {
            // 1 文字以上、かつ終端されている
            return i >= 1;
        }
        auto isNqchar = false
            || c == 0x21
            || (0x23 <= c && c <= 0x5B)
            || (0x5D <= c && c <= 0x7E);
        if (!(isNqchar || c == 0x20)) // scope_token or 区切り文字
        {
            return false;
        }
    }
    return false;
}

bool IsValidState(const char* state, size_t stateSize) NN_NOEXCEPT
{
    // https://tools.ietf.org/html/rfc6749#appendix-A
    for (auto i = 0u; i < stateSize; ++ i)
    {
        auto c = state[i];
        if (c == '\0')
        {
            // 1 文字以上、かつ終端されている
            return i >= 1;
        }
        if (!((c >= 0x20) && c <= (0x7E)))
        {
            return false;
        }
    }
    return false;
}

bool IsValidNonce(const char* nonce, size_t nonceSize) NN_NOEXCEPT
{
    // http://openid.net/specs/openid-connect-core-1_0.html
    // によると Nonce は 'string' らしいが、ここでは値域を限定する。
    return IsValidState(nonce, nonceSize);
}

void GenerateStateString(char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    // State srting は UUID を使うことにする
    detail::Uuid source[] = {detail::GenerateUuid()};

    NN_SDK_ASSERT(bufferSize >= 128); // TORIAEZU
    detail::GetEncodedSha256Hash(buffer, bufferSize, source, sizeof(source), util::Base64::Mode_UrlSafe);
}

void GenerateCodeVerifier(CodeVerifier* pCodeVerifier) NN_NOEXCEPT
{
    static const char Table[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYXabcdefghijklmnopqrstuvwxyz-._~";
    NN_STATIC_ASSERT(sizeof(Table) == 26 * 2 + 10 + 4 + 1);
    auto seed = detail::GenerateUuid();
    std::seed_seq seq(seed._data, seed._data + std::extent<decltype(seed._data)>::value);
    std::mt19937 mt(seq);
    std::uniform_int_distribution<> distribution(0, sizeof(Table) - 2);
    for (auto i = 0; i < sizeof(pCodeVerifier->data) / sizeof(pCodeVerifier->data[0]); ++ i)
    {
        pCodeVerifier->data[i] = Table[distribution(mt)];
    }
}

void GenerateCodeChallenge(char* buffer, size_t bufferSize, const CodeVerifier& codeVerifier)
{
    NN_SDK_ASSERT(bufferSize >= 256); // 実際は (128 * 4) / 3 + 1 ほどあれば十分
#if !defined(NN_SDK_BUILD_RELEASE)
    for (auto c : codeVerifier.data)
    {
        NN_SDK_ASSERT(std::isalnum(c) || c == '-' || c == '.' || c == '_' || c == '~');
    }
#endif
    detail::GetEncodedSha256Hash(buffer, bufferSize, codeVerifier.data, sizeof(codeVerifier), util::Base64::Mode_UrlSafe);
}

}}} // ~namespace nn::account::http
