﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_RightsUtil.h>

#include <algorithm>

#include <nn/account/account_ResultPrivate.h>
#include <nn/account/detail/account_Log.h>
#include <nn/es/es_Result.h>
#include <nn/es/es_RightsApi.h>
#include <nn/fs/fs_RightsId.h>

namespace nn { namespace account { namespace detail {
namespace {
Result ListUsersWithRightsRestrictionImpl(
    int* pOutCount, Uid buffer[], int capacity,
    const arp::ApplicationLaunchProperty& launchProperty,
    const user::UserRegistry& userRegistry, const baas::BaasOperator& baasOp) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_SDK_ASSERT(capacity >= UserCountMax);
    NN_UNUSED(capacity);
    NN_UNUSED(launchProperty);

    Uid users[UserCountMax];
    NN_RESULT_DO(userRegistry.ListAllUsers(users, std::extent<decltype(users)>::value));
    auto userCount = static_cast<int>(std::count_if(users, users + std::extent<decltype(users)>::value, [](const auto& u) -> auto { return u; }));

    // アプリを利用できる NA の一覧を取得
    NintendoAccountId naList[UserCountMax];
    int naCount;
    auto hasRestrictedAccount = es::ListAccountRestrictedRightsUser(&naCount, naList, std::extent<decltype(naList)>::value);

    // アカウントに限定された権利の利用者が登録されていない場合は全員 Open 可能
    NN_RESULT_THROW_UNLESS(hasRestrictedAccount, ResultUserQualificationNotLimited());

    // ユーザーの NA が上のリストに含まれていれば OK
    NN_DETAIL_ACCOUNT_TRACE("User with qualifications\n");
    int ct = 0;
    for (auto i = 0; i < userCount; ++ i)
    {
        NintendoAccountId naId;
        auto r = baasOp.GetLinkedNintendoAccountId(&naId, users[i]);
        if (true
            && r.IsSuccess()
            && std::find(naList, naList + naCount, naId) != naList + naCount)
        {
            NN_DETAIL_ACCOUNT_TRACE(" - %016llx_%016llx (naId=%016llx)\n", users[i]._data[0], users[i]._data[1], naId.id);
            buffer[ct ++] = users[i];
        }
    }
    *pOutCount = ct;
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutCount);
    NN_UNUSED(buffer);
    NN_UNUSED(capacity);
    NN_UNUSED(&launchProperty);
    NN_UNUSED(&userRegistry);
    NN_UNUSED(&baasOp);

    NN_RESULT_THROW(ResultUserQualificationNotLimited());
#endif
}
} // ~namespace nn::account::detail::<anonymous>

Result ListUsersWithRightsRestriction(
    int* pOutCount, Uid buffer[], int capacity,
    const arp::ApplicationLaunchProperty& launchProperty,
    const user::UserRegistry& userRegistry, const baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    auto r = ListUsersWithRightsRestrictionImpl(pOutCount, buffer, capacity, launchProperty, userRegistry, baasOp);
    if (!r.IsSuccess())
    {
        NN_RESULT_THROW_UNLESS(fs::ResultTargetNotFound::Includes(r) || es::ResultRightsIdNotFound::Includes(r), r);
        // nspd 起動時など、権利情報を取得できない場合は制約を設けない
        NN_DETAIL_ACCOUNT_TRACE("Failed to load right info: 0x%016llx on %d\n", launchProperty.id.value, static_cast<int>(launchProperty.storageId));
        NN_RESULT_THROW(ResultUserQualificationNotLimited());
    }
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::account::detail
