﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "account_BaasProblemDetails.h"

#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/account_NintendoAccountTypes.h>

#include <nn/nn_Common.h>
#include <nn/image/image_Common.h>

namespace nn { namespace account { namespace baas {

// Image API への画像のアップロード
class ImageUploadAdaptor
    : public BaasCommonAdaptor<16, 32>
{
    NN_DISALLOW_COPY(ImageUploadAdaptor);
private:
    typedef BaasCommonAdaptor<16, 32> Base;

    json::LookupEntry m_UrlEntry;
    json::LookupEntry m_FormatEntry;
    bool m_Adapted;

    char m_Url[UserProfile::ImageUrlBytesForTransfer];
    char m_Format[8];

public:
    typedef Base::JsonPathType JsonPathType;

protected:
    virtual Result AdaptImpl() NN_NOEXCEPT final NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT final NN_OVERRIDE;

public:
    ImageUploadAdaptor(const char* expectedFormat, size_t length) NN_NOEXCEPT;
    void GetUrl(char* urlBuffer, size_t urlBufferSize) const NN_NOEXCEPT;
};

// User リソースの取得
template <int NodeDepthMax, int PathLengthMax>
class UserResourceAdaptorWithBase
    : public BaasCommonAdaptor<NodeDepthMax, PathLengthMax>
{
    NN_DISALLOW_COPY(UserResourceAdaptorWithBase);
private:
    typedef BaasCommonAdaptor<NodeDepthMax, PathLengthMax> Base;

    const detail::AbstractLocalStorage& m_Storage;

    json::LookupEntry1 m_IdEntry;
    json::LookupEntry1 m_NicknameEntry;
    json::LookupEntry1 m_ImageUrlEntry;
    json::LookupEntry1 m_LocalDataEntry;
    json::LookupEntry1 m_NaIdEntry;
    bool m_Adapted;

    NetworkServiceAccountId m_Id;
    char m_Nickname[NicknameBytesMax + 1];
    char m_ImageUrl[UserProfile::ImageUrlBytesForTransfer];
    detail::Uuid m_ExtraDataCacheId;
    NintendoAccountId m_NaId;

public:
    typedef typename Base::JsonPathType JsonPathType;

protected:
    virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE;

public:
    UserResourceAdaptorWithBase(const char* basePath, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
    ~UserResourceAdaptorWithBase() NN_NOEXCEPT;
    NetworkServiceAccountId GetUserId() const NN_NOEXCEPT;
    bool TryGetNintendoAccountId(NintendoAccountId* pOut) const NN_NOEXCEPT;
    Result LoadUserProfile(UserProfile* pOut) const NN_NOEXCEPT;
};
class UserResourceAdaptor
    : public UserResourceAdaptorWithBase<8, 32>
{
    NN_DISALLOW_COPY(UserResourceAdaptor);

public:
    explicit UserResourceAdaptor(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
};

// NA との link 専用
class UserLinkAdaptor
    : public BaasCommonAdaptor<16, 64>
{
    NN_DISALLOW_COPY(UserLinkAdaptor);
private:
    typedef BaasCommonAdaptor<16, 64> Base;

    json::LookupEntry m_IdEntry;
    json::LookupEntry m_NaIdEntry;
    bool m_Adapted;

    NetworkServiceAccountId m_Id;
    NintendoAccountId m_NaId;

public:
    typedef Base::JsonPathType JsonPathType;

protected:
    virtual Result AdaptImpl() NN_NOEXCEPT final NN_OVERRIDE;
    virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT final NN_OVERRIDE;

public:
    UserLinkAdaptor() NN_NOEXCEPT;
    NetworkServiceAccountId GetUserId() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Adapted);
        return m_Id;
    }
    bool TryGetNintendoAccountId(NintendoAccountId* pOut) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Adapted);
        if (!m_NaId)
        {
            return false;
        }
        *pOut = m_NaId;
        return true;
    }
};

}}} // ~namespace nn::account::baas

/* --------------------------------------------------------------------------------------------
    実装
*/

#include "../detail/account_CacheUtil.h"
#include <nn/account/account_ResultPrivate.h>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace baas {

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
        NN_SDK_LOG("    %s : %s%s\n", (e)? "+": "-", (e).path0, (e).path1)

template <int NodeDepthMax, int PathLengthMax>
inline UserResourceAdaptorWithBase<NodeDepthMax, PathLengthMax>::UserResourceAdaptorWithBase(const char* base, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : m_Storage(storage)
    , m_IdEntry(base, ".id")
    , m_NicknameEntry(base, ".nickname")
    , m_ImageUrlEntry(base, ".thumbnailUrl")
    , m_LocalDataEntry(base, ".extras.self.nxAccount")
    , m_NaIdEntry(base, ".links.nintendoAccount.id")
    , m_Adapted(false)
    , m_ExtraDataCacheId(detail::InvalidUuid)
{
}
template <int NodeDepthMax, int PathLengthMax>
inline UserResourceAdaptorWithBase<NodeDepthMax, PathLengthMax>::~UserResourceAdaptorWithBase() NN_NOEXCEPT
{
    if (m_ExtraDataCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_ExtraDataCacheId, m_Storage);
        m_ExtraDataCacheId = detail::InvalidUuid;
    }
}
template <int NodeDepthMax, int PathLengthMax>
inline Result UserResourceAdaptorWithBase<NodeDepthMax, PathLengthMax>::AdaptImpl() NN_NOEXCEPT
{
    if (!(m_IdEntry && m_NicknameEntry && m_ImageUrlEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: BaaS UserResource failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NicknameEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ImageUrlEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_LocalDataEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NaIdEntry);
        NN_RESULT_THROW(ResultBaasDataBroken());
    }
    if (!m_NaIdEntry)
    {
        m_NaId = InvalidNintendoAccountId;
    }
    if (!m_LocalDataEntry)
    {
        m_ExtraDataCacheId = detail::InvalidUuid;
    }

    m_Adapted = true;
    NN_RESULT_SUCCESS;
}
template <int NodeDepthMax, int PathLengthMax>
inline bool UserResourceAdaptorWithBase<NodeDepthMax, PathLengthMax>::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (m_IdEntry.CanAccept(jsonPath))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto id = detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            if (id != 0x00ull)
            {
                m_Id.id = id;
                m_IdEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_NicknameEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_Nickname))
        {
            strncpy(m_Nickname, value, sizeof(m_Nickname));
            NN_SDK_ASSERT(strnlen(m_Nickname, sizeof(m_Nickname)) < sizeof(m_Nickname));
            m_Nickname[valueLength] = '\0';
            m_NicknameEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_ImageUrlEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_ImageUrl))
        {
            strncpy(m_ImageUrl, value, sizeof(m_ImageUrl));
            NN_SDK_ASSERT(strnlen(m_ImageUrl, sizeof(m_ImageUrl)) < sizeof(m_ImageUrl));
            m_ImageUrl[valueLength] = '\0';
            m_ImageUrlEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_LocalDataEntry.CanAccept(jsonPath))
    {
        if (valueLength <= UserProfile::ExtraDataBytesForTransfer)
        {
            detail::Uuid cacheId;
            auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_Storage);
            if (Base::SetIoResult(r).IsSuccess())
            {
                m_ExtraDataCacheId = cacheId;
                m_LocalDataEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_NaIdEntry.CanAccept(jsonPath))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto id = detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            if (id != 0x00ull)
            {
                m_NaId.id = id;
                m_NaIdEntry.MarkAccepted();
            }
        }
        return true;
    }
    return false;
}
template <int NodeDepthMax, int PathLengthMax>
inline Result UserResourceAdaptorWithBase<NodeDepthMax, PathLengthMax>::LoadUserProfile(UserProfile* pOut) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Adapted);

    size_t extraDataSize = 0;
    if (m_LocalDataEntry)
    {
        NN_SDK_ASSERT(m_ExtraDataCacheId);
        NN_RESULT_DO((detail::CacheUtil::LoadCacheFile<UserProfile::ExtraDataBytesForTransfer, ResultBaasDataBroken>(
            &extraDataSize, &pOut->base.extraData, sizeof(pOut->base.extraData), m_ExtraDataCacheId, m_Storage)));
    }
    if (extraDataSize < sizeof(pOut->base.extraData))
    {
        std::memset(&pOut->base.extraData[extraDataSize], 0x00, sizeof(pOut->base.extraData) - extraDataSize);
    }
    std::strncpy(pOut->base.nickname, m_Nickname, sizeof(pOut->base.nickname));
    std::strncpy(pOut->imageUrl, m_ImageUrl, sizeof(pOut->imageUrl));
    NN_RESULT_SUCCESS;
}
template <int NodeDepthMax, int PathLengthMax>
inline NetworkServiceAccountId UserResourceAdaptorWithBase<NodeDepthMax, PathLengthMax>::GetUserId() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Adapted);
    return m_Id;
}
template <int NodeDepthMax, int PathLengthMax>
inline bool UserResourceAdaptorWithBase<NodeDepthMax, PathLengthMax>::TryGetNintendoAccountId(NintendoAccountId* pOut) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Adapted);
    if (!m_NaId)
    {
        return false;
    }
    *pOut = m_NaId;
    return true;
}

#undef NN_ACCOUNT_PRINT_ENTRY_STATE

}}} // ~namespace nn::account::baas
