﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_BaasRegistrationAdaptor.h"

#include "../detail/account_ByteUtil.h"
#include <nn/account/baas/account_ResultForBaas.h>

#include <cstring>

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace baas {

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
        NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

/* --------------------------------------------------------------------------------------------
    DeviceAccountRegistrationAdaptor
*/

DeviceAccountRegistrationAdaptor::DeviceAccountRegistrationAdaptor() NN_NOEXCEPT
    : m_UserIdEntry("$.id")
    , m_LoginIdEntry("$.deviceAccounts[0].id")
    , m_LoginPasswordEntry("$.deviceAccounts[0].password")
    , m_IsAdapted(false)
{
    const BaasCredential InitialCredential = NN_ACCOUNT_BAAS_CREDENTIAL_INITIALIZER;
    m_Credential = InitialCredential;
}
Result DeviceAccountRegistrationAdaptor::AdaptImpl() NN_NOEXCEPT
{
    // 不正な応答 (内容が不足している)
    if (!(m_UserIdEntry && m_LoginIdEntry && m_LoginPasswordEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: BaaS CreateDeviceAccount Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_UserIdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_LoginIdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_LoginPasswordEntry);
        NN_RESULT_THROW(ResultBaasDataBroken());
    }
    m_IsAdapted = true;
    NN_RESULT_SUCCESS;
}
bool DeviceAccountRegistrationAdaptor:: UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    NN_SDK_ASSERT(valueLength >= 0);
    if (m_UserIdEntry.CanAccept(jsonPath))
    {
        if (static_cast<uint32_t>(valueLength) <= sizeof(uint64_t) * 2)
        {
            m_Id.id = detail::ExtractHexadecimal<uint64_t>(
                value, static_cast<size_t>(valueLength));
            m_UserIdEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_LoginIdEntry.CanAccept(jsonPath))
    {
        if (static_cast<uint32_t>(valueLength) <= sizeof(uint64_t) * 2)
        {
            m_Credential.loginId = detail::ExtractHexadecimal<uint64_t>(
                value, static_cast<size_t>(valueLength));
            m_LoginIdEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_LoginPasswordEntry.CanAccept(jsonPath))
    {
        if (static_cast<uint32_t>(valueLength) == sizeof(m_Credential.loginPassword))
        {
            std::strncpy(m_Credential.loginPassword, value, sizeof(m_Credential.loginPassword));
            m_LoginPasswordEntry.MarkAccepted();
        }
        return true;
    }
    return false;
}

#undef NN_ACCOUNT_PRINT_ENTRY_STATE

}}} // ~namespace nn::account::baas
