﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_BaasProfileSynchronizer.h"

#include "../detail/account_CacheUtil.h"
#include "../profile/account_ProfileAdaptor.h"
#include <nn/account/http/account_SimpleDownloader.h>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_ScopeExit.h>

namespace nn {
namespace account {
namespace baas {

namespace {
Result ApplyUserProfile(
    profile::ProfileAdaptor& adaptor,
    const profile::ProfileBase& profile, const profile::UserData& userData, const char* imageUrl, size_t imageUrlSize,
    const BaasUserDriver& baasUserDriver, const detail::AbstractLocalStorage& storage,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT(imageUrl != nullptr);
    NN_SDK_ASSERT(imageUrlSize > 0);

    // 大容量バッファの確保 (関数を抜けるまで解放しない)
    auto largeBuffer = resource.pLargeBufferAllocator->Allocate();

    // 画像のDL
    const void* image = nullptr;
    size_t imageSize = 0;
    if (imageUrl[0] != '\0')
    {
        NN_RESULT_DO(baasUserDriver.DownloadImage(
            &imageSize, largeBuffer.GetAddress(), largeBuffer.GetSize(),
            imageUrl, imageUrlSize,
            resource.curlHandle,
            pCancellable));
        image = largeBuffer.GetAddress();
    }

    // インポート
    auto lock = storage.AcquireWriterLock();
    NN_RESULT_DO(adaptor.Import(profile, userData, image, imageSize));
    NN_RESULT_DO(storage.Commit());
    NN_RESULT_SUCCESS;
}

Result TryImportUserProfile(
    bool* pOutImported, profile::ProfileAdaptor& adaptor,
    const UserProfile& profile,
    const BaasUserDriver& baasUserDriver, const detail::AbstractLocalStorage& storage,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // サーバー側のデータの検証
    profile::ProfileBase remote;
    profile::UserData userData;
    if (!adaptor.Decode(&remote, &userData, profile.base).IsSuccess())
    {
        *pOutImported = false;
        NN_RESULT_SUCCESS;
    }

    // 自身に導入可能かの検証
    profile::ProfileAdaptor::EvaluationResult eval;
    NN_RESULT_DO(adaptor.Evaluate(&eval, remote));

    bool imported = false;
    switch (eval)
    {
    case profile::ProfileAdaptor::EvaluationResult_Stay:
        imported = true; // 更新した場合と等しい
        break;
    case profile::ProfileAdaptor::EvaluationResult_Import:
        NN_RESULT_DO(ApplyUserProfile(
            adaptor, remote, userData, profile.imageUrl, sizeof(profile.imageUrl),
            baasUserDriver, storage, resource, pCancellable));
        imported = true;
        break;
    case profile::ProfileAdaptor::EvaluationResult_Export:
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    *pOutImported = imported;
    NN_RESULT_SUCCESS;
}

Result ExportBaasUserProfileImpl(
    const NetworkServiceAccountId& id, const profile::ProfileAdaptor& adaptor,
    const BaasUserDriver& baasUserDriver, const detail::AbstractLocalStorage& storage,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // ニックネームとユーザーデータのシリアライズ
    UserProfile profile;
    NN_RESULT_DO(adaptor.AcquireProfile(&profile.base));

    // 大容量バッファの確保
    auto largeBuffer = resource.pLargeBufferAllocator->Allocate();

    // プロフィール画像のシリアライズ
    detail::Uuid cacheId = detail::InvalidUuid;
    auto r = adaptor.CreateProfileImageCache(&cacheId, largeBuffer.GetAddress(), largeBuffer.GetSize());
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultUserNotExist::Includes(r), r);
    NN_UTIL_SCOPE_EXIT
    {
        if (cacheId)
        {
            detail::CacheUtil::DeleteCacheFile(cacheId, storage);
        }
    };

    if (cacheId)
    {
        // プロフィール画像のアップロード
        NN_RESULT_DO(baasUserDriver.UploadImage(
            profile.imageUrl, sizeof(profile.imageUrl), id, cacheId,
            resource.curlHandle, resource.buffer.address, resource.buffer.size,
            largeBuffer.GetAddress(), largeBuffer.GetSize(),
            pCancellable));
    }
    else
    {
        // プロフィール画像がなければ thumbnail_url 属性は空
        profile.imageUrl[0] = '\0';
    }

    NN_RESULT_DO(baasUserDriver.UploadProfile(
        id, profile,
        resource.curlHandle, resource.buffer.address, resource.buffer.size,
        pCancellable));

    NN_RESULT_SUCCESS;
}

Result SynchronizeBaasUserProfileImpl(
    profile::ProfileStorage& profileStorage,
    BaasUserResourceCache& resourceCache,
    const Uid& user, const NetworkServiceAccountId& id, const UserProfile& profile,
    const BaasUserDriver& baasUserDriver, const detail::AbstractLocalStorage& storage,
    const ExecutionResource& resource, detail::Cancellable* pCancellable,
    bool importFirst) NN_NOEXCEPT
{
    // プロフィールのインポートを試行
    profile::ProfileAdaptor adaptor(&profileStorage, user, importFirst);
    bool imported;
    NN_RESULT_DO(TryImportUserProfile(&imported, adaptor, profile, baasUserDriver, storage, resource, pCancellable));
    if (!imported)
    {
        // インポートされなければエクスポート
        NN_RESULT_DO(ExportBaasUserProfileImpl(id, adaptor, baasUserDriver, storage, resource, pCancellable));
    }

    // 同期した情報をキャッシュしておく
    profile::ProfileBase base;
    profileStorage.GetProfile(&base, nullptr, user);
    resourceCache.Store(user, base.author, base.timeStamp);

    NN_SDK_LOG("[nn::account] User profile: %s\n", imported ? "Imported" : "Exported");
    NN_RESULT_SUCCESS;
}
} // ~namespace nn::account::baas

Result ExportBaasUserProfile(
    profile::ProfileStorage& profileStorage,
    BaasUserResourceCache& resourceCache,
    const Uid& user, const NetworkServiceAccountId& id,
    const BaasUserDriver& baasUserDriver, const detail::AbstractLocalStorage& storage,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    profile::ProfileAdaptor adaptor(&profileStorage, user);
    NN_RESULT_DO(ExportBaasUserProfileImpl(id, adaptor, baasUserDriver, storage, resource, pCancellable));

    // 同期した情報をキャッシュしておく
    profile::ProfileBase base;
    profileStorage.GetProfile(&base, nullptr, user);
    resourceCache.Store(user, base.author, base.timeStamp);
    NN_RESULT_SUCCESS;
}

Result SynchronizeBaasUserProfile(
    profile::ProfileStorage& profileStorage,
    BaasUserResourceCache& resourceCache,
    const Uid& user, const NetworkServiceAccountId& id,
    const BaasUserDriver& baasUserDriver, const detail::AbstractLocalStorage& storage,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    UserProfile profile;
    NN_RESULT_DO(baasUserDriver.DownloadProfile(
        &profile, id, resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    return SynchronizeBaasUserProfileImpl(profileStorage, resourceCache, user, id, profile, baasUserDriver, storage, resource, pCancellable, false);
}

Result SynchronizeBaasUserProfile(
    profile::ProfileStorage& profileStorage,
    BaasUserResourceCache& resourceCache,
    const Uid& user, const NetworkServiceAccountId& id, const UserProfile& profile,
    const BaasUserDriver& baasUserDriver, const detail::AbstractLocalStorage& storage,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    return SynchronizeBaasUserProfileImpl(profileStorage, resourceCache, user, id, profile, baasUserDriver, storage, resource, pCancellable, true);
}

Result DownloadProfileImage(
    size_t* pOutActualSize, void* image, size_t imageSize,
    const char* url,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    http::SimpleDownloader downloader(resource.curlHandle, pCancellable);
    NN_RESULT_DO(downloader.Initialize(image, imageSize));
    NN_RESULT_DO(downloader.SetUrl(url));
    NN_RESULT_DO(downloader.SetTimeoutSeconds(300)); // プロフィール画像のダウンロードのタイムアウトは長く取る

    size_t actualSize;
    NN_RESULT_DO(downloader.Invoke(&actualSize));
    NN_RESULT_THROW_UNLESS(actualSize <= ProfileImageBytesMax, ResultBaasDataBroken());

    *pOutActualSize = actualSize;
    NN_RESULT_SUCCESS;
}

} // ~namespace nn::account::baas
}
}
