﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_BaasLoginAdaptor.h"

#include <nn/account/baas/account_ResultForBaas.h>
#include "../detail/account_CacheUtil.h"
#include "../detail/account_PathUtil.h"
#include "../detail/account_TimeUtil.h"

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace baas {

#define NN_ACCOUNT_PRINT_ENTRY_STATE(e) \
        NN_SDK_LOG("    %s : %s\n", (e)? "+": "-", (e).path)

/* --------------------------------------------------------------------------------------------
    ClientLoginAdaptor
*/

ClientLoginAdaptor::ClientLoginAdaptor(ClientAccessTokenCache& accessTokenCache) NN_NOEXCEPT
    : m_AccessTokenCache(accessTokenCache)
    , m_ExpirationEntry("$.expiresIn")
    , m_TokenEntry("$.accessToken")
    , m_ExpirationAbs(-1)
    , m_TokenCacheId(detail::InvalidUuid)
{
}
ClientLoginAdaptor::~ClientLoginAdaptor() NN_NOEXCEPT
{
    if (m_TokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_TokenCacheId, m_AccessTokenCache.GetStorageRef());
        m_TokenCacheId = detail::InvalidUuid;
    }
}
Result ClientLoginAdaptor::AdaptImpl() NN_NOEXCEPT
{
    // 不正な応答 (内容が不足している)
    if (!(m_ExpirationEntry && m_TokenEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: BaaS LoginByClient Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ExpirationEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_TokenEntry);
        NN_RESULT_THROW(ResultBaasDataBroken());
    }

    // トークンの保存
    m_AccessTokenCache.Store(m_ExpirationAbs, m_TokenCacheId);
    m_TokenCacheId = detail::InvalidUuid;
    NN_RESULT_SUCCESS;
}
bool ClientLoginAdaptor::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_ExpirationEntry.CanAccept(jsonPath))
    {
        m_ExpirationAbs = value + detail::GetUptimeInSeconds();
        m_ExpirationEntry.MarkAccepted();
        return true;
    }
    return false;
}
bool ClientLoginAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (m_TokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::BaasAccessTokenSizeMax))
        {
            detail::Uuid cacheId;
            auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_AccessTokenCache.GetStorageRef());
            if (SetIoResult(r).IsSuccess())
            {
                m_TokenCacheId = cacheId;
                m_TokenEntry.MarkAccepted();
            }
        }
        return true;
    }
    return false;
}

/* --------------------------------------------------------------------------------------------
    UserLoginAdaptor
*/

UserLoginAdaptor::UserLoginAdaptor(
    UserAccessTokenCache& accessTokenCache,
    UserIdTokenCache& idTokenCache,
    const detail::ApplicationInfo& appInfo) NN_NOEXCEPT
    : m_AppInfo(appInfo)
    , m_AccessTokenCache(accessTokenCache)
    , m_IdTokenCache(idTokenCache)
    , m_IdEntry("$.user.id")
    , m_ExpirationEntry("$.expiresIn")
    , m_AccessTokenEntry("$.accessToken")
    , m_IdTokenEntry("$.idToken")
    , m_NaIdEntry("$.user.links.nintendoAccount.id")
    , m_Adapted(false)
    , m_ExpirationAbs(-1LL)
    , m_AccessTokenCacheId(detail::InvalidUuid)
    , m_IdTokenCacheId(detail::InvalidUuid)
{
}
UserLoginAdaptor::~UserLoginAdaptor() NN_NOEXCEPT
{
    if (m_AccessTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_AccessTokenCacheId, m_AccessTokenCache.GetStorageRef());
        m_AccessTokenCacheId = detail::InvalidUuid;
    }
    if (m_IdTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_IdTokenCacheId, m_IdTokenCache.GetStorageRef());
        m_IdTokenCacheId = detail::InvalidUuid;
    }
}
Result UserLoginAdaptor::AdaptImpl() NN_NOEXCEPT
{
    // 内容が不足している (NintendoAccountId はなくてもよい。)
    if (!(m_IdEntry && m_ExpirationEntry && m_AccessTokenEntry && m_IdTokenEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: BaaS LoginByDeviceAccount Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ExpirationEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_AccessTokenEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdTokenEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NaIdEntry);
        NN_RESULT_THROW(ResultBaasDataBroken());
    }
    if (!m_NaIdEntry)
    {
        m_NaId = InvalidNintendoAccountId;
    }

    // トークンの保存
    m_AccessTokenCache.Store(m_Id, m_ExpirationAbs, m_AccessTokenCacheId);
    m_AccessTokenCacheId = detail::InvalidUuid;
    if (m_AppInfo)
    {
        m_IdTokenCache.Store(m_Id, m_AppInfo, m_ExpirationAbs, m_IdTokenCacheId);
        m_IdTokenCacheId = detail::InvalidUuid;
    }
    m_Adapted = true;
    NN_RESULT_SUCCESS;

}
bool UserLoginAdaptor::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_ExpirationEntry.CanAccept(jsonPath))
    {
        m_ExpirationAbs = value + detail::GetUptimeInSeconds();
        m_ExpirationEntry.MarkAccepted();
        return true;
    }
    return false;
}
bool UserLoginAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (m_IdEntry.CanAccept(jsonPath))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto id = detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            if (id != 0x00ull)
            {
                m_Id.id = id;
                m_IdEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_AccessTokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::BaasAccessTokenSizeMax))
        {
            detail::Uuid cacheId;
            auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_AccessTokenCache.GetStorageRef());
            if (SetIoResult(r).IsSuccess())
            {
                m_AccessTokenCacheId = cacheId;
                m_AccessTokenEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_IdTokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::BaasIdTokenSizeMax))
        {
            if (m_AppInfo)
            {
                detail::Uuid cacheId;
                auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_IdTokenCache.GetStorageRef());
                if (SetIoResult(r).IsSuccess())
                {
                    m_IdTokenCacheId = cacheId;
                    m_IdTokenEntry.MarkAccepted();
                }
            }
            else
            {
                m_IdTokenEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_NaIdEntry.CanAccept(jsonPath))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto id = detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            if (id != 0x00ull)
            {
                m_NaId.id = id;
                m_NaIdEntry.MarkAccepted();
            }
        }
        return true;
    }
    return false;
}

/* --------------------------------------------------------------------------------------------
    UserLoginAdaptorWithFederation
*/

UserFederationLoginAdaptor::UserFederationLoginAdaptor(
    UserAccessTokenCache& accessTokenCache,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : UserResourceAdaptorWithBase<16, 64>("$.user", storage)
    , m_AccessTokenCache(accessTokenCache)
    , m_ExpirationEntry("$.expiresIn")
    , m_AccessTokenEntry("$.accessToken")
    , m_IdTokenEntry("$.idToken")
    , m_Adapted(false)
    , m_ExpirationAbs(-1LL)
    , m_AccessTokenCacheId(detail::InvalidUuid)
{
}
UserFederationLoginAdaptor::~UserFederationLoginAdaptor() NN_NOEXCEPT
{
    if (m_AccessTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_AccessTokenCacheId, m_AccessTokenCache.GetStorageRef());
        m_AccessTokenCacheId = detail::InvalidUuid;
    }
}
Result UserFederationLoginAdaptor::AdaptImpl() NN_NOEXCEPT
{
    NN_RESULT_DO(Base::AdaptImpl());

    if (!(m_ExpirationEntry && m_AccessTokenEntry && m_IdTokenEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: BaaS LoginByDeviceAccount Failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ExpirationEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_AccessTokenEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdTokenEntry);
        NN_RESULT_THROW(ResultBaasDataBroken());
    }

    // トークンの保存
    m_AccessTokenCache.Store(Base::GetUserId(), m_ExpirationAbs, m_AccessTokenCacheId);
    m_AccessTokenCacheId = detail::InvalidUuid;
    m_Adapted = true;
    NN_RESULT_SUCCESS;

}
bool UserFederationLoginAdaptor::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_ExpirationEntry.CanAccept(jsonPath))
    {
        m_ExpirationAbs = value + detail::GetUptimeInSeconds();
        m_ExpirationEntry.MarkAccepted();
        return true;
    }
    return false;
}
bool UserFederationLoginAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (Base::UpdateImpl(jsonPath, value, valueLength))
    {
        return true;
    }
    else if (m_AccessTokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::BaasAccessTokenSizeMax))
        {
            detail::Uuid cacheId;
            auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_AccessTokenCache.GetStorageRef());
            if (SetIoResult(r).IsSuccess())
            {
                m_AccessTokenCacheId = cacheId;
                m_AccessTokenEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_IdTokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::BaasIdTokenSizeMax))
        {
            m_IdTokenEntry.MarkAccepted();
        }
        return true;
    }
    return false;
}

/* --------------------------------------------------------------------------------------------
    UserLoginAdaptorForGuest
*/

UserLoginAdaptorForGuest::
UserLoginAdaptorForGuest(
    baas::GuestUserProfile* pOutProfile,
    UserIdTokenCache& idTokenCache,
    const detail::ApplicationInfo& appInfo) NN_NOEXCEPT
    : m_pOutProfile(pOutProfile)
    , m_AppInfo(appInfo)
    , m_IdTokenCache(idTokenCache)
    , m_IdEntry("$.user.id")
    , m_ExpirationEntry("$.expiresIn")
    , m_IdTokenEntry("$.idToken")
    , m_NaIdEntry("$.user.links.nintendoAccount.id")
    , m_NicknameEntry("$.user.nickname")
    , m_ImageUrlEntry("$.user.thumbnailUrl")
    , m_LocalDataEntry("$.user.extras.self.nxAccount")
    , m_ExpirationAbs(-1LL)
    , m_IdTokenCacheId(detail::InvalidUuid)
{
    NN_SDK_ASSERT_NOT_NULL(m_pOutProfile);
    NN_SDK_ASSERT(m_AppInfo);
}
UserLoginAdaptorForGuest::~UserLoginAdaptorForGuest() NN_NOEXCEPT
{
    if (m_IdTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_IdTokenCacheId, m_IdTokenCache.GetStorageRef());
    }
}
Result UserLoginAdaptorForGuest::AdaptImpl() NN_NOEXCEPT
{
    if (!(m_IdEntry && m_ExpirationEntry && m_IdTokenEntry && m_NaIdEntry && m_NicknameEntry && m_ImageUrlEntry))
    {
        NN_SDK_LOG(
            "[nn::account] -----------------------------------------------\n"
            "  Error: BaaS UserResource failed\n");
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ExpirationEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_IdTokenEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NaIdEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_NicknameEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_ImageUrlEntry);
        NN_ACCOUNT_PRINT_ENTRY_STATE(m_LocalDataEntry);
        NN_RESULT_THROW(ResultBaasDataBroken());
    }

    // これまで一度も PATCH self したことがない。
    NN_RESULT_THROW_UNLESS(m_LocalDataEntry, ResultBaasDataNotExist());

    // ID トークンの保存
    m_IdTokenCache.Store(m_pOutProfile->id, m_AppInfo, m_ExpirationAbs, m_IdTokenCacheId);
    m_IdTokenCacheId = detail::InvalidUuid;
    NN_RESULT_SUCCESS;
}
bool UserLoginAdaptorForGuest::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    if (m_ExpirationEntry.CanAccept(jsonPath))
    {
        m_ExpirationAbs = value + detail::GetUptimeInSeconds();
        m_ExpirationEntry.MarkAccepted();
        return true;
    }
    return false;
}
bool UserLoginAdaptorForGuest::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    if (m_IdEntry.CanAccept(jsonPath))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto id = detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            if (id != 0x00ull)
            {
                m_pOutProfile->id.id = id;
                m_IdEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_IdTokenEntry.CanAccept(jsonPath))
    {
        if (valueLength <= static_cast<int>(detail::BaasIdTokenSizeMax))
        {
            detail::Uuid cacheId;
            auto r = detail::CacheUtil::StoreCacheFile(&cacheId, value, valueLength, m_IdTokenCache.GetStorageRef());
            if (SetIoResult(r).IsSuccess())
            {
                m_IdTokenCacheId = cacheId;
                m_IdTokenEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_NaIdEntry.CanAccept(jsonPath))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto id = detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            if (id != 0x00ull)
            {
                m_pOutProfile->linkedNaId.id = id;
                m_NaIdEntry.MarkAccepted();
            }
        }
        return true;
    }
    else if (m_NicknameEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutProfile->nickname))
        {
            strncpy(m_pOutProfile->nickname, value, sizeof(m_pOutProfile->nickname));
            m_pOutProfile->nickname[valueLength] = '\0';
            m_NicknameEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_ImageUrlEntry.CanAccept(jsonPath))
    {
        if (valueLength < sizeof(m_pOutProfile->imageUrl))
        {
            strncpy(m_pOutProfile->imageUrl, value, sizeof(m_pOutProfile->imageUrl));
            m_pOutProfile->imageUrl[valueLength] = '\0';
            m_ImageUrlEntry.MarkAccepted();
        }
        return true;
    }
    else if (m_LocalDataEntry.CanAccept(jsonPath))
    {
        if (valueLength <= UserProfile::ExtraDataBytesForTransfer)
        {
            m_LocalDataEntry.MarkAccepted();
        }
        return true;
    }
    return false;
}

#undef NN_ACCOUNT_PRINT_ENTRY_STATE

}}} // ~namespace nn::account::baas
