﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Selector.h>
#include <nn/account/account_SelectorForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForSystemServices.h>

#include "detail/account_ShimLibraryUtility.h"
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_TypesForSelectorImpl.h>

#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account {

namespace {

UiSettings GetUiSettings(
    const UserSelectionSettings& settings,
    const UserSelectionSettingsForSystemService& settingsForSystemService,
    bool forSystem,
    const ApplicationId& appId = ApplicationId::GetInvalidId() ) NN_NOEXCEPT
{
    // ユーザー追加ボタンの表示の可否
    bool isPermitted;
    NN_ABORT_UNLESS_RESULT_SUCCESS( IsUserRegistrationRequestPermitted( &isPermitted ) );

    UiSettings ui = {};
    ui.mode = UiMode_SelectUser;
    NN_STATIC_ASSERT( sizeof( ui.select.invalidUidList ) == sizeof( settings.invalidUidList ) );
    std::memcpy( ui.select.invalidUidList, settings.invalidUidList, sizeof( ui.select.invalidUidList ) );
    ui.select.applicationId = appId;
    ui.select.isSkipEnabled = settings.isSkipEnabled;
    ui.select.isNetworkServiceAccountRequired = settings.isNetworkServiceAccountRequired;
    ui.select.forSystem = forSystem;
    ui.select.isUserRegistrationPermitted = isPermitted;
    ui.select.showSkipButton = settings.showSkipButton;
    ui.select.additionalSelect = settings.additionalSelect;
    ui.select.invalidateUnqualified = false; // TODO 情報公開後 Settings の値を入れる

    ui.select.showAddButton = settingsForSystemService.showAddButton;
    ui.select.purpose = settingsForSystemService.purpose;

    return ui;
}

} // ~namespace nn::account::<anonymous>

#define RETURN_IF_MATCH(v) \
    case v: \
        return v

UiMode UiSettings::GetUiMode() const NN_NOEXCEPT
{
    switch( mode )
    {
    RETURN_IF_MATCH( UiMode_SelectUser );
    RETURN_IF_MATCH( UiMode_CreateUser );
    RETURN_IF_MATCH( UiMode_EnsureNsaAvailable );
    RETURN_IF_MATCH( UiMode_EditUserIcon );
    RETURN_IF_MATCH( UiMode_EditUserNickname );
    RETURN_IF_MATCH( UiMode_CreateUserForStarter );
    RETURN_IF_MATCH( UiMode_ApplicationAuthorization );
    RETURN_IF_MATCH( UiMode_IntroduceExternalNsa );
    RETURN_IF_MATCH( UiMode_IntroduceExternalNsaForRegistration );
    RETURN_IF_MATCH( UiMode_LinkNnidWithNa );
    RETURN_IF_MATCH( UiMode_RequireLicense );
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
NintendoAccountStartupDialogType UiSettings::GetNintendoAccountStartupDialogType() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( mode == UiMode_EnsureNsaAvailable );
    switch( ensureNsaAvailable.startupDialogType )
    {
    RETURN_IF_MATCH( NintendoAccountStartupDialogType_LoginAndCreate );
    RETURN_IF_MATCH( NintendoAccountStartupDialogType_Login );
    RETURN_IF_MATCH( NintendoAccountStartupDialogType_Create );
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
LicenseType UiSettings::GetLicenseType() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( mode == UiMode_RequireLicense );
    switch( requireLicense.licenseType )
    {
        RETURN_IF_MATCH( LicenseType_Op2Common );
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

#undef RETURN_IF_MATCH

Result ShowUserSelectorForSystem( Uid* pOut, const UserSelectionSettings& settings, const UserSelectionSettingsForSystemService& settingsForSystemService ) NN_NOEXCEPT
{
    return detail::ShowUserSelectorCommon( pOut, GetUiSettings( settings, settingsForSystemService, true/*forSystem*/ ) );
}

Result ShowUserSelectorForLauncher( Uid* pOut, const UserSelectionSettings& settings, const nn::ApplicationId& appId ) NN_NOEXCEPT
{
    return detail::ShowUserSelectorCommon( pOut, GetUiSettings( settings, DefaultUserSelectionSettingsForSystemService, true/*forSystem*/, appId ) );
}

Result ShowUserSelector( Uid* pOut, const UserSelectionSettings& settings ) NN_NOEXCEPT
{
    return detail::ShowUserSelectorCommon( pOut, GetUiSettings( settings, DefaultUserSelectionSettingsForSystemService, false/*forSystem*/ ) );
}

Result ShowUserCreator() NN_NOEXCEPT
{
    bool isPermitted;
    NN_RESULT_DO(IsUserRegistrationRequestPermitted(&isPermitted));
    NN_ABORT_UNLESS(isPermitted);

    UiSettings settings = {};
    settings.mode = UiMode_CreateUser;

    return detail::StartPselApplet( settings );
}

Result ShowUserIconEditor( const nn::account::Uid& user ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( user );

    UiSettings settings = {};
    settings.mode = UiMode_EditUserIcon;
    settings.editUserIcon.uid = user;

    return detail::StartPselApplet( settings );
}

Result ShowUserNicknameEditor( const nn::account::Uid& user ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( user );

    UiSettings settings = {};
    settings.mode = UiMode_EditUserNickname;
    settings.editUserNickname.uid = user;

    return detail::StartPselApplet( settings );
}

Result ShowUserCreatorForStarter() NN_NOEXCEPT
{
    UiSettings settings = {};
    settings.mode = UiMode_CreateUserForStarter;

    return detail::StartPselApplet( settings );
}

Result ShowNintendoAccountNnidLinker( const Uid& user ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( user );

    UiSettings settings = {};
    settings.mode = UiMode_LinkNnidWithNa;
    settings.linkNnidWithNa.uid = user;

    return detail::StartPselApplet( settings );
}

Result GetUiResultFromContextStack( UiReturnArg* pOut, nn::applet::StorageHandle handle ) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    NN_SDK_REQUIRES_NOT_NULL( pOut );
    NN_RESULT_DO( nn::applet::ReadFromStorage( handle, 0, pOut, sizeof( *pOut ) ) );
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED( pOut );
    NN_UNUSED( handle );
    NN_RESULT_THROW( ResultNotSupported() );
#endif
}

Result ShowUiToIntroduceExternalNetworkServiceAccountForRegistration( const SessionId& sessionId ) NN_NOEXCEPT
{
    UiSettings settings = {};
    settings.mode = UiMode_IntroduceExternalNsaForRegistration;
    std::memcpy( &settings.introduceExternalNsaForRegistration.sessionId, &sessionId, sizeof( settings.introduceExternalNsaForRegistration.sessionId ) );

    return detail::StartPselApplet( settings );
}

Result ShowLicenseRequirementsForNetworkService( const Uid& uid ) NN_NOEXCEPT
{
    auto errorCode = err::ErrorCode::GetInvalidErrorCode();
    return ShowLicenseRequirementsForNetworkService( uid, errorCode );
}

Result ShowLicenseRequirementsForNetworkService( const UserHandle& handle ) NN_NOEXCEPT
{
    Uid uid;
    NN_RESULT_DO( GetUserId( &uid, handle ) );
    auto errorCode = err::ErrorCode::GetInvalidErrorCode();
    return ShowLicenseRequirementsForNetworkService( uid, errorCode );
}

Result ShowLicenseRequirementsForNetworkService( const UserHandle& handle, const err::ErrorCode& errorCode ) NN_NOEXCEPT
{
    Uid uid;
    NN_RESULT_DO( GetUserId( &uid, handle ) );
    return ShowLicenseRequirementsForNetworkService( uid, errorCode );
}

Result ShowLicenseRequirementsForNetworkService( const Uid& uid, const err::ErrorCode& errorCode ) NN_NOEXCEPT
{
    UiSettings settings = {};
    settings.mode = UiMode_RequireLicense;
    settings.requireLicense.uid = uid;
    settings.requireLicense.licenseType = LicenseType_Op2Common;
    settings.requireLicense.errorCode = errorCode;

    return detail::StartPselApplet( settings );
}

bool TrySelectUserImplicitly( Uid* pOut, const ImplicitUserSelectionSettings& arg ) NN_NOEXCEPT
{
    Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::TrySelectUserWithoutInteraction( &uid, arg.isNetworkServiceAccountRequired ));
    if ( !uid )
    {
        return false;
    }
    *pOut = uid;
    return true;
}

}} // ~namespace nn::account
