﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_ApiForAdministrators.h>

#include "detail/account_ApiUtil.h"
#include <nn/account/profile/account_ProfileTypes.h>
#include <nn/account/account_Config.h>
#include <nn/account/account_IAccountService.sfdl.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_IServiceObject.h>

#include "detail/account_ShimLibraryUtility.h"

namespace nn { namespace account {

ProfileEditor& ProfileEditor::Swap(ProfileEditor& rhs) NN_NOEXCEPT
{
    std::swap(m_Uid, rhs.m_Uid);
    std::swap(m_Ptr, rhs.m_Ptr);
    std::swap(m_Base, rhs.m_Base);
    std::swap(m_UserData, rhs.m_UserData);
    return *this;
}
ProfileEditor::ProfileEditor(const Uid& user, profile::IProfileEditor* ptr) NN_NOEXCEPT
    : m_Uid(user)
    , m_Ptr(ptr)
    , m_Base(profile::DefaultProfileBase)
    , m_UserData(profile::DefaultUserData)
{
    NN_SDK_ASSERT(ptr);
}
Result ProfileEditor::GetProfileImageSize(size_t* pOutSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pOutSize);
    uint32_t rawSize;
    NN_RESULT_DO(m_Ptr->GetImageSize(&rawSize));
    *pOutSize = rawSize;
    NN_RESULT_SUCCESS;
}
ProfileEditor::ProfileEditor() NN_NOEXCEPT
    : m_Uid(InvalidUid)
    , m_Ptr(nullptr)
    , m_Base(profile::DefaultProfileBase)
    , m_UserData(profile::DefaultUserData)
{
}
ProfileEditor::ProfileEditor(ProfileEditor&& rhs) NN_NOEXCEPT
    : m_Uid(rhs.m_Uid)
    , m_Ptr(rhs.m_Ptr)
    , m_Base(rhs.m_Base)
    , m_UserData(rhs.m_UserData)
{
    rhs.m_Uid = InvalidUid;
    rhs.m_Ptr = nullptr;
    rhs.m_Base = profile::DefaultProfileBase;
    rhs.m_UserData = profile::DefaultUserData;
}
ProfileEditor::~ProfileEditor() NN_NOEXCEPT
{
    if (m_Ptr != nullptr)
    {
        sf::ReleaseSharedObject(m_Ptr);
    }
}
ProfileEditor& ProfileEditor::operator=(ProfileEditor&& rhs) NN_NOEXCEPT
{
    ProfileEditor tmp(std::move(rhs));
    tmp.Swap(*this);
    return *this;
}

Result ProfileEditor::Refresh() NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ProfileEditor);
    NN_SDK_REQUIRES(m_Uid);
    NN_RESULT_DO(m_Ptr->Get(&m_Base, &m_UserData));
    NN_SDK_ASSERT(m_Base);
    NN_RESULT_SUCCESS;
}
bool ProfileEditor::IsLocalOrigin() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);
    return m_Base.author == m_Uid;
}
uint64_t ProfileEditor::GetTimeStamp() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);
    return m_Base.timeStamp;
}
void ProfileEditor::GetNickname(Nickname* pOut) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);
    NN_SDK_REQUIRES(pOut);

    std::strncpy(pOut->name, m_Base.nickname, sizeof(m_Base.nickname));
    pOut->name[NicknameBytesMax] = '\0';
}
void ProfileEditor::GetUserData(char* outUserData, size_t bytes) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);
    NN_SDK_REQUIRES(outUserData);
    NN_SDK_REQUIRES(bytes == sizeof(m_UserData));
    NN_UNUSED(bytes);
    NN_SDK_ASSERT(m_Base);
    std::memcpy(outUserData, m_UserData.data, sizeof(m_UserData.data));
}
Result ProfileEditor::LoadProfileImage(size_t* pOutActualSize, void* outImage, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);
    NN_SDK_REQUIRES(pOutActualSize);

    if (outImage == nullptr)
    {
        return GetProfileImageSize(pOutActualSize);
    }
    NN_SDK_REQUIRES(bufferSize > 0);

    uint32_t rawSize;
    NN_RESULT_DO(m_Ptr->LoadImage(&rawSize, sf::OutBuffer(reinterpret_cast<char*>(outImage), bufferSize)));
    *pOutActualSize = rawSize;
    NN_RESULT_SUCCESS;
}

void ProfileEditor::SetNickname(const Nickname& in) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);
    NN_SDK_REQUIRES(strnlen(in.name, sizeof(in.name)) <= NicknameBytesMax);
    std::strncpy(m_Base.nickname, in.name, sizeof(m_Base.nickname));
}
void ProfileEditor::SetUserData(const char* userData, size_t bytes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Uid);
    NN_SDK_REQUIRES(userData);
    NN_SDK_REQUIRES(bytes == sizeof(m_UserData));
    NN_UNUSED(bytes);
    std::memcpy(m_UserData.data, userData, sizeof(m_UserData.data));
}
Result ProfileEditor::Flush() NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ProfileEditor);
    NN_SDK_REQUIRES(m_Uid);
    return m_Ptr->Store(m_Base, m_UserData);
}
Result ProfileEditor::FlushWithImage(const void* image, size_t imageSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, ProfileEditor);
    NN_SDK_REQUIRES(m_Uid);
    NN_SDK_REQUIRES(image);
    return m_Ptr->StoreWithImage(m_Base, m_UserData, sf::InBuffer(reinterpret_cast<const char*>(image), imageSize));
}

}} // ~namespace nn::account
