﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/baas/account_Interface.sfdl.h>
#include <nn/account/baas/account_BaasTypes.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_CachedNintendoAccountInfo.h>
#include <nn/account/account_TypesForSelectorImpl.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>

#include "detail/account_ShimLibraryUtility.h"

#include <cstring>
#include <limits>
#include <utility>

#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

namespace nn { namespace account {

namespace {

Result RequestNetworkServiceAccountAvailableImpl(const Uid& user, NintendoAccountStartupDialogType dialogType) NN_NOEXCEPT
{
    // Psel の起動
    UiSettings settings = {};
    settings.mode = UiMode_EnsureNsaAvailable;
    settings.ensureNsaAvailable.uid = user;
    settings.ensureNsaAvailable.startupDialogType = dialogType;
    return detail::StartPselApplet(settings);
}

} // ~namespace nn::account::<anonymous>

/* --------------------------------------------------------------------------------------------
    アプリケーション向け実装
 */
Result EnsureNetworkServiceAccountAvailable(const UserHandle& handle) NN_NOEXCEPT
{
    auto r = reinterpret_cast<baas::IManagerForApplication*>(handle._context)->CheckAvailability();
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkServiceAccountUnavailable::Includes(r), r);
    if (r.IsSuccess())
    {
        NN_RESULT_SUCCESS;
    }
    NN_SDK_ASSERT(ResultNetworkServiceAccountUnavailable::Includes(r));

    Uid user;
    NN_RESULT_DO(GetUserId(&user, handle));
    return RequestNetworkServiceAccountAvailableImpl(user, NintendoAccountStartupDialogType_LoginAndCreate);
}
Result IsNetworkServiceAccountAvailable(bool* pOut, const UserHandle& handle) NN_NOEXCEPT
{
    auto r = reinterpret_cast<baas::IManagerForApplication*>(handle._context)->CheckAvailability();
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkServiceAccountUnavailable::Includes(r), r);
    *pOut = r.IsSuccess();
    NN_RESULT_SUCCESS;
}
Result GetNetworkServiceAccountId(NetworkServiceAccountId* pOutId, const UserHandle& handle) NN_NOEXCEPT
{
    return reinterpret_cast<baas::IManagerForApplication*>(handle._context)->GetAccountId(pOutId);
}
Result EnsureNetworkServiceAccountIdTokenCacheAsync(AsyncContext* pOutContext, const UserHandle& handle) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckInternetAvailability());

    sf::SharedPointer<detail::IAsyncContext> asyncContextPtr;
    NN_RESULT_DO(reinterpret_cast<baas::IManagerForApplication*>(handle._context)->EnsureIdTokenCacheAsync(&asyncContextPtr));

    Uid user;
    NN_RESULT_DO(GetUserId(&user, handle));
    *pOutContext = AsyncContext(asyncContextPtr.Detach(), user);
    NN_RESULT_SUCCESS;
}
Result LoadNetworkServiceAccountIdTokenCache(size_t* pOutActualSize, char* buffer, size_t bufferSize, const UserHandle& handle) NN_NOEXCEPT
{
    uint32_t sizeActual;
    NN_RESULT_DO(reinterpret_cast<baas::IManagerForApplication*>(handle._context)->LoadIdTokenCache(&sizeActual, sf::OutBuffer(buffer, bufferSize)));
    *pOutActualSize = sizeActual;
    NN_RESULT_SUCCESS;
}

Result CreateNintendoAccountAuthorizationRequest(
    NintendoAccountAuthorizationRequestContext* pOutRequest,
    const UserHandle& handle,
    const NintendoAccountAuthorizationRequestParameters& param, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(workBufferSize <= std::numeric_limits<uint32_t>::max());
    NN_RESULT_DO(detail::CheckInternetAvailability());

    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % os::MemoryPageSize == 0);
    NN_SDK_REQUIRES(workBufferSize % os::MemoryPageSize == 0);
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForNintendoAccountAuthorizationRequestContext);
    os::TransferMemory memory(workBuffer, workBufferSize, os::MemoryPermission_ReadWrite);

    Uid user;
    NN_RESULT_DO(GetUserId(&user, handle));

    sf::SharedPointer<nas::IAuthorizationRequest> p;
    NN_RESULT_DO(reinterpret_cast<baas::IManagerForApplication*>(handle._context)->CreateAuthorizationRequest(
        &p, param, sf::NativeHandle(memory.Detach(), true), static_cast<uint32_t>(workBufferSize)));

    NintendoAccountAuthorizationRequestContext context(p.Detach(), user);
    NN_RESULT_DO(context.Invoke());

    *pOutRequest = std::move(context);
    NN_RESULT_SUCCESS;
}

Result LoadCachedNintendoAccountInfo(
    CachedNintendoAccountInfo* pOut, const UserHandle& handle,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForCachedNintendoAccountInfo);

    // アプリ向けには Availability を検査する
    auto ptr = reinterpret_cast<baas::IManagerForApplication*>(handle._context);
    NN_RESULT_DO(ptr->CheckAvailability());

    NintendoAccountId naId;
    NN_UNUSED(naId);
    return ptr->GetNintendoAccountUserResourceCacheForApplication(
        &naId, &pOut->m_Base, sf::OutBuffer(reinterpret_cast<char*>(workBuffer), workBufferSize));
}

/* --------------------------------------------------------------------------------------------
    NetworkServiceAccountManager
 */
NetworkServiceAccountManager::NetworkServiceAccountManager() NN_NOEXCEPT
    : m_User(InvalidUid)
    , m_Ptr(nullptr)
{
}
NetworkServiceAccountManager::NetworkServiceAccountManager(const Uid& user, baas::IManagerForSystemService* ptr) NN_NOEXCEPT
    : m_User(user)
    , m_Ptr(ptr)
{
}
NetworkServiceAccountManager::~NetworkServiceAccountManager() NN_NOEXCEPT
{
    if (m_Ptr != nullptr)
    {
        sf::ReleaseSharedObject(m_Ptr);
    }
}
NetworkServiceAccountManager& NetworkServiceAccountManager::Swap(NetworkServiceAccountManager& rhs) NN_NOEXCEPT
{
    std::swap(m_User, rhs.m_User);
    std::swap(m_Ptr, rhs.m_Ptr);
    return *this;
}
baas::IManagerForSystemService* NetworkServiceAccountManager::GetPtr() const NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    return m_Ptr;
}
Result NetworkServiceAccountManager::EnsureNetworkServiceAccountAvailable(NintendoAccountStartupDialogType dialogType) NN_NOEXCEPT
{
    auto r = CheckNetworkServiceAccountAvailability();
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkServiceAccountUnavailable::Includes(r), r);
    if (r.IsSuccess())
    {
        NN_RESULT_SUCCESS;
    }
    NN_SDK_ASSERT(ResultNetworkServiceAccountUnavailable::Includes(r));

    return RequestNetworkServiceAccountAvailableImpl(m_User, dialogType);
}
Result NetworkServiceAccountManager::IsNetworkServiceAccountAvailable(bool* pOut) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    auto r = m_Ptr->CheckAvailability();
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkServiceAccountUnavailable::Includes(r), r);
    *pOut = r.IsSuccess();
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountManager::CheckNetworkServiceAccountAvailability() NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    return m_Ptr->CheckAvailability();
}
Result NetworkServiceAccountManager::GetNetworkServiceAccountId(NetworkServiceAccountId* pOutId) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    return m_Ptr->GetAccountId(pOutId);
}
Result NetworkServiceAccountManager::GetNintendoAccountId(NintendoAccountId* pOutId) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    NN_SDK_REQUIRES_NOT_NULL(pOutId);
    return m_Ptr->GetNintendoAccountId(pOutId);
}
Result NetworkServiceAccountManager::LoadCachedNintendoAccountInfo(
    CachedNintendoAccountInfoForSystemService* pOut,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES(workBufferSize >= RequiredBufferSizeForCachedNintendoAccountInfo);
    return m_Ptr->GetNintendoAccountUserResourceCache(&pOut->m_Id, &pOut->m_Base, sf::OutBuffer(reinterpret_cast<char*>(workBuffer), workBufferSize));
}

Result NetworkServiceAccountManager::RefreshCachedNintendoAccountInfoAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    sf::SharedPointer<detail::IAsyncContext> asyncContextPtr;
    NN_RESULT_DO(m_Ptr->RefreshNintendoAccountUserResourceCacheAsync(&asyncContextPtr));
    *pOutContext = AsyncContext(asyncContextPtr.Detach(), m_User);
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountManager::RefreshCachedNintendoAccountInfoAsyncIfTimeElapsed(bool* pOutMatched, AsyncContext* pOutContext, TimeSpan span) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    sf::SharedPointer<detail::IAsyncContext> asyncContextPtr;
    NN_SDK_ASSERT(span.GetSeconds() <= std::numeric_limits<uint32_t>::max());
    auto sec = static_cast<uint32_t>(span.GetSeconds());
    NN_RESULT_DO(m_Ptr->RefreshNintendoAccountUserResourceCacheAsyncIfSecondsElapsed(pOutMatched, &asyncContextPtr, sec));
    *pOutContext = (*pOutMatched ? AsyncContext(asyncContextPtr.Detach(), m_User) : AsyncContext());
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountManager::GetCachedNetworkServiceLicenseInfo(CachedNetworkServiceLicenseInfo* pOut) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    int32_t rawLicense;
    time::PosixTime expiration;
    NN_RESULT_DO(m_Ptr->GetNetworkServiceLicenseCache(&rawLicense, &expiration));
    switch (rawLicense)
    {
    case NetworkServiceLicenseKind_None:
    case NetworkServiceLicenseKind_Common:
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    pOut->kind = static_cast<NetworkServiceLicenseKind>(rawLicense);
    pOut->expiration = expiration;
    pOut->license = static_cast<NetworkServiceLicense>(rawLicense);
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountManager::RefreshCachedNetworkServiceLicenseInfoAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    sf::SharedPointer<detail::IAsyncContext> asyncContextPtr;
    NN_RESULT_DO(m_Ptr->RefreshNetworkServiceLicenseCacheAsync(&asyncContextPtr));
    *pOutContext = AsyncContext(asyncContextPtr.Detach(), m_User);
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountManager::RefreshCachedNetworkServiceLicenseInfoAsyncIfTimeElapsed(bool* pOutMatched, AsyncContext* pOutContext, TimeSpan span) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    sf::SharedPointer<detail::IAsyncContext> asyncContextPtr;
    NN_SDK_ASSERT(span.GetSeconds() <= std::numeric_limits<uint32_t>::max());
    auto sec = static_cast<uint32_t>(span.GetSeconds());
    NN_RESULT_DO(m_Ptr->RefreshNetworkServiceLicenseCacheAsyncIfSecondsElapsed(pOutMatched, &asyncContextPtr, sec));
    *pOutContext = (*pOutMatched ? AsyncContext(asyncContextPtr.Detach(), m_User) : AsyncContext());
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountManager::EnsureNetworkServiceAccountIdTokenCacheAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    sf::SharedPointer<detail::IAsyncContext> asyncContextPtr;
    NN_RESULT_DO(m_Ptr->EnsureIdTokenCacheAsync(&asyncContextPtr));
    *pOutContext = AsyncContext(asyncContextPtr.Detach(), m_User);
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountManager::LoadNetworkServiceAccountIdTokenCache(size_t* pOutActualSize, char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);

    uint32_t sizeActual;
    NN_RESULT_DO(m_Ptr->LoadIdTokenCache(
        &sizeActual, sf::OutBuffer(buffer, bufferSize)));
    *pOutActualSize = sizeActual;
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountManager::CreateNintendoAccountAuthorizationRequest(
    NintendoAccountAuthorizationRequestContext* pOutRequest,
    uint64_t clientId, const char* redirectUri, size_t redirectUriSize,
    const NintendoAccountAuthorizationRequestParameters& param, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);
    NN_SDK_ASSERT(workBufferSize <= std::numeric_limits<uint32_t>::max());

    nas::NasClientInfo clientInfo = {clientId, {}};
    NN_SDK_REQUIRES(strnlen(redirectUri, redirectUriSize) < redirectUriSize);
    NN_SDK_REQUIRES(strnlen(redirectUri, redirectUriSize) < sizeof(clientInfo.redirectUri));
    NN_UNUSED(redirectUriSize);
    std::strncpy(clientInfo.redirectUri, redirectUri, sizeof(clientInfo.redirectUri));
    clientInfo.redirectUri[sizeof(clientInfo.redirectUri) - 1] = '\0';

    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % os::MemoryPageSize == 0);
    NN_SDK_REQUIRES(workBufferSize % os::MemoryPageSize == 0);
    os::TransferMemory memory(workBuffer, workBufferSize, os::MemoryPermission_ReadWrite);

    sf::SharedPointer<nas::IAuthorizationRequest> p;
    NN_RESULT_DO(m_Ptr->CreateAuthorizationRequest(&p, clientInfo, param, sf::NativeHandle(memory.Detach(), true), static_cast<uint32_t>(workBufferSize)));

    NintendoAccountAuthorizationRequestContext context(p.Detach(), m_User);
    NN_RESULT_DO(context.Invoke());

    *pOutRequest = std::move(context);
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountManager::DebugCreateNintendoAccountAuthorizationRequest(
    NintendoAccountAuthorizationRequestContext* pOutRequest,
    uint64_t clientId, const char* redirectUri, size_t redirectUriSize,
    const NintendoAccountAuthorizationRequestParameters& param, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);
    NN_SDK_ASSERT(workBufferSize <= std::numeric_limits<uint32_t>::max());

    nas::NasClientInfo clientInfo = {clientId, {}};
    NN_SDK_REQUIRES(strnlen(redirectUri, redirectUriSize) < redirectUriSize);
    NN_SDK_REQUIRES(strnlen(redirectUri, redirectUriSize) < sizeof(clientInfo.redirectUri));
    NN_UNUSED(redirectUriSize);
    std::strncpy(clientInfo.redirectUri, redirectUri, sizeof(clientInfo.redirectUri));
    clientInfo.redirectUri[sizeof(clientInfo.redirectUri) - 1] = '\0';

    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % os::MemoryPageSize == 0);
    NN_SDK_REQUIRES(workBufferSize % os::MemoryPageSize == 0);
    os::TransferMemory memory(workBuffer, workBufferSize, os::MemoryPermission_ReadWrite);

    sf::SharedPointer<nas::IAuthorizationRequest> p;
    NN_RESULT_DO(m_Ptr->CreateAuthorizationRequest(&p, clientInfo, param, sf::NativeHandle(memory.Detach(), true), static_cast<uint32_t>(workBufferSize)));

    NintendoAccountAuthorizationRequestContext context(p.Detach(), m_User);

    *pOutRequest = std::move(context);
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountManager::SetSystemProgramIdentification(const SystemProgramIdentification& identification) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NetworkServiceAccountManager);
    return m_Ptr->SetSystemProgramIdentification(identification, 0x00ull);
}

/* --------------------------------------------------------------------------------------------
    NetworkServiceAccountAdministrator
 */
NetworkServiceAccountAdministrator::NetworkServiceAccountAdministrator() NN_NOEXCEPT
    : NetworkServiceAccountManager(InvalidUid, nullptr)
{
}
NetworkServiceAccountAdministrator::NetworkServiceAccountAdministrator(const Uid& user, baas::IAdministrator* ptr) NN_NOEXCEPT
    : NetworkServiceAccountManager(user, ptr)
{
    // 基底で Release されるので、 Administrator はリリースしない。
}
NetworkServiceAccountAdministrator& NetworkServiceAccountAdministrator::Swap(NetworkServiceAccountAdministrator& rhs) NN_NOEXCEPT
{
    NetworkServiceAccountManager::Swap(rhs);
    return *this;
}
Result NetworkServiceAccountAdministrator::IsNetworkServiceAccountRegistered(bool* pOut) NN_NOEXCEPT
{
    return GetPtr()->IsRegistered(pOut);
}
Result NetworkServiceAccountAdministrator::RegisterAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> ptr;
    NN_RESULT_DO(GetPtr()->RegisterAsync(&ptr));
    *pOutContext = AsyncContext(ptr.Detach(), NetworkServiceAccountManager::GetUid());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::UnregisterAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> ptr;
    NN_RESULT_DO(GetPtr()->UnregisterAsync(&ptr));
    *pOutContext = AsyncContext(ptr.Detach(), NetworkServiceAccountManager::GetUid());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::DeleteRegistrationInfoLocally() NN_NOEXCEPT
{
    return GetPtr()->DeleteRegistrationInfoLocally();
}
Result NetworkServiceAccountAdministrator::SynchronizeProfileAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> ptr;
    NN_RESULT_DO(GetPtr()->SynchronizeProfileAsync(&ptr));
    *pOutContext = AsyncContext(ptr.Detach(), NetworkServiceAccountManager::GetUid());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::SynchronizeProfileAsyncIfTimeElapsed(bool* pOutMatched, AsyncContext* pOutContext, TimeSpan span) NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> ptr;
    NN_SDK_ASSERT(span.GetSeconds() <= std::numeric_limits<uint32_t>::max());
    auto sec = static_cast<uint32_t>(span.GetSeconds());
    NN_RESULT_DO(GetPtr()->SynchronizeProfileAsyncIfSecondsElapsed(pOutMatched, &ptr, sec));
    *pOutContext = (*pOutMatched ? AsyncContext(ptr.Detach(), NetworkServiceAccountManager::GetUid()) : AsyncContext());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::IsLinkedWithNintendoAccount(bool* pOut) NN_NOEXCEPT
{
    return GetPtr()->IsLinkedWithNintendoAccount(pOut);
}
Result NetworkServiceAccountAdministrator::CreateProcedureToLinkWithNintendoAccount(NintendoAccountLinkageProcedure* pOut) NN_NOEXCEPT
{
    sf::SharedPointer<nas::IOAuthProcedureForNintendoAccountLinkage> ptr;
    NN_RESULT_DO(GetPtr()->CreateProcedureToLinkWithNintendoAccount(&ptr));
    NN_SDK_ASSERT(ptr);
    *pOut = NintendoAccountLinkageProcedure(GetUid(), ptr.Detach());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::ResumeProcedureToLinkWithNintendoAccount(NintendoAccountLinkageProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(SessionId) == sizeof(detail::Uuid));

    detail::Uuid innerId;
    std::memcpy(&innerId, &sessionId, sizeof(innerId));

    sf::SharedPointer<nas::IOAuthProcedureForNintendoAccountLinkage> ptr;
    NN_RESULT_DO(GetPtr()->ResumeProcedureToLinkWithNintendoAccount(&ptr, innerId));
    NN_SDK_ASSERT(ptr);
    *pOut = NintendoAccountLinkageProcedure(GetUid(), ptr.Detach());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::CreateProcedureToUpdateLinkageStateOfNintendoAccount(NintendoAccountLinkageStateUpdateProcedure* pOut) NN_NOEXCEPT
{
    sf::SharedPointer<http::IOAuthProcedure> ptr;
    NN_RESULT_DO(GetPtr()->CreateProcedureToUpdateLinkageStateOfNintendoAccount(&ptr));
    NN_SDK_ASSERT(ptr);
    *pOut = NintendoAccountLinkageStateUpdateProcedure(GetUid(), ptr.Detach());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::ResumeProcedureToUpdateLinkageStateOfNintendoAccount(NintendoAccountLinkageStateUpdateProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(SessionId) == sizeof(detail::Uuid));

    detail::Uuid innerId;
    std::memcpy(&innerId, &sessionId, sizeof(innerId));

    sf::SharedPointer<http::IOAuthProcedure> ptr;
    NN_RESULT_DO(GetPtr()->ResumeProcedureToUpdateLinkageStateOfNintendoAccount(&ptr, innerId));
    NN_SDK_ASSERT(ptr);
    *pOut = NintendoAccountLinkageStateUpdateProcedure(GetUid(), ptr.Detach());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::CreateProcedureToLinkNnidWithNintendoAccount(NintendoAccountNnidLinkageProcedure* pOut) NN_NOEXCEPT
{
    sf::SharedPointer<http::IOAuthProcedure> ptr;
    NN_RESULT_DO(GetPtr()->CreateProcedureToLinkNnidWithNintendoAccount(&ptr));
    NN_SDK_ASSERT(ptr);
    *pOut = NintendoAccountNnidLinkageProcedure(GetUid(), ptr.Detach());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::ResumeProcedureToLinkNnidWithNintendoAccount(NintendoAccountNnidLinkageProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(SessionId) == sizeof(detail::Uuid));

    detail::Uuid innerId;
    std::memcpy(&innerId, &sessionId, sizeof(innerId));

    sf::SharedPointer<http::IOAuthProcedure> ptr;
    NN_RESULT_DO(GetPtr()->ResumeProcedureToLinkNnidWithNintendoAccount(&ptr, innerId));
    NN_SDK_ASSERT(ptr);
    *pOut = NintendoAccountNnidLinkageProcedure(GetUid(), ptr.Detach());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::ProxyProcedureToAcquireApplicationAuthorizationForNintendoAccount(NintendoAccountApplicationAuthorizationProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(SessionId) == sizeof(detail::Uuid));

    detail::Uuid innerId;
    std::memcpy(&innerId, &sessionId, sizeof(innerId));

    sf::SharedPointer<http::IOAuthProcedure> ptr;
    NN_RESULT_DO(GetPtr()->ProxyProcedureToAcquireApplicationAuthorizationForNintendoAccount(&ptr, innerId));
    NN_SDK_ASSERT(ptr);
    *pOut = NintendoAccountApplicationAuthorizationProcedure(GetUid(), ptr.Detach());
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountAdministrator::GetRequiredLicenseCache(NetworkServiceLicense* pOut, const ApplicationId& applicationId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, ApplicationId::GetInvalidId());

    uint32_t req;
    NN_RESULT_DO(GetPtr()->GetServiceEntryRequirementCache(&req, applicationId));
    switch (req)
    {
    case baas::BaasUserServiceEntryRequirement_None:
        *pOut = NetworkServiceLicense_None;
        break;
    case baas::BaasUserServiceEntryRequirement_Op2CommonLicense:
        *pOut = NetworkServiceLicense_Common;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountAdministrator::InvalidateRequiredLicenseCache(const ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, ApplicationId::GetInvalidId());

    return GetPtr()->InvalidateServiceEntryRequirementCache(applicationId);
}

Result NetworkServiceAccountAdministrator::InvalidateNetworkServiceAccountIdTokenCache(const ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, ApplicationId::GetInvalidId());

    return GetPtr()->InvalidateTokenCache(applicationId);
}

Result NetworkServiceAccountAdministrator::TryRecoverNintendoAccountUserStateAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> ptr;
    NN_RESULT_DO(GetPtr()->TryRecoverNintendoAccountUserStateAsync(&ptr));
    *pOutContext = AsyncContext(ptr.Detach(), NetworkServiceAccountManager::GetUid());
    NN_RESULT_SUCCESS;
}

Result NetworkServiceAccountAdministrator::DebugSetNetworkServiceAccountAvailabilityError(Result expect) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(expect.IsFailure(), ResultNotSupported());

#define NN_ACCOUNT_SET_AVAILABILITY_ERROR(pDst, name) \
    NN_RESULT_CATCH(Result ## name) \
    { \
        handled = true; \
        *(pDst) = detail::NsaAvailabilityError_ ## name; \
    }

    bool handled = false;
    NN_UNUSED(handled);
    detail::NsaAvailabilityError error = detail::NsaAvailabilityError_NetworkServiceAccountCredentialBroken;
    NN_RESULT_TRY(expect)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NetworkServiceAccountCredentialBroken)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NetworkServiceAccountUnmanaged)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NetworkServiceAccountBanned)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NintendoAccountLinkageBroken)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NintendoAccountStateOtherButInteractionRequired)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NintendoAccountStateDeleted)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NintendoAccountStateBanned)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NintendoAccountStateSuspended)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NintendoAccountStateWithdrawn)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NintendoAccountStateTermsAgreementRequired)
    NN_ACCOUNT_SET_AVAILABILITY_ERROR(&error, NintendoAccountStateReauthorizationRequired)
    NN_RESULT_CATCH_ALL
    {
        NN_RESULT_THROW(ResultNotSupported());
    }
    NN_RESULT_END_TRY;
    NN_SDK_ASSERT(handled);

#undef NN_ACCOUNT_SET_AVAILABILITY_ERROR

    return GetPtr()->DebugSetAvailabilityErrorDetail(error);
}
Result NetworkServiceAccountAdministrator::DebugUnlinkNintendoAccountAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> ptr;
    NN_RESULT_DO(GetPtr()->DebugUnlinkNintendoAccountAsync(&ptr));
    NN_RESULT_THROW_UNLESS(ptr, ResultOutOfSessionObject());
    *pOutContext = AsyncContext(ptr.Detach(), NetworkServiceAccountManager::GetUid());
    NN_RESULT_SUCCESS;
}
Result NetworkServiceAccountAdministrator::UploadProfileAsync(AsyncContext* pOutContext) NN_NOEXCEPT
{
    sf::SharedPointer<detail::IAsyncContext> ptr;
    NN_RESULT_DO(GetPtr()->UploadProfileAsync(&ptr));
    NN_RESULT_THROW_UNLESS(ptr, ResultOutOfSessionObject());
    *pOutContext = AsyncContext(ptr.Detach(), NetworkServiceAccountManager::GetUid());
    NN_RESULT_SUCCESS;
}

}} // ~namespace nn::account
