﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnd/tmp451/tmp451.h>

#include "tmp451_BusConfig.h"
#include "tmp451_Debug.h"
#include "tmp451_InternalFunction.h"
#include "tmp451_SettingsInterrupt.h"
#include "tmp451_Specification.h"

namespace nnd {
namespace tmp451 {
namespace detail {

namespace{
bool g_IsInterruptBoundPin1 = false;   // Pin1 に対する割り込み
bool g_IsInterruptBoundPin2 = false;   // Pin2 に対する割り込み
}

nn::Result GetPin1DetectionSettings(Temperature* pOutRemoteLimit, Temperature* pOutLocalLimit, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    TemperatureByteData data;

    // 現在のコンフィグ値からレンジを確認
    MeasurementRange range;
    NN_RESULT_DO(GetMeasurementRange(&range));

    // Remote, THERM, 上限, 整数部
    NN_RESULT_DO(ReadRegister(&data.integerByte, bus, RegRead::RemoteThermLimit));
    data.decimalByte = 0;    // 小数部は無いので 0
    *pOutRemoteLimit = ConvertByteData2Temperature(data, range);

    // Local, THERM, 上限, 整数部
    NN_RESULT_DO(ReadRegister(&data.integerByte, bus, RegRead::LocalThermLimit));
    data.decimalByte = 0;    // 小数部は無いので 0
    *pOutLocalLimit = ConvertByteData2Temperature(data, range);

    NN_RESULT_SUCCESS;
}

nn::Result SetPin1DetectionSettings(Temperature remoteLimit, Temperature localLimit, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    TemperatureByteData tempByte;

    // 現在のコンフィグ値からレンジを確認
    MeasurementRange range;
    NN_RESULT_DO(GetMeasurementRange(&range));

    uint8_t cmdSet[2] = { 0, 0 };

    // Remote, THERM, 上限, 整数部
    tempByte = ConvertTemperature2ByteData(remoteLimit, range);
    cmdSet[0] = static_cast<uint8_t>(RegWrite::RemoteThermLimit);
    cmdSet[1] = static_cast<uint8_t>(tempByte.integerByte);
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    // Remote, THERM, 上限, 整数部
    tempByte = ConvertTemperature2ByteData(localLimit, range);
    cmdSet[0] = static_cast<uint8_t>(RegWrite::LocalThermLimit);
    cmdSet[1] = static_cast<uint8_t>(tempByte.integerByte);
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    NN_RESULT_SUCCESS;
}

nn::Result GetPin2DetectionSettings(Temperature* pOutRemoteLowLimit, Temperature* pOutRemoteHighLimit,
                                    Temperature* pOutLocalLowLimit, Temperature* pOutLocalHighLimit,
                                    DetectionMode* pOutDetection, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    TemperatureByteData data;

    // 現在のコンフィグ値を確認
    ConfigReg config;
    NN_RESULT_DO(ReadConfig(&config, bus));

    // モード確認
    *pOutDetection = config.pin2Mode;

    //---- リモート
    NN_RESULT_DO(ReadRegister(&data.integerByte, bus, RegRead::RemoteHighlimitInt));// 上限：整数部
    NN_RESULT_DO(ReadRegister(&data.decimalByte, bus, RegRead::RemoteHighLimitDec));// 上限：小数部
    *pOutRemoteHighLimit = ConvertByteData2Temperature(data, config.range);


    NN_RESULT_DO(ReadRegister(&data.integerByte, bus, RegRead::RemoteLowlimitInt));// 下限：整数部
    NN_RESULT_DO(ReadRegister(&data.decimalByte, bus, RegRead::RemoteLowLimitDec));// 下限：小数部
    *pOutRemoteLowLimit = ConvertByteData2Temperature(data, config.range);

    //---- ローカル
    // ローカルに対して小数点以下の限度値設定は無い。
    NN_RESULT_DO(ReadRegister(&data.integerByte, bus, RegRead::LocalHighLimit));// 上限：整数部
    data.decimalByte = 0;
    *pOutLocalHighLimit = ConvertByteData2Temperature(data, config.range);

    NN_RESULT_DO(ReadRegister(&data.integerByte, bus, RegRead::LocalLowLimit));// 下限：整数部
    data.decimalByte = 0;
    *pOutLocalLowLimit = ConvertByteData2Temperature(data, config.range);

    NN_RESULT_SUCCESS;
}

nn::Result SetPin2DetectionSettings(Temperature remoteLowLimit, Temperature remoteHighLimit,
                                    Temperature locaLowlLimit, Temperature localHighLimit,
                                    DetectionMode mode, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    TemperatureByteData tempByte;

    // 現在のコンフィグ値からレンジを確認
    MeasurementRange range;
    NN_RESULT_DO(GetMeasurementRange(&range));

    uint8_t cmdSet[2] = { 0, 0 };

    //---- ローカル
    // ローカルに小数部は無い
    // 上限
    tempByte = ConvertTemperature2ByteData(localHighLimit, range);
    cmdSet[0] = static_cast<uint8_t>(RegWrite::LocalHighLimit);
    cmdSet[1] = static_cast<uint8_t>(tempByte.integerByte);
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    // 下限
    tempByte = ConvertTemperature2ByteData(locaLowlLimit, range);
    cmdSet[0] = static_cast<uint8_t>(RegWrite::LocalLowLimit);
    cmdSet[1] = static_cast<uint8_t>(tempByte.integerByte);
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    //---- リモート
    // 上限
    tempByte = ConvertTemperature2ByteData(remoteHighLimit, range);
    cmdSet[0] = static_cast<uint8_t>(RegWrite::RemoteHighlimitInt);
    cmdSet[1] = static_cast<uint8_t>(tempByte.integerByte);
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    cmdSet[0] = static_cast<uint8_t>(RegWrite::RemoteHighLimitDec);
    cmdSet[1] = static_cast<uint8_t>(tempByte.decimalByte);
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    // 下限
    tempByte = ConvertTemperature2ByteData(remoteLowLimit, range);
    cmdSet[0] = static_cast<uint8_t>(RegWrite::RemoteLowlimitInt);
    cmdSet[1] = static_cast<uint8_t>(tempByte.integerByte);
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    cmdSet[0] = static_cast<uint8_t>(RegWrite::RemoteLowLimitDec);
    cmdSet[1] = static_cast<uint8_t>(tempByte.decimalByte);
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));


    //---- モード
    // 現在のコンフィグ値を確認
    uint8_t data;
    NN_RESULT_DO(ReadRegister(&data, bus, RegRead::Configuration));

    // Read
    switch (mode)
    {
    case DetectionMode::Off:
        {
            // 7bit を 1
            data = data | (1 << 7);
            // 5bit を 0
            data = data & ~(1 << 5);
        }
        break;
    case DetectionMode::Therm:
        {
            // 5bit を 1
            data = data | (1 << 5);
        }
        break;
    case DetectionMode::Alert:
        {
            // 7bit を 0
            data = data & ~(1 << 7);
            // 5bit を 0
            data = data & ~(1 << 5);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    // config レジスタに書き込み
    cmdSet[0] = static_cast<uint8_t>(RegWrite::Configuration);
    cmdSet[1] = data;
    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));

    NN_RESULT_SUCCESS;
}

nn::Result BindInterrupt(nn::os::SystemEventType* pEvent, SignalPin pin, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    switch (pin)
    {
    case SignalPin::Pin1:
        {
            if (!g_IsInterruptBoundPin1)
            {
                NN_RESULT_DO(BindInterruptPin1(pEvent, bus));
                g_IsInterruptBoundPin1 = true;
                NN_RESULT_SUCCESS;
            }
            else
            {
                return ResultAlreadyBound();
            }
        }
        break;
    case SignalPin::Pin2:
        {
            if (!g_IsInterruptBoundPin2)
            {
                NN_RESULT_DO(BindInterruptPin2(pEvent, bus));
                g_IsInterruptBoundPin2 = true;
                NN_RESULT_SUCCESS;
            }
            else
            {
                return ResultAlreadyBound();
            }
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

nn::Result UnbindInterrupt(SignalPin pin, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    switch (pin)
    {
    case SignalPin::Pin1:
        {
            if (g_IsInterruptBoundPin1)
            {
                NN_RESULT_DO(UnbindInterruptPin1(bus));
                g_IsInterruptBoundPin1 = false;
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_RESULT_SUCCESS;
            }
        }
        break;
    case SignalPin::Pin2:
        {
            if (g_IsInterruptBoundPin2)
            {
                NN_RESULT_DO(UnbindInterruptPin2(bus));
                g_IsInterruptBoundPin2 = false;
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_RESULT_SUCCESS;
            }
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void SetInterruptEnable(bool enable, SignalPin pin, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    switch (pin)
    {
    case SignalPin::Pin1:
        {
            SetInterruptPin1Enable(enable, bus);
        }
        break;
    case SignalPin::Pin2:
        {
            SetInterruptPin2Enable(enable, bus);
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

nn::Result GetThermHysterisis(uint8_t* pOutHysterisis, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    NN_RESULT_DO(ReadRegister(pOutHysterisis, bus, RegRead::ThermHysterisis));
    NN_RESULT_SUCCESS;
}

nn::Result SetThermHysterisis(uint8_t hysterisis, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::ConversionRate),
        static_cast<uint8_t>(hysterisis),
    };

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));
    NN_RESULT_SUCCESS;
}

nn::Result GetConsecutiveAlert(AlertConsecutive* pOutConsecutive, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t data;
    NN_RESULT_DO(ReadRegister(&data, bus, RegRead::ConsecutiveAlert));

    switch ((data >> 1) & 0x07)
    {
    case 0x00:
        {
            *pOutConsecutive = AlertConsecutive::Consecutive1;
        }
        break;
    case 0x01:
        {
            *pOutConsecutive = AlertConsecutive::Consecutive2;
        }
        break;
    case 0x03:
        {
            *pOutConsecutive = AlertConsecutive::Consecutive3;
        }
        break;
    case 0x07:
        {
            *pOutConsecutive = AlertConsecutive::Consecutive4;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

nn::Result SetConsecutiveAlert(AlertConsecutive consecutive, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t data = 0;

    switch (consecutive)
    {
    case AlertConsecutive::Consecutive1:
        {
            data = 0x00;
        }
        break;
    case AlertConsecutive::Consecutive2:
        {
            data = 0x01;
        }
        break;
    case AlertConsecutive::Consecutive3:
        {
            data = 0x03;
        }
        break;
    case AlertConsecutive::Consecutive4:
        {
            data = 0x07;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::ConversionRate),
        data,
    };

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));
    NN_RESULT_SUCCESS;

}

} // detail
} // tmp451
} // nnd
