﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/gpio/gpio.h>
#include <nnd/tmp451/tmp451.h>

#include "tmp451_Debug.h"
#include "tmp451_InternalFunction.h"
#include "tmp451_SettingsInterrupt.h"

namespace nnd {
namespace tmp451 {
namespace detail {

nn::Result BindInterruptPin1(nn::os::SystemEventType* pEvent, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // NX で Pin1 は PMIC に接続されているためシステムが検知できない
    NN_ABORT("Interrupt of Pin1 is not supported on NX.\n");
}

nn::Result BindInterruptPin2(nn::os::SystemEventType* pEvent, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // イベントを紐付け
    NN_RESULT_DO(::nn::gpio::BindInterrupt(pEvent, &bus.gpioSessionInterruptPin2));

    // 割り込み有効化
    SetInterruptPin2Enable(true, bus);

    NN_RESULT_SUCCESS;
}

// Pin1 に紐付いたシステムイベントを解除する
nn::Result UnbindInterruptPin1(BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // NX で Pin1 は PMIC に接続されているためシステムが検知できない
    NN_ABORT("Interrupt of Pin1 is not supported on NX.\n");
}

// Pin2 に紐付いたシステムイベントを解除する
nn::Result UnbindInterruptPin2(BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 割り込み無効化
    SetInterruptPin2Enable(false, bus);

    // 紐付けを解除
    ::nn::gpio::UnbindInterrupt(&bus.gpioSessionInterruptPin2);

    NN_RESULT_SUCCESS;
}

// Pin1 に紐付いた割り込みの許可状態を設定する
void SetInterruptPin1Enable(bool enable, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // NX で Pin1 は PMIC に接続されているためシステムが検知できない
    NN_ABORT("Interrupt of Pin1 is not supported on NX.\n");
}

// Pin2 に紐付いた割り込みの許可状態を設定する
void SetInterruptPin2Enable(bool enable, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 割り込みをクリアする
    ::nn::gpio::ClearInterruptStatus(&bus.gpioSessionInterruptPin2);

    // 割り込み設定
    ::nn::gpio::SetInterruptEnable(&bus.gpioSessionInterruptPin2, enable);
}

} // detail
} // tmp451
} // nnd
