﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/i2c/i2c.h>
#include <nnd/tmp451/tmp451.h>

#include "tmp451_BusConfig.h"
#include "tmp451_Debug.h"
#include "tmp451_InternalFunction.h"
#include "tmp451_Specification.h"

namespace nnd {
namespace tmp451 {
namespace detail {

namespace{
const uint8_t ExtendedRangeOffset = 0x40;   // Extended モード有効時のオフセット 64 (0x40)
}

nn::Result ReadRegister(uint8_t* config, BusSessions bus, RegRead cmd) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 現在の値を取得
    uint8_t cmdGet = static_cast<uint8_t>(cmd);
    uint8_t commandList[nn::i2c::CommandListLengthCountMax];

    nn::i2c::CommandListFormatter  commandListFormatter(commandList, sizeof(commandList));
    std::memset(commandList, 0, sizeof(commandList));

    commandListFormatter.EnqueueSendCommand( nnd::tmp451::detail::I2cTransStart, &cmdGet, sizeof(cmdGet) );
    commandListFormatter.EnqueueReceiveCommand( nnd::tmp451::detail::I2cTransStartStop, nnd::tmp451::detail::RegSize );

    NN_RESULT_DO(nn::i2c::ExecuteCommandList(config, nnd::tmp451::detail::RegSize, bus.i2cSession, commandList, commandListFormatter.GetCurrentLength()));
    NN_RESULT_SUCCESS;
}

nn::Result ReadConfig(ConfigReg* config, BusSessions bus) NN_NOEXCEPT
{
    uint8_t data;
    NN_RESULT_DO(ReadRegister(&data, bus, RegRead::Configuration));

    // ALERT/THERM2
    if ((data >> 5) & 0x01)
    {
        config->pin2Mode = DetectionMode::Therm;
    }
    else
    {
        // MASK1
        if ((data >> 7) & 0x01)
        {
            config->pin2Mode = DetectionMode::Off;
        }
        else
        {
            config->pin2Mode = DetectionMode::Alert;
        }
    }

    // SD
    if ((data >> 6) & 0x01)
    {
        config->isShutdown = true;
    }
    else
    {
        config->isShutdown = false;
    }

    // RANGE
    if ((data >> 2) & 0x01)
    {
        config->range = MeasurementRange::Extended;
    }
    else
    {
        config->range = MeasurementRange::Standard;
    }

    NN_RESULT_SUCCESS;
}

// 計測モードに応じて、温度値をレジスタ書き込み値に変換
TemperatureByteData ConvertTemperature2ByteData(Temperature temperature, MeasurementRange range) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 整数部のオフセット
    int offset = 0;
    if(range == MeasurementRange::Extended)
    {
        offset = ExtendedRangeOffset;
    }
    // TODO: Standard の時のクリッピング

    return TemperatureByteData {
        static_cast<uint8_t>(temperature.integer + offset),
        static_cast<uint8_t>(((temperature.decimal / DecimalUnit) << 4) & 0xf0)
    };
}

// 計測モードに応じて、レジスタの値から温度値に変換
Temperature ConvertByteData2Temperature(TemperatureByteData regData, MeasurementRange range) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 整数部のオフセット
    int offset = 0;
    if(range == MeasurementRange::Extended)
    {
        offset = static_cast<int>(ExtendedRangeOffset);
    }
    // TODO: Standard の時のクリッピング

    return Temperature {
        regData.integerByte - offset,
        ((regData.decimalByte & 0xf0) >> 4) * DecimalUnit
    };
}

nn::Result ResetDevice(BusSessions sessions) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    //  I2Cのジェネラルコールでのリセットコマンドが用意されている(0x00 06)が、
    //  他のデバイスへの影響を考慮して使用しない。
    //  代わりに POR での初期値をセットする。
    uint8_t PorValues[][2] = {
            {static_cast<uint8_t>(RegWrite::Configuration),         0x00},
            {static_cast<uint8_t>(RegWrite::ConversionRate),        0x08},
            {static_cast<uint8_t>(RegWrite::LocalHighLimit),        0x55},
            {static_cast<uint8_t>(RegWrite::LocalLowLimit),         0x00},
            {static_cast<uint8_t>(RegWrite::RemoteHighlimitInt),    0x55},
            {static_cast<uint8_t>(RegWrite::RemoteLowlimitInt),     0x00},
            {static_cast<uint8_t>(RegWrite::RemoteOffsetInt),       0x00},
            {static_cast<uint8_t>(RegWrite::RemoteOffsetDec),       0x00},
            {static_cast<uint8_t>(RegWrite::RemoteHighLimitDec),    0x00},
            {static_cast<uint8_t>(RegWrite::RemoteLowLimitDec),     0x00},
            {static_cast<uint8_t>(RegWrite::RemoteThermLimit),      0x6C},
            {static_cast<uint8_t>(RegWrite::LocalThermLimit),       0x55},
            {static_cast<uint8_t>(RegWrite::ThermHysterisis),       0x0A},
            {static_cast<uint8_t>(RegWrite::ConsecutiveAlert),      0x01},
            {static_cast<uint8_t>(RegWrite::EtaFactorCorrection),   0x00},
            {static_cast<uint8_t>(RegWrite::DigitalFilterControl),  0x00},
    };
    for(size_t i=0 ; i < sizeof(PorValues) / sizeof(PorValues[0]); i++)
    {
        NN_RESULT_DO(nn::i2c::Send(sessions.i2cSession, &PorValues[i][0], sizeof(PorValues[0]) / sizeof(PorValues[0][0]), detail::I2cTransStartStop));
    }
    NN_RESULT_SUCCESS;
}


} // detail
} // tmp451
} // nnd
