﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new> // for placement new

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_TimeSpan.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_TypedStorage.h>

#include <nnd/lsm6ds3/lsm6ds3.h>

#include "detail/lsm6ds3_Driver.h"

namespace {

nn::util::TypedStorage< nnd::lsm6ds3::detail::Driver, sizeof(nnd::lsm6ds3::detail::Driver), sizeof(void*)>  g_Driver;

nnd::lsm6ds3::detail::Driver& GetDriver() NN_NOEXCEPT
{
    return reinterpret_cast<nnd::lsm6ds3::detail::Driver&>(g_Driver);
}

}

namespace nnd {
namespace lsm6ds3 {

nn::Result Initialize() NN_NOEXCEPT
{
    new(&g_Driver) detail::Driver; // placement new で初期化
    return GetDriver().Initialize(NN_SYSTEM_THREAD_PRIORITY(spi, InterruptHandler),
                                  ::nn::os::GetCurrentCoreNumber());
}

nn::Result Initialize(int priority, int idealCoreNumber) NN_NOEXCEPT
{
    new(&g_Driver) detail::Driver; // placement new で初期化
    return GetDriver().Initialize(priority, idealCoreNumber);
}

void Finalize() NN_NOEXCEPT
{
    GetDriver().Finalize();
}

void ResetDevice() NN_NOEXCEPT
{
    GetDriver().ResetDevice();
}

void GetState(SixAxisSensorCountState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    GetDriver().GetState(pOutState);
}

int GetStatesForPeriodicReceiveMode(SixAxisSensorCountState* pOutState, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    return GetDriver().GetStatesForPeriodicReceiveMode(pOutState, count);
}

void StartPeriodicReceiveMode(nn::TimeSpan interval, char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT
{
    GetDriver().StartPeriodicReceiveMode(interval, receiveBuffer, receiveBufferLength);
}

void StopPeriodicReceiveMode() NN_NOEXCEPT
{
    GetDriver().StopPeriodicReceiveMode();
}

void GetNoiseReductionFilter(bool* pOutEnable) NN_NOEXCEPT
{
    GetDriver().GetNoiseReductionFilter(pOutEnable);
}

void SetNoiseReductionFilter(bool enable) NN_NOEXCEPT
{
    GetDriver().SetNoiseReductionFilter(enable);
}

// Acceleration
void StartAccelerometer() NN_NOEXCEPT
{
    GetDriver().StartAccelerometer();
}

void StopAccelerometer() NN_NOEXCEPT
{
    GetDriver().StopAccelerometer();
}

void GetAccelerometerFsr(AccelerometerFsr* pOutAccelerometerFsr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAccelerometerFsr);
    GetDriver().GetAccelerometerFsr(pOutAccelerometerFsr);
}

void SetAccelerometerFsr(AccelerometerFsr accelerometerFsr) NN_NOEXCEPT
{
    GetDriver().SetAccelerometerFsr(accelerometerFsr);
}

void GetAccelerometerOdr(AccelerometerOdr* pOutAccelerometerOdr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAccelerometerOdr);
    GetDriver().GetAccelerometerOdr(pOutAccelerometerOdr);
}

void SetAccelerometerOdr(AccelerometerOdr accelerometerOdr) NN_NOEXCEPT
{
    GetDriver().SetAccelerometerOdr(accelerometerOdr);
}

// Gyroscope
void StartGyroscope() NN_NOEXCEPT
{
    GetDriver().StartGyroscope();
}

void StopGyroscope() NN_NOEXCEPT
{
    GetDriver().StopGyroscope();
}

void GetGyroscopeFsr(GyroscopeFsr* pOutGyroscopeFsr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGyroscopeFsr);
    GetDriver().GetGyroscopeFsr(pOutGyroscopeFsr);
}

void SetGyroscopeFsr(GyroscopeFsr gyroscopeFsr) NN_NOEXCEPT
{
    GetDriver().SetGyroscopeFsr(gyroscopeFsr);
}

void GetGyroscopeOdr(GyroscopeOdr* pOutGyroscopeOdr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGyroscopeOdr);
    GetDriver().GetGyroscopeOdr(pOutGyroscopeOdr);
}

void SetGyroscopeOdr(GyroscopeOdr gyroscopeOdr) NN_NOEXCEPT
{
    GetDriver().SetGyroscopeOdr(gyroscopeOdr);
}

} // lsm6ds3
} // nnd
