﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @file
 * @brief   各種ユーティリティ関数
 */

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>

#include <nn/result/result_HandlingUtility.h>

#include "lsm6ds3_Debug.h"
#include "lsm6ds3_Driver.h"
#include "lsm6ds3_Register.h"
#include "lsm6ds3_Util.h"

namespace {

struct AccelerometerFsrSetting
{
    nnd::lsm6ds3::AccelerometerFsr       fsr;
    nnd::lsm6ds3::detail::AccFsrRegValue reg;
};

struct GyroscopeFsrSetting
{
    nnd::lsm6ds3::GyroscopeFsr            fsr;
    nnd::lsm6ds3::detail::GyroFsrRegValue reg;
};

struct AccelerometerOdrSetting
{
    nnd::lsm6ds3::AccelerometerOdr       odr;
    nnd::lsm6ds3::detail::AccOdrRegValue reg;
};

struct GyroscopeOdrSetting
{
    nnd::lsm6ds3::GyroscopeOdr            odr;
    nnd::lsm6ds3::detail::GyroOdrRegValue reg;
};

const AccelerometerFsrSetting AccelerometerFsrSettings[] =
{
    { nnd::lsm6ds3::AccelerometerFsr_2G  , nnd::lsm6ds3::detail::AccFsrRegValue::AccFsrRegValue_2G },
    { nnd::lsm6ds3::AccelerometerFsr_4G  , nnd::lsm6ds3::detail::AccFsrRegValue::AccFsrRegValue_4G },
    { nnd::lsm6ds3::AccelerometerFsr_8G  , nnd::lsm6ds3::detail::AccFsrRegValue::AccFsrRegValue_8G },
    { nnd::lsm6ds3::AccelerometerFsr_16G , nnd::lsm6ds3::detail::AccFsrRegValue::AccFsrRegValue_16G},
};

const GyroscopeFsrSetting GyroscopeFsrSettings[] =
{
    { nnd::lsm6ds3::GyroscopeFsr_125dps , nnd::lsm6ds3::detail::GyroFsrRegValue::GyroFsrRegValue_125dps   },
    { nnd::lsm6ds3::GyroscopeFsr_245dps , nnd::lsm6ds3::detail::GyroFsrRegValue::GyroFsrRegValue_245dps   },
    { nnd::lsm6ds3::GyroscopeFsr_500dps , nnd::lsm6ds3::detail::GyroFsrRegValue::GyroFsrRegValue_500dps   },
    { nnd::lsm6ds3::GyroscopeFsr_1000dps, nnd::lsm6ds3::detail::GyroFsrRegValue::GyroFsrRegValue_1000dps  },
    { nnd::lsm6ds3::GyroscopeFsr_2000dps, nnd::lsm6ds3::detail::GyroFsrRegValue::GyroFsrRegValue_2000dps  },
};

const AccelerometerOdrSetting AccelerometerOdrSettings[] =
{
    { nnd::lsm6ds3::AccelerometerOdr_0Hz   , nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_0Hz    },
    { nnd::lsm6ds3::AccelerometerOdr_13Hz  , nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_13Hz   },
    { nnd::lsm6ds3::AccelerometerOdr_26Hz  , nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_26Hz   },
    { nnd::lsm6ds3::AccelerometerOdr_52Hz  , nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_52Hz   },
    { nnd::lsm6ds3::AccelerometerOdr_104Hz , nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_104Hz  },
    { nnd::lsm6ds3::AccelerometerOdr_208Hz , nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_208Hz  },
    { nnd::lsm6ds3::AccelerometerOdr_416Hz , nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_416Hz  },
    { nnd::lsm6ds3::AccelerometerOdr_833Hz , nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_833Hz  },
    { nnd::lsm6ds3::AccelerometerOdr_1660Hz, nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_1660Hz },
    { nnd::lsm6ds3::AccelerometerOdr_3330Hz, nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_3330Hz },
    { nnd::lsm6ds3::AccelerometerOdr_6660Hz, nnd::lsm6ds3::detail::AccOdrRegValue::AccOdrRegValue_6660Hz },
};

const GyroscopeOdrSetting GyroscopeOdrSettings[] =
{
    { nnd::lsm6ds3::GyroscopeOdr_0Hz    , nnd::lsm6ds3::detail::GyroOdrRegValue::GyroOdrRegValue_0Hz    },
    { nnd::lsm6ds3::GyroscopeOdr_13Hz   , nnd::lsm6ds3::detail::GyroOdrRegValue::GyroOdrRegValue_13Hz   },
    { nnd::lsm6ds3::GyroscopeOdr_26Hz   , nnd::lsm6ds3::detail::GyroOdrRegValue::GyroOdrRegValue_26Hz   },
    { nnd::lsm6ds3::GyroscopeOdr_52Hz   , nnd::lsm6ds3::detail::GyroOdrRegValue::GyroOdrRegValue_52Hz   },
    { nnd::lsm6ds3::GyroscopeOdr_104Hz  , nnd::lsm6ds3::detail::GyroOdrRegValue::GyroOdrRegValue_104Hz  },
    { nnd::lsm6ds3::GyroscopeOdr_208Hz  , nnd::lsm6ds3::detail::GyroOdrRegValue::GyroOdrRegValue_208Hz  },
    { nnd::lsm6ds3::GyroscopeOdr_416Hz  , nnd::lsm6ds3::detail::GyroOdrRegValue::GyroOdrRegValue_416Hz  },
    { nnd::lsm6ds3::GyroscopeOdr_833Hz  , nnd::lsm6ds3::detail::GyroOdrRegValue::GyroOdrRegValue_833Hz  },
    { nnd::lsm6ds3::GyroscopeOdr_1660Hz , nnd::lsm6ds3::detail::GyroOdrRegValue::GyroOdrRegValue_1660Hz },
};

}

namespace nnd {
namespace lsm6ds3 {
namespace detail {

AccelerometerFsr ConvertToAccelerometerFsr(AccFsrRegValue reg)
{
    for(const auto& setting : AccelerometerFsrSettings)
    {
        if(setting.reg == reg)
        {
            return setting.fsr;
        }
    }
    NN_ABORT("Invalid reg=%x\n", reg);
}

AccFsrRegValue ConvertToAccFsrRegValue(AccelerometerFsr fsr)
{
    for(const auto& setting : AccelerometerFsrSettings)
    {
        if(setting.fsr == fsr)
        {
            return setting.reg;
        }
    }
    NN_ABORT("Invalid fsr=%x\n", fsr);
}

GyroscopeFsr ConvertToGyroscopeFsr(GyroFsrRegValue reg)
{
    for(const auto& setting : GyroscopeFsrSettings)
    {
        if(setting.reg == reg)
        {
            return setting.fsr;
        }
    }
    NN_ABORT("Invalid reg=%x\n", reg);
}

GyroFsrRegValue ConvertToGyroFsrRegValue(GyroscopeFsr fsr)
{
    for(const auto& setting : GyroscopeFsrSettings)
    {
        if(setting.fsr == fsr)
        {
            return setting.reg;
        }
    }
    NN_ABORT("Invalid fsr=%x\n", fsr);
}

AccelerometerOdr ConvertToAccelerometerOdr(AccOdrRegValue reg)
{
    for(const auto& setting : AccelerometerOdrSettings)
    {
        if(setting.reg == reg)
        {
            return setting.odr;
        }
    }
    NN_ABORT("Invalid reg=%x\n", reg);
}

AccOdrRegValue ConvertToAccOdrRegValue(AccelerometerOdr odr)
{
    for(const auto& setting : AccelerometerOdrSettings)
    {
        if(setting.odr == odr)
        {
            return setting.reg;
        }
    }
    NN_ABORT("Invalid odr=%x\n", odr);
}

GyroscopeOdr ConvertToGyroscopeOdr(GyroOdrRegValue reg)
{
    for(const auto& setting : GyroscopeOdrSettings)
    {
        if(setting.reg == reg)
        {
            return setting.odr;
        }
    }
    NN_ABORT("Invalid reg=%x\n", reg);
}

GyroOdrRegValue ConvertToGyroOdrRegValue(GyroscopeOdr odr)
{
    for(const auto& setting : GyroscopeOdrSettings)
    {
        if(setting.odr == odr)
        {
            return setting.reg;
        }
    }
    NN_ABORT("Invalid odr=%x\n", odr);
}

} // detail
} // lsm6ds3
} // nnd
